package org.lsst.ccs.subsystem.monitor.ui;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.Point;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.Popup;
import javax.swing.PopupFactory;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.border.LineBorder;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import org.freehep.application.studio.Studio;
import org.lsst.ccs.bus.MetadataStatus;
import org.lsst.ccs.bus.TrendingStatus;
import org.lsst.ccs.bus.ValueNotification;
import org.lsst.ccs.plugin.jas3.trending.TrendingService;
import org.lsst.ccs.subsystem.monitor.data.MonitorChan;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.monitor.data.MonitorState;

/*
 * MonitorTrendingTable.java
 *
 * @author turri
 */
public class MonitorTrendingTable extends javax.swing.JPanel {

    final static String[] colNames = {"Description", "Value", "Units",
                                      "Low Limit", "Al.", "High Limit", "Al.",
                                      "Name"};
    final static Class[] colTypes = {String.class, TrendingValue.class,
                                     String.class, Double.class,
                                     AlarmMarker.class, Double.class,
                                     AlarmMarker.class, String.class};
    final static Color colGood = new Color(160, 255, 160),
                       colError = new Color(255, 160, 160),
                       colOffln = new Color(160, 200, 255),
                       colPopup = new Color(255, 255, 160);
    final static Font myFont = new Font("Helvetica", Font.PLAIN, 12);
    final static Font changeFont = new Font("Helvetica", Font.BOLD, 12);

    final static int DESCRIPTION_IND = 0;
    final static int VALUE_IND = 1;
    final static int UNITS_IND = 2;
    final static int LOW_LIMIT_IND = 3;
    final static int LOW_ALARM_IND = 4;
    final static int HIGH_LIMIT_IND = 5;
    final static int HIGH_ALARM_IND = 6;
    final static int NAME_IND = 7;

    Map<String, SubsysDesc> subsysMap = new HashMap<>();
    CommandSender sender;
    int lastFocusRow;
    Popup popup;


    /** Creates new form MonitorTrendingTable */
    public MonitorTrendingTable(CommandSender sender) {
        initComponents();
        JTableHeader hdr = jTable1.getTableHeader();
        hdr.setReorderingAllowed(false);
        hdr.setSize(hdr.getWidth(), hdr.getHeight() + 2);
        jTable1.setDefaultRenderer(TrendingValue.class,
                                   new TrendingTableCellRenderer());
        jTable1.setDefaultRenderer(Double.class, new LimitsCellRenderer());
        jTable1.setDefaultRenderer(String.class, new TextCellRenderer());
        jTable1.setDefaultRenderer(AlarmMarker.class, new AlarmCellRenderer());
        jTable1.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
        jTable1.setRowSelectionAllowed(false);
        jTable1.setColumnSelectionAllowed(false);
        jTable1.setFont(myFont);
        jTable1.setRowHeight(jTable1.getRowHeight() + 2);

        this.sender = sender;
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jScrollPane1 = new javax.swing.JScrollPane();
        jTable1 = new javax.swing.JTable();

        setLayout(new javax.swing.BoxLayout(this, javax.swing.BoxLayout.LINE_AXIS));

        jTable1.setModel(new TrendingTableModel());
        jTable1.setName("monTable");
        jTable1.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                jTable1MouseClicked(evt);
            }
        });
        jScrollPane1.setViewportView(jTable1);

        add(jScrollPane1);
    }// </editor-fold>//GEN-END:initComponents

    private void jTable1MouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_jTable1MouseClicked
        if (popup != null) {
            popup.hide();
            popup = null;
        }
        JTable source = (JTable)evt.getSource();
        Point point = evt.getPoint();
        int row = source.rowAtPoint(point);
        SubsysDesc desc = findSubsys(row);
        if (desc == null) return;
        if (evt.getClickCount() == 1) {
            int column = source.columnAtPoint(point);
            if (column != LOW_ALARM_IND && column != HIGH_ALARM_IND) return;
            String name = (String)source.getModel().getValueAt(row, column);
            if (name == null || name.length() == 0) return;
            PopupFactory fact = PopupFactory.getSharedInstance();
            popup = fact.getPopup(null, desc.alarmMap.get(name),
                                  evt.getXOnScreen(), evt.getYOnScreen());
            popup.show();
        }
        else if (evt.getClickCount() == 2) {
            String name = (String)source.getModel().getValueAt(row, NAME_IND);
            if (name.length() == 0) return;
            String[] path = {desc.name, name};
            Studio studio = (Studio)Studio.getApplication();
            if (studio == null) return;
            TrendingService trending = (TrendingService)studio.getLookup()
                                         .lookup(TrendingService.class);
            if (trending == null) return;
            trending.show(path);
        }
    }//GEN-LAST:event_jTable1MouseClicked

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JTable jTable1;
    // End of variables declaration//GEN-END:variables

    public void setSubsystems(String... sNames)
    {
        for (String sName : sNames) {
            SubsysDesc desc = new SubsysDesc();
            desc.name = sName;
            subsysMap.put(sName, desc);
        }
    }

    public void updateTableModel(String sName, MonitorFullState status)
    {
        SubsysDesc desc = subsysMap.get(sName);
        if (desc == null) return;
        SwingUtilities.invokeLater(new UpdateTrendingTableModel(desc, status));
    }

    public void updateTableValues(TrendingStatus status)
    {
        SubsysDesc desc = subsysMap.get(status.getOrigin());
        if (desc == null) return;
        SwingUtilities.invokeLater(new UpdateTrendingTable(desc, status));
    }

    public void updateTableValue(String sName, String iName, double value)
    {
        SubsysDesc desc = subsysMap.get(sName);
        if (desc == null) return;
        SwingUtilities.invokeLater(new UpdateTableValue(desc, iName, value));
    }

    public void updateTableLimit(MetadataStatus status)
    {
        SubsysDesc desc = subsysMap.get(status.getOrigin());
        if (desc == null) return;
        SwingUtilities.invokeLater(new UpdateLimitValue(desc, status));
    }

    public void updateTableState(String sName, MonitorState status)
    {
        SubsysDesc desc = subsysMap.get(sName);
        if (desc == null) return;
        SwingUtilities.invokeLater(new UpdateState(desc, status));
    }

    SubsysDesc findSubsys(int row) {
        for (SubsysDesc desc : subsysMap.values()) {
            if (row >= desc.start && row < desc.end) return desc;
        }
        return null;
    }

    int getChannelState(int row) {
        SubsysDesc desc = findSubsys(row);
        int sRow = desc.tsRowMap.get(row - desc.start);
        return sRow < 0 ? -2 : !desc.onlineChans.get(sRow) ? -1
                 : !desc.goodChans.get(sRow) ? 0 : 1;
    }

    boolean hasLowLimitChanged(int row) {
        SubsysDesc desc = findSubsys(row);
        int sRow = desc.tsRowMap.get(row - desc.start);
        return sRow >= 0 ? desc.lowLimitChange.get(sRow) : false;
    }
        
    boolean hasHighLimitChanged(int row) {
        SubsysDesc desc = findSubsys(row);
        int sRow = desc.tsRowMap.get(row - desc.start);
        return sRow >= 0 ? desc.highLimitChange.get(sRow) : false;
    }

    boolean isNewSection(int row) {
        SubsysDesc desc = findSubsys(row);
        return desc.tsRowMap.get(row - desc.start) < 0;
    }

    private static String fmt(double val) {
        return String.format("%.2f  ", val);
    }

    class TrendingTableModel extends DefaultTableModel
                             implements TableModelListener {

        public TrendingTableModel() {
            super(colNames, 0);
            addTableModelListener(this);
        }

        @Override
        public Class getColumnClass(int column) {
            return colTypes[column];
        }

        @Override
        public boolean isCellEditable(int row, int column) {
            if (column == LOW_LIMIT_IND || column == HIGH_LIMIT_IND) {
                return !(getValueAt(row, column) == null);
            }
            return false;
        }

        @Override
        public void tableChanged(TableModelEvent e)
        {
            int column = e.getColumn();
            if (column != LOW_LIMIT_IND && column != HIGH_LIMIT_IND) return;
            boolean isLow = column == LOW_LIMIT_IND;
            int row = e.getFirstRow();
            SubsysDesc desc = findSubsys(row);
            int sRow = desc.tsRowMap.get(row - desc.start);
            double newValue = (Double)getValueAt(row, column);
            double oldValue = isLow ? desc.lowLimits.get(sRow)
                : desc.highLimits.get(sRow);

            if (newValue != oldValue) {
                setValueAt(oldValue, row, column);    // Put it back for now
                sender.sendCommand(desc.name,
                                   (String)getValueAt(row, NAME_IND), "change",
                                   isLow ? "limitLo" : "limitHi", newValue);
            }
        }

        void addTrendingRow(SubsysDesc desc, MonitorChan chan) {
            String[] descrip = chan.getDescription().split("\\\\", 2);
            if (descrip.length == 2) {
                if (descrip[0].length() == 0) {
                    desc.indent = false;
                }
                else {
                    insertRow(desc.end++,
                              new Object[]{descrip[0], null, "", null, "",
                                           null, "", ""});
                    desc.tsRowMap.add(-1);
                    desc.indent = true;
                }
                descrip[0] = descrip[1];
            }
            desc.chanMap.put(chan.getName(), desc.tsRowMap.size());
            desc.stRowMap.add(desc.tsRowMap.size());
            desc.tsRowMap.add(desc.lowLimits.size());
            desc.lowLimits.add(chan.getLowLimit());
            desc.highLimits.add(chan.getHighLimit());
            insertRow(desc.end++,
                      new Object[]{(desc.indent ? "   " : "") + descrip[0],
                                   chan.getValue(), chan.getUnits(),
                                   chan.getLowLimit(), chan.getLowAlarm(),
                                   chan.getHighLimit(), chan.getHighAlarm(),
                                   chan.getName()});
        }

        void updateValues(SubsysDesc desc, TrendingStatus s) {
            for (ValueNotification vn : (List<ValueNotification>)s.getValue()) {
                Integer tRow = desc.chanMap.get(vn.getName());
                if (tRow == null) return;
                setValueAt(vn.getData(), desc.start + tRow, VALUE_IND);
            }
        }

        void updateValue(SubsysDesc desc, String name, double value) {
            Integer tRow = desc.chanMap.get(name);
            if (tRow == null) return;
            setValueAt(value, desc.start + tRow, VALUE_IND);
        }

        void updateLimitValue(SubsysDesc desc, MetadataStatus s) {
            boolean high = s.getMetadataName().equals("alarmHigh");
            double limit = Double.valueOf(s.getMetadataValue());
            Integer column, tRow = desc.chanMap.get(s.getDataName());
            if (tRow == null) return;
            if (high) {
                desc.highLimits.set(desc.tsRowMap.get(tRow), limit);
                column = HIGH_LIMIT_IND;
            }
            else {
                desc.lowLimits.set(desc.tsRowMap.get(tRow), limit);
                column = LOW_LIMIT_IND;
            }
            setValueAt(limit, desc.start + tRow, column);
        }

        void updateState(SubsysDesc desc, MonitorState s)
        {
            desc.goodChans = s.getGoodChans();
            desc.onlineChans = s.getOnlineChans();
            BitSet updateHigh = (BitSet)s.getHighLimitChange().clone();
            updateHigh.xor(desc.highLimitChange);
            desc.highLimitChange.xor(updateHigh);
            for (int sRow = updateHigh.nextSetBit(0); sRow >= 0;
                 sRow = updateHigh.nextSetBit(sRow + 1)) {
                setValueAt(desc.highLimits.get(sRow),
                           desc.start + desc.stRowMap.get(sRow),
                           HIGH_LIMIT_IND);
            }
            BitSet updateLow = (BitSet)s.getLowLimitChange().clone();
            updateLow.xor(desc.lowLimitChange);
            desc.lowLimitChange.xor(updateLow);
            for (int sRow = updateLow.nextSetBit(0); sRow >= 0;
                 sRow = updateLow.nextSetBit(sRow + 1)) {
                setValueAt(desc.lowLimits.get(sRow),
                           desc.start + desc.stRowMap.get(sRow),
                           LOW_LIMIT_IND);
            }
        }

    }

    class SubsysDesc {

        String name;
        int start, size, end;
        boolean indent;
        Map<String, JLabel> alarmMap = new HashMap<>();
        Map<String, Integer> chanMap = new HashMap<>();
        List<Integer> tsRowMap = new ArrayList<>(), stRowMap = new ArrayList<>();
        BitSet goodChans = new BitSet(), onlineChans = new BitSet(),
               lowLimitChange = new BitSet(), highLimitChange = new BitSet();
        List<Double> lowLimits = new ArrayList<>(),
                     highLimits = new ArrayList<>();

    }

    class TrendingValue {
    }

    class AlarmMarker {
    }

    class UpdateTrendingTable implements Runnable {

        SubsysDesc desc;
        TrendingStatus s;

        UpdateTrendingTable(SubsysDesc desc, TrendingStatus s) {
            this.desc = desc;
            this.s = s;
        }

        @Override
        public void run() {
            ((TrendingTableModel)jTable1.getModel()).updateValues(desc, s);
        }
    }

    class UpdateTableValue implements Runnable {

        SubsysDesc desc;
        String n;
        double v;

        UpdateTableValue(SubsysDesc desc, String n, double v) {
            this.desc = desc;
            this.n = n;
            this.v = v;
        }

        @Override
        public void run() {
            ((TrendingTableModel)jTable1.getModel()).updateValue(desc, n, v);
        }
    }

    class UpdateLimitValue implements Runnable {

        SubsysDesc desc;
        MetadataStatus s;

        UpdateLimitValue(SubsysDesc desc, MetadataStatus s) {
            this.desc = desc;
            this.s = s;
        }

        @Override
        public void run() {
            ((TrendingTableModel)jTable1.getModel()).updateLimitValue(desc, s);
        }

    }

    class UpdateState implements Runnable {

        SubsysDesc desc;
        MonitorState s;

        UpdateState(SubsysDesc desc, MonitorState s) {
            this.desc = desc;
            this.s = s;
        }

        @Override
        public void run() {
            ((TrendingTableModel)jTable1.getModel()).updateState(desc, s);
        }

    }

    class UpdateTrendingTableModel implements Runnable {

        SubsysDesc desc;
        MonitorFullState s;

        UpdateTrendingTableModel(SubsysDesc desc, MonitorFullState s) {
            this.desc = desc;
            this.s = s;
        }

        @Override
        public void run() {

            desc.indent = false;
            desc.alarmMap.clear();
            desc.chanMap.clear();
            desc.tsRowMap.clear();
            desc.stRowMap.clear();
            desc.goodChans.clear();
            desc.onlineChans.clear();
            desc.lowLimitChange.clear();
            desc.highLimitChange.clear();
            desc.lowLimits.clear();
            desc.highLimits.clear();

            Map<String, String> alarms = s.getAlarms();
            for (String name : alarms.keySet()) {
                JLabel label = new JLabel(alarms.get(name));
                label.setBorder(LineBorder.createBlackLineBorder());
                label.setBackground(colPopup);
                label.setFont(myFont);
                desc.alarmMap.put(name, label);
            }

            TrendingTableModel model = (TrendingTableModel)jTable1.getModel();
            for (int j = 0; j < desc.size; j++) {
                model.removeRow(desc.start);
            }

            desc.end = desc.start;
            for (MonitorChan chan : s.getChannels()) {
                model.addTrendingRow(desc, chan);
            }
            desc.size = desc.end - desc.start;

            int row = 0;
            for (SubsysDesc d : subsysMap.values()) {
                d.start = row;
                d.end = (row += d.size);
            }

            for (int c = 0; c < jTable1.getColumnCount(); c++) {
                TableColumnModel colModel = jTable1.getColumnModel();
                TableColumn col = colModel.getColumn(c);
                TableCellRenderer rndr;
                Component comp;

                rndr = jTable1.getTableHeader().getDefaultRenderer();
                comp = rndr.getTableCellRendererComponent(jTable1,
                                                          col.getHeaderValue(),
                                                          false, false, 0, 0);
                int width = comp.getPreferredSize().width;

                rndr = jTable1.getCellRenderer(0, c);
                Class colClass = jTable1.getColumnClass(c);
                if (colClass.equals(String.class)
                    || colClass.equals(AlarmMarker.class)) {
                    for (int r = desc.start; r < desc.end; r++) {
                        Object value = jTable1.getValueAt(r, c);
                        comp = rndr.getTableCellRendererComponent(jTable1, value,
                                                                  false, false,
                                                                  r, c);
                        width = Math.max(width, comp.getPreferredSize().width);
                    }
                }
                else {
                    comp = rndr.getTableCellRendererComponent(jTable1, -999.99,
                                                              false, false, 0, c);
                    width = Math.max(width, comp.getPreferredSize().width);
                }
                col.setPreferredWidth(width + 4);
            }

            model.updateState(desc, s.getMonitorState());
        }

    }

    class LimitsCellRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value,
                                                           boolean isSelected,
                                                           boolean hasFocus,
                                                           int row, int column)
        {
            Component c;
            if (value == null) {
                c = super.getTableCellRendererComponent(table, "", false, false,
                                                        row, column);
                return c;
            }

            c = super.getTableCellRendererComponent(table, fmt((Double)value),
                                                    false, hasFocus, row, column);
            if (column == LOW_LIMIT_IND && hasLowLimitChanged(row)
                || column == HIGH_LIMIT_IND && hasHighLimitChanged(row)) {
                c.setFont(changeFont);
                c.setForeground(Color.blue);
            }
            else {
                c.setForeground(Color.black);
            }
            ((JLabel)c).setHorizontalAlignment(SwingConstants.RIGHT);
            if (hasFocus) {
                lastFocusRow = row;
            }

            return c;
        }

    }

    class TrendingTableCellRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value,
                                                           boolean isSelected,
                                                           boolean hasFocus,
                                                           int row, int column)
        {
            Component c;
            if (value == null) {
                c = super.getTableCellRendererComponent(table, "", false, false,
                                                        row, column);
                c.setBackground(Color.WHITE);
            }
            else {
                c = super.getTableCellRendererComponent(table, fmt((Double)value),
                                                        false, false, row, column);
                int state = getChannelState(row);
                c.setBackground(state > 0 ? colGood : state == 0 ? colError
                                : state == -1 ? colOffln : Color.WHITE);
                ((JLabel)c).setHorizontalAlignment(SwingConstants.RIGHT);
            }

            return c;
        }
    }

    class AlarmCellRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value,
                                                           boolean isSelected,
                                                           boolean hasFocus,
                                                           int row, int column)
        {
            String text = (String)value;
            text = text == null || text.length() == 0 ? "" : "  \u2713";
            return super.getTableCellRendererComponent(table, text, false,
                                                       false, row, column);
        }

    }

    class TextCellRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row, int column)
        {
            Component c = super.getTableCellRendererComponent(table,
                                                              "  " + (String)value,
                                                              false, false,
                                                              row, column);
            if (column == DESCRIPTION_IND && isNewSection(row)) {
                c.setFont(changeFont);
            }

            return c;
        }

    }

}
