package org.lsst.ccs.subsystem.common.actions;

import java.io.Serializable;
import org.lsst.ccs.bus.data.Alert;

/**
 * An descriptions of the Actions that can be taken by the Quadbox.
 * This enumeration, when sent as an Alarm payload, will trigger a response
 * from the Quadbox to turn off the bulk power supply switch (by name).
 * 
 * @author The LSST CCS Team.
 * 
 */
public class BulkPsPowerAction implements Serializable {

    /**
     * The type of Actions that can be taken.
     */
    public enum Type {
        TURN_BULK_PS_OFF;
    }
    
    /**
     * Add a given RebPowrAction payload to a given Alert.
     * @param alert      The Alert to which the payload is to be added
     * @param actionType The action to be taken.
     * @param bulkPsSwitchName     The name of the bulk power supply switch.
     * 
     */
    public static void addBulkPsPowerActionPayloadToAlert(Alert alert, Type actionType, String bulkPsSwitchName) {
        BulkPsPowerAction action = new BulkPsPowerAction(actionType, bulkPsSwitchName);
        alert.addAlertData(BulkPsPowerAction.class.getSimpleName(), action);
    }
    
    /**
     * Get a RebPowerAction payload from the provided Alert.
     * @param alert The Alert that might contain the payload
     * @return      The contained RebPowerAction payload. null if there is no payload.
     */
    public static BulkPsPowerAction getBulkPsPowerActionPayloadFromAlert(Alert alert) {
        if ( alert.getAlertDataNames().contains(BulkPsPowerAction.class.getSimpleName()) ) {
            return (BulkPsPowerAction)alert.getAlertData(BulkPsPowerAction.class.getSimpleName());
        }
        return null;
    }
    
    private final Type actionType;
    private final String bulkPsSwitchName;
    
    BulkPsPowerAction(Type actionType, String psName) {
        this.actionType = actionType;
        this.bulkPsSwitchName = psName;
    }
    
    public Type getType() {
        return actionType;
    }
    
    public String getBulkPsSwitchName() {
        return bulkPsSwitchName;
    }
            
    
}
