package org.lsst.ccs.subsystem.comcamvacuum;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.logging.Level;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.Measurement;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.subsystem.common.actions.ComCamSharedVacState;
import org.lsst.ccs.subsystem.common.MonitorTaskControl;
import org.lsst.ccs.subsystem.comcamvacuum.constants.MonitorControl;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.twistorr.TwisTorr84;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.comcamvacuum.alerts.ComCamVacuumAlerts;
import org.lsst.ccs.subsystem.comcamvacuum.constants.PLCState;
import org.lsst.ccs.subsystem.comcamvacuum.constants.ConditionState;
import org.lsst.ccs.subsystem.comcamvacuum.constants.LatchState;
import org.lsst.ccs.subsystem.comcamvacuum.constants.Devices;
import org.lsst.ccs.subsystem.comcamvacuum.constants.DeviceState;
//import org.lsst.ccs.subsystem.vacuum.constants.ImageChannel;
import org.lsst.ccs.subsystem.comcamvacuum.constants.SwitchEnable;
import org.lsst.ccs.subsystem.comcamvacuum.constants.SwitchState;
import org.lsst.ccs.subsystem.comcamvacuum.data.ComCamVacuumAgentProperties;
import org.lsst.ccs.subsystem.comcamvacuum.constants.VacuumState;
import org.lsst.ccs.subsystem.comcamvacuum.data.VacuumException;
import org.lsst.ccs.subsystem.comcamvacuum.data.VacSysState;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.monitor.Alarm;

import org.lsst.ccs.utilities.logging.Logger;

//import org.lsst.ccs.common.devices.power;
import org.lsst.ccs.subsystem.common.devices.turbopump.TwisTorr84Device;
import org.lsst.ccs.subsystem.common.devices.refrigeration.AVCCryoDevice;
import org.lsst.ccs.subsystem.common.devices.vacuum.GPVacMon835Device;

import org.lsst.ccs.subsystem.common.devices.refrigeration.data.RefrigerationConfig;

/**
 * The ComCamVacuumSubsystem is designed specifically for running ComCam
 * thermal and vacuum control,
 */
public class ComCamVacuumSubsystem extends Subsystem implements HasLifecycle, AlertListener {

    // Instances of the foreline pressure appearing below when there
    // is no foreline pressure gauge intended for ComCamare for placeholders
    // for a future similar check

    private static final boolean TEST_ONLY = false;
    
    private static final double
	PRESS_ATMOS = 759.0,
	COMCAM_PRESS_HIGH = 20.0,
	TURBO_MAX = 81000,
	speedCryoTurboLow = 0.1 * TURBO_MAX,
	speedCryoTurboHigh  = 0.5 * TURBO_MAX;

    // Configuration parameters
    @ConfigurationParameter(category="Vacuum", isFinal=true)
    private volatile double pressTurboLow; // 5.0
    @ConfigurationParameter(category="Vacuum", isFinal=true)
    private volatile double pressForelineLow; // 5.0
    @ConfigurationParameter(category="Vacuum", isFinal=true)
    private volatile double pressDiffLow; // 0.09
    @ConfigurationParameter(category="Vacuum", isFinal=true)
    private volatile double pressDiffHigh; // 20.0
    @ConfigurationParameter(category="Vacuum", isFinal=true)
    private volatile double pressIonOff; // 1.1e-5
    @ConfigurationParameter(category="Vacuum", isFinal=true)
    private volatile double pressIonEnable; // 1.2e-6
    @ConfigurationParameter(category="Vacuum", isFinal=true)
    private volatile double pressVacuum; // 1.0e-7                


    private static final int
	    MAX_PRESS_ERRORS = 2,
            MAX_SPEED_ERRORS = 0,
            DELAY_ION_OFF = 10000;

    private static final int[] switchChannels = new int[VacSysState.NUM_SWITCHES];

    static {
        switchChannels[VacSysState.SW_CRYO_ION_PUMP1] = IonPumpDevice.CHAN_CIP1;
	// ************* need to add ... HN 20200128 !!!   --- NOPE ... not connected 
	//        switchChannels[VacSysState.SW_CRYO_VALVE] = VacPlutoDevice.SW_OPEN_VCR00;
        switchChannels[VacSysState.SW_CRYO_VALVE] = 0;
        switchChannels[VacSysState.SW_CRYOTEL_CRYO] = RefrigerationConfig.CryoTelType.CRYO.ordinal();
        switchChannels[VacSysState.SW_CRYOTEL_COLD1] = RefrigerationConfig.CryoTelType.COLD1.ordinal();
        switchChannels[VacSysState.SW_CRYOTEL_COLD2] = RefrigerationConfig.CryoTelType.COLD2.ordinal();
	//        switchChannels[VacSysState.SW_CRYOTEL_PWR] = 0;

    }
    private static final Map<TwisTorr84.PumpStatus, DeviceState> turboStateMap = new HashMap<>();

    static {
        turboStateMap.put(TwisTorr84.PumpStatus.STOP, DeviceState.STOPPED);
        turboStateMap.put(TwisTorr84.PumpStatus.WAIT_INTLK, DeviceState.WAITING);
        turboStateMap.put(TwisTorr84.PumpStatus.STARTING, DeviceState.STARTNG);
        turboStateMap.put(TwisTorr84.PumpStatus.NORMAL, DeviceState.NORMAL);
        turboStateMap.put(TwisTorr84.PumpStatus.BRAKING, DeviceState.BRAKING);
        turboStateMap.put(TwisTorr84.PumpStatus.FAIL, DeviceState.FAILED);
        turboStateMap.put(TwisTorr84.PumpStatus.AUTO_TUNING, DeviceState.AUTOTUN);
    }

    
    private static final List<Integer> cryotels = new ArrayList<>();
    static {
        cryotels.add(VacSysState.SW_CRYOTEL_CRYO);
        cryotels.add(VacSysState.SW_CRYOTEL_COLD1);
        cryotels.add(VacSysState.SW_CRYOTEL_COLD2);
    }
 
    private static final Map<Integer, ComCamVacuumAlerts> latchAlertMap = new HashMap<>();
    static {
        latchAlertMap.put(VacSysState.LATCH_CR_VACUUM, ComCamVacuumAlerts.CRYO_VACUUM_BAD);
	//        latchAlertMap.put(VacSysState.LATCH_CR_GATE_NFC, ComCamVacuumAlerts.CRYO_GATE_FORCED_SHUT);
	//        latchAlertMap.put(VacSysState.LATCH_CR_GATE_AO, ComCamVacuumAlerts.CRYO_GATE_CANNOT_OPEN);
        latchAlertMap.put(VacSysState.LATCH_CR_PUMP, ComCamVacuumAlerts.CRYO_TURBO_PUMP_BAD);
    }
    private static final Map<String, Integer> revAlertMap = new HashMap<>();
    static {
        for (int cond : latchAlertMap.keySet()) {
            revAlertMap.put(latchAlertMap.get(cond).getId(), cond);
        }
    }


    @LookupName
    private String name;

//    private static final Logger LOG = Logger.getLogger("org.lsst.ccs.subsystem.comcamvacuum");
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService agentPropertiesService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService apts;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService ass;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private VacPlutoDevice plutoDevc = null;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private CryoTurboDevice cryoTurboDevc;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private IonPumpDevice ionPumpDevc;
//    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
//    private AVCCryoDevice cryotelDevc;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    Map<String, AVCCryoDevice> avcMap = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, Channel> channelMap = new LinkedHashMap<>();

    // From Groovy file
    private String cryoPressChan, turboPressChan, turboSpeedChan, forelinePressChan, CryoTempChan,
            Cold1TempChan, Cold2TempChan, IonPumpVoltageChan;
    private List<Integer> switches;

    
    // General

    private static final Logger LOG = Logger.getLogger(ComCamVacuumSubsystem.class.getName());

    private final VacSysState vacState = new VacSysState();
    private final ComCamSharedVacState comCamSharedVacState = new ComCamSharedVacState();
    private Set<Integer> switchSet;
    private final Device[] switchDevices = new Device[VacSysState.NUM_SWITCHES];
    private Channel cryoPressure, turboPressure, turboSpeed, forelinePressure, CryoTemp, Cold1Temp, Cold2Temp;
    private boolean running = true;
    private final Map<String, Integer> switchNameMap = new LinkedHashMap<>();
    private final Map<String, Boolean> activeAlertMap = new HashMap<>();
    private final Map<String, Boolean> activeAlarmMap = new HashMap<>();

    private long cryoIonOverStartTime = 0;
    private long vacBadTime = 0;


    private boolean plcActive = true;

    private volatile boolean alertResponseEnabled = true;

    private boolean monAlertEnabledVATShut = false;

    private MonitorTaskControl monitorControl;


//    private final Map<String, ChanParams> monChans = new HashMap<>();
//    private final ClientFactory clientFactory;
//    private ComCamVacuumCommands vacuumCommands;
    private String last_alertStr;

//    private final GlobalProc globalProc = new GlobalProc();
    private boolean useInfoAlerts = System.getProperty("org.lsst.ccs.subsystem.comcamvacuumPowerInfoAlerts", "false").contains("true");

    static class SensorData {
        Channel channel;
        int maxErrors;
        int numErrors = 0;
        double value = 0.0;
        double goodValue = Double.NaN;

        private SensorData(Channel channel, int maxErrors) {
            this.channel = channel;
            this.maxErrors = maxErrors;
        }
    }

    //    private SensorData cryoMainPr;
    private SensorData cryoFlinePr, cryoMainPr, cryoTurboPr, cryoTurboSp;

    

    /**
     * Main constructor for the ComCamVacuumSubsystem, normally invoked from the
     * corresponding groovy file.
     *
     */

    public ComCamVacuumSubsystem() {
        super("comcam-vacuum", AgentInfo.AgentType.WORKER);
    }

    @Override
    public void init() {
        //Alert registration step
        alertService.registerAlert(ComCamVacuumAlerts.VACUUM_PLC_NOT_ALIVE.getAlert());
	//        alertService.registerAlert(ComCamVacuumAlerts.CRYO_GATE_FORCED_SHUT.getAlert());
	//        alertService.registerAlert(ComCamVacuumAlerts.CRYO_GATE_CANNOT_OPEN.getAlert());
        alertService.registerAlert(ComCamVacuumAlerts.CRYO_TURBO_PUMP_BAD.getAlert());
    }

    
    
    @Override
    public void postInit() {

	// Add alert listener
        alertService.addListener(this);

        //Set a property to define that this Agent is a vacuum subsystem.
        agentPropertiesService.setAgentProperty(ComCamVacuumAgentProperties.COMCAMVACUUM_TYPE_AGENT_PROPERTY, ComCamVacuumSubsystem.class.getCanonicalName());

        if (switches != null) {
            switchSet = new HashSet<>(switches);
        } else {
            MonitorLogUtils.reportConfigError(LOG, name, "Switch list", "not specified");
        }
	//hh
        if (plutoDevc != null) {
            for (int cond : plutoDevc.getLatchIds()) {
                vacState.addLatch(cond);
                vacState.setLatch(cond, LatchState.CLEAR);
		LOG.info("Cleared latch cond  "+cond);
            }
             for (int cond : plutoDevc.getConditionIds()) {
                vacState.addCondition(cond);
                vacState.setCondition(cond, ConditionState.NO);
            }
        } else {
            MonitorLogUtils.reportConfigError(LOG, name, "Pluto device", "not specified");
        }

        if (cryoTurboDevc == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "Cryo turbo pump device", "not specified");
        }
        if (ionPumpDevc == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "Ion pump device", "not specified");
        }
        if (forelinePressChan != null) {
            forelinePressure = channelMap.get(forelinePressChan);
        }
        cryoFlinePr = new SensorData(forelinePressure, MAX_PRESS_ERRORS);

	if (cryoPressChan != null) {
	    cryoPressure = channelMap.get(cryoPressChan);
	}
        if (cryoPressure == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "Cryo pressure channel", "not specified or not defined");
        }
        cryoMainPr = new SensorData(cryoPressure, MAX_PRESS_ERRORS);

        if (turboPressChan != null) {
            turboPressure = channelMap.get(turboPressChan);
        }
        if (turboPressure == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "Turbo pump pressure channel", "not specified or not defined");
        }
        cryoTurboPr = new SensorData(turboPressure, MAX_PRESS_ERRORS);

        if (turboSpeedChan != null) {
            turboSpeed = channelMap.get(turboSpeedChan);
        }
	cryoTurboSp = new SensorData(turboSpeed, MAX_SPEED_ERRORS);

        if (turboSpeed == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "Turbo pump speed channel", "not specified or not defined");
        }

        List<Integer> ipChannels = ionPumpDevc.getChannelNumbers();
	LOG.info("ipChannel Names = " + ionPumpDevc.getChannelNames());
	LOG.info("ipChannel Numbers = " + ionPumpDevc.getChannelNumbers());
        Iterator swIter = switchSet.iterator();
        while (swIter.hasNext()) {
            int sw = (Integer) swIter.next();
            switch (sw) {
                case VacSysState.SW_CRYO_TURBO:
                    switchDevices[sw] = cryoTurboDevc;
                    break;
                case VacSysState.SW_CRYO_VALVE:
                    switchDevices[sw] = null;
                    break;
		  
	        case VacSysState.SW_CRYO_ION_PUMP1:
                    switchDevices[sw] = ionPumpDevc;
                    if (!ipChannels.contains(switchChannels[sw])) {
                        swIter.remove();
                    }
                    break;
                case VacSysState.SW_CRYOTEL_CRYO:
                    LOG.info("attempting to setup switch for CryoTelCryo");
                    switchDevices[sw] = avcMap.get("cryo");
                    break;
                case VacSysState.SW_CRYOTEL_COLD1:
                    switchDevices[sw] = avcMap.get("cold1");
                    break;
                case VacSysState.SW_CRYOTEL_COLD2:
                    switchDevices[sw] = avcMap.get("cold2");
                    break;
            }
        }

        for (int sw : switchSet) {
            vacState.addSwitch(sw);
            vacState.setSwitchState(sw, SwitchState.OFFLINE);
            vacState.setSwitchEnable(sw, SwitchEnable.OFF);
        }

        for (String swName : SwitchNames.NAME_MAP.keySet()) {
            int sw = SwitchNames.NAME_MAP.get(swName);
            if (switchSet.contains(sw)) {
                switchNameMap.put(swName, sw);
            }
        }

        GPVacMon835Device vqm = new GPVacMon835Device();

        LOG.info("Configured the status aggregator to listen to subsystems: " + System.getProperty("org.lsst.ccs.subsystem.teststand", "ts") + " " + System.getProperty("org.lsst.ccs.subsystem.rebps", "ccs-rebps") + " " + System.getProperty("org.lsst.ccs.subsystem.comcamvacuum", "comcamvacuum"));
    }

    @Override
    public void build() {
        // Create the monitor task control object and node
        monitorControl = MonitorTaskControl.createNode(this, MonitorControl.NODE_NAME);


        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("vacuum-state",
                () -> updateVacuumState()).withPeriod(Duration.ofMillis(1000));
        apts.scheduleAgentPeriodicTask(pt);

        ass.registerState(VacuumState.class, "ComCam Vacuum state", this);
        ass.updateAgentState(VacuumState.UNKNOWN);

        for (ComCamVacuumAlerts alert : ComCamVacuumAlerts.values()) {
            activeAlertMap.put(alert.getId(), false);
        }
    }

    @Override
    public void postStart() {
        LOG.info("ComCamVacuum subsystem started");
    }

    /**
     ***************************************************************************
     **
     ** Sleep - what a waste
     * **************************************************************************
     */
    public void Sleep(double secs) {
        if (secs <= 0) {
            return;
        }
        try {
            Thread.sleep((int) (secs * 1000));
        } catch (InterruptedException ex) {
            LOG.error("Rude awakening!" + ex);
        }
    }

    /**
     * Gets the state of the Vacuum system.
     *
     * @return The vacuum state
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the vacuum system state")
    public VacSysState getVacuumState() {
	//        vacState.setTickMillis(getTickPeriod());
        vacState.setTickMillis(monitorControl.getPublishPeriod());

        return vacState;
    }

    /**
     * Gets the shared state of the Vacuum system.
     *
     * @return The vacuum state
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the vacuum system state")
    public ComCamSharedVacState getComCamSharedVacState() {
        comCamSharedVacState.setTickMillis(getTickPeriod());
        return comCamSharedVacState;
    }

    /**
     * Gets the list of switch names.
     *
     * @return The switch names.
     * @throws VacuumException
     */
    @Command(type = Command.CommandType.QUERY, description = "Get switch names")
    public List<String> getSwitchNames() throws VacuumException {
        return new ArrayList(switchNameMap.keySet());
    }

    /**
     * Turns a switch on or off.
     *
     * @param sw The switch number.
     * @param on Whether to turn on or off
     * @throws VacuumException
     */
    @Command(type = Command.CommandType.ACTION, description = "Turn on/off a switch")
    public void setSwitchOn(@Argument(description = "The switch number") int sw,
            @Argument(description = "Whether to turn on") boolean on) throws VacuumException {
        try {
            if (!switchSet.contains(sw)) {
                throw new VacuumException("Invalid switch number: " + sw);
            }
            setSwitch(sw, on);
        } finally {
            publishState();
        }

    }

    /**
     * Turns a named switch on or off.
     *
     * @param name The switch name.
     * @param on Whether to turn on or off
     * @throws VacuumException
     */
    @Command(type = Command.CommandType.ACTION, description = "Turn on/off a named switch")
    public void setNamedSwitchOn(@Argument(description = "The switch name") String name,
            @Argument(description = "Whether to turn on") boolean on) throws VacuumException {
        Integer sw = switchNameMap.get(name);
        try {
            if (sw == null) {
                throw new VacuumException("Invalid switch name: " + name);
            }
            setSwitch(sw, on);
        } finally {
            publishState();
        }
    }

    /**
     * Turns a switch on or off.
     *
     * @param sw The switch number.
     * @param on Whether to turn on or off
     * @throws VacuumException
     */
    private void setSwitch(int sw, boolean on) throws VacuumException {
        SwitchState state = vacState.getSwitchState(sw);
	/*
        HashMap<String, Integer> switchNumMap = new HashMap<String, Integer>();
        for(Map.Entry<Integer, String> entry: switchNameMap.entrySet()){
            switchNumMap.put(entry.getValue(), entry.getKey());
        }
	*/
	//        LOG.info("setSwitch called for sw = " + sw +  " ( " + this.getSwitchNames().get(sw)+ " )");
        LOG.info("setSwitch called for sw = " + sw );
        if (state == SwitchState.OFFLINE) {
	    //h            LOG.info("Switch sw = " + sw + " ( " + this.getSwitchNames().get(sw)+ " )  is offline");
	    if (!(sw == vacState.SW_CRYO_VALVE))
		return;
	    else
		LOG.info("proceeding exceptionally for the VAT valve switch which will directly set the state as a request"); // HN

        }
        
        SwitchEnable enable = vacState.getSwitchEnable(sw);
        // setting switch on is rejected here if enable not on
        if (on && enable != SwitchEnable.ON && enable != SwitchEnable.WAS_ON) {
            LOG.info("enable = " + enable + " SwitchEnable.ON = " + SwitchEnable.ON + " SwitchEnable.WAS_ON = " + SwitchEnable.WAS_ON);
            return;
        }
	//h        LOG.info("proceeding with setting switch sw = " + sw + " ( " + this.getSwitchNames().get(sw)+ " ) ");
        Device swDevice = switchDevices[sw];
        try {
            if (swDevice instanceof TwisTorr84Device) {
                if (on) {
                    ((TwisTorr84Device) swDevice).startTurboPump();
                } else {
                    ((TwisTorr84Device) swDevice).stopTurboPump();
                }

            } else if (swDevice == ionPumpDevc) {
                ionPumpDevc.setChannelOn(switchChannels[sw], on);
            } else if (sw == VacSysState.SW_CRYO_VALVE) {

		LOG.info("Setting VAT valve REQUEST state");

		if (on) {
		    vacState.setSwitchState(sw, SwitchState.ON);
		    setMonAlertEnabledVATShut(false); // allow VAT valve opening after closing by monitoring
		} else {
		    vacState.setSwitchState(sw, SwitchState.OFF);
		}

            } else if (sw == VacSysState.SW_CRYOTEL_CRYO || sw == VacSysState.SW_CRYOTEL_COLD1 || sw == VacSysState.SW_CRYOTEL_COLD2) {
                LOG.info("setting CryoTel (sw = " + sw + " ) to state " + (on ? "ON" : "OFF"));
                ((AVCCryoDevice) swDevice).setMode(on ? "ON" : "OFF");
            } else {
                LOG.info("swDevice not identified for sw=" + sw);
            }
        } catch (DriverException e) {
            throw new VacuumException(e);
        }
    }

    /**
     * Gets whether a switch is on.
     *
     * @param sw The switch number.
     * @return Whether the switch is on
     * @throws VacuumException
     */
    private Boolean isSwitchOn(int sw) {
        Boolean value = false;
        Device swDevice = switchDevices[sw];
        if (swDevice instanceof TwisTorr84Device) {
            try {
                DeviceState st = turboStateMap.get(((TwisTorr84Device) swDevice).readTurboStatus());
                value = st != DeviceState.STOPPED && st != DeviceState.BRAKING;
            } catch (DriverException e) {
                value = null;
            }

        } else if (swDevice == plutoDevc) {
	    if (plutoDevc!=null) 
		value = plutoDevc.isSwitchOn(switchChannels[sw]);

        } else if (swDevice == ionPumpDevc) {
            value = ionPumpDevc.isChannelOn(switchChannels[sw]);
        } else if (sw == VacSysState.SW_CRYOTEL_CRYO || sw == VacSysState.SW_CRYOTEL_COLD1 || sw == VacSysState.SW_CRYOTEL_COLD2) {
            try {
                //            } else if (swDevice instanceof AVCCryoDevice) {
                value = ((AVCCryoDevice) swDevice).getMode().contains("ON") || ((AVCCryoDevice) swDevice).getMode().contains("POWER");
            } catch (DriverException ex) {
                LOG.error(ex);
            }
        } else if (sw == VacSysState.SW_CRYO_VALVE) {

	    value = null;  // HN
	    if (vacState.getSwitchState(sw) != SwitchState.OFFLINE) // HN
		value = vacState.getSwitchState(sw) == SwitchState.ON;  // HN

        }
        return value;
    }

    /**
     *  Clears a latched condition.
     *
     *  @param  cond  The condition number.
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Clear a condition")
    public void clearLatch(@Argument(description="The condition number") int cond) throws VacuumException
    {
        try {
            plutoDevc.clearLatch(cond);
        }
        finally {
            publishState();
        }
    }


   /**                                                                                                                        
     *  Reads the gauges and the state of the turbo pumps.                                                                     
     *                                                                                                                         
     *  @return  Whether valid values were available for all devices                                                           
     */
    private boolean readSensors() {
        boolean valid = true;
	//	valid &= readSensor(cryoFlinePr);
	cryoFlinePr.value = 0.0;
        valid &= readSensor(cryoMainPr);
	//	valid &= readSensor(cryoTurboPr);
	cryoTurboPr.value = cryoMainPr.value;
	valid &= readSensor(cryoTurboSp);

        return valid;
    }


    /**
     *  Reads a sensor, allowing for possible temporary errors.
     * 
     *  @param  data  The sensor data object
     *  @return  Whether valid value was available                                                                                                                
     */
	 private boolean readSensor(SensorData data) {
	 Measurement meas = data.channel.getLastMeasurement();
	 if (meas.getCCSTimestamp() == null) return false;
	 data.value = meas.getValue();
	 if (!Double.isNaN(data.value)) {
	     data.goodValue = data.value;
	     data.numErrors = 0;
	 }
	 else if (data.numErrors++ < data.maxErrors) {
	     data.value = data.goodValue;
	 }
	 return true;
     }



    
    /**
     * Sets the update period.
     *
     * @param value The update period (milliseconds) to set.
     */
    @Command(type = Command.CommandType.ACTION, description = "Set the update interval")
    public void setUpdatePeriod(@Argument(description = "The tick period (ms)") int value) {
        setTickPeriod(value);
        vacState.setTickMillis(getTickPeriod());
        publishState();
    }

    /**
     * Updates the vacuum system state periodically.
     *
     * The vacuum state consists mainly of the state of the switches (lines)
     * being controlled, along with whether they can be turned on.
     *
     * Pressures and pump speeds are read to determine which switches are
     * enabled and/or need to be turned off. If the state changes it is
     * published.
     */
    private void updateVacuumState() {
//        LOG.info("updateVacuumState called");

	
	if (!running) return;

	boolean changed = monitorControl.hasPeriodChanged();
        changed |= updatePlcState();
        changed |= updateLatchState();
        changed |= updateCondState();
        if (readSensors()) {
            changed |= updateSwitchState();
	    //            changed |= updateCryoState();
        }



        for (int sw : cryotels) {
            SwitchState oldState = vacState.getSwitchState(sw);

            Boolean isOn = isSwitchOn(sw);
            SwitchState state = isOn != null ? isOn ? SwitchState.ON : SwitchState.OFF : SwitchState.OFFLINE;
            SwitchEnable enabled = SwitchEnable.ON;

	    if (state != oldState || enabled != vacState.getSwitchEnable(sw)) {
		//h                LOG.info("Enabling switch sw=" + sw + " ( " + this.getSwitchNames().get(sw)+ " ) enabled = " + enabled);
		vacState.setSwitchState(sw, state);
                vacState.setSwitchEnable(sw, enabled);
                changed = true;
            }
	    /*
            if (devState != vacState.getDeviceState(sw)) {
                vacState.setDeviceState(sw, devState);
                changed = true;
            }
	    */
        }

        if (changed) {
            publishState();
        }



	/* ----- */

        VacuumState vState;
	ComCamSharedVacState.VacuumState cvState;

        Double cryoPr = cryoMainPr.value;

    if (Double.isNaN(cryoPr)) {
            vState = VacuumState.UNKNOWN;
            cvState = ComCamSharedVacState.VacuumState.UNKNOWN;
        } else if (cryoPr <= pressVacuum && vacState.getSwitchEnable(VacSysState.SW_CRYO_VALVE) == SwitchEnable.ON) {
            vState = VacuumState.VACUUM;
            cvState = ComCamSharedVacState.VacuumState.VACUUM;
        } else if (vacState.getSwitchState(VacSysState.SW_CRYO_ION_PUMP1) == SwitchState.ON && vacState.getSwitchEnable(VacSysState.SW_CRYO_VALVE) == SwitchEnable.ON) {
            vState = VacuumState.ION_ON;
            cvState = ComCamSharedVacState.VacuumState.ION_ON;
        } else if (vacState.getSwitchEnable(VacSysState.SW_CRYO_ION_PUMP1) == SwitchEnable.ON && vacState.getSwitchEnable(VacSysState.SW_CRYO_VALVE) == SwitchEnable.ON) {
            vState = VacuumState.ION_OFF;
            cvState = ComCamSharedVacState.VacuumState.ION_OFF;
        } else if (vacState.getSwitchState(VacSysState.SW_CRYO_TURBO) == SwitchState.ON && vacState.getSwitchEnable(VacSysState.SW_CRYO_VALVE) == SwitchEnable.ON) {
            vState = VacuumState.TURBO_ON;
            cvState = ComCamSharedVacState.VacuumState.TURBO_ON;
        } else if (vacState.getSwitchEnable(VacSysState.SW_CRYO_TURBO) == SwitchEnable.ON && vacState.getSwitchEnable(VacSysState.SW_CRYO_VALVE) == SwitchEnable.ON) {
            vState = VacuumState.TURBO_OFF;
            cvState = ComCamSharedVacState.VacuumState.TURBO_OFF;
        } else if (cryoPr < PRESS_ATMOS) {
            vState = VacuumState.FORELINE;
            cvState = ComCamSharedVacState.VacuumState.FORELINE;
        } else {
            vState = VacuumState.OFF;
            cvState = ComCamSharedVacState.VacuumState.OFF;
        }
        if (vState != vacState.getVacuumState()) {
            vacState.setVacuumState(vState);
            ass.updateAgentState(vState);
            changed = true;
        }

        if (changed) {
            publishState();
        }

	// publish the shared state
        if (cvState != comCamSharedVacState.getVacuumState()) {
            comCamSharedVacState.setVacuumState(cvState);
        }
	comCamSharedVacState.setPressure(cryoPr);

	comCamSharedVacState.set_COND_CR_TRB_PRESS_10(vacState.getCondition(VacSysState.COND_CR_TRB_PRESS_10) == ConditionState.YES);
	comCamSharedVacState.set_COND_CR_VACUUM_01(vacState.getCondition(VacSysState.COND_CR_VACUUM_01) == ConditionState.YES);
	comCamSharedVacState.set_COND_CR_VACUUM_001(vacState.getCondition(VacSysState.COND_CR_VACUUM_001) == ConditionState.YES);
	comCamSharedVacState.setValveEnable(vacState.getSwitchEnable(VacSysState.SW_CRYO_VALVE) == SwitchEnable.ON);

        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(ComCamSharedVacState.KEY, getComCamSharedVacState()));

    }

    @Command(type = Command.CommandType.ACTION, description = "Enable/disable alert response")
    public void enableAlertResponse(boolean enable) {
        this.alertResponseEnabled = enable;
    }

    @Command(type = Command.CommandType.QUERY, description = "Get true/false if the alert response is enabled")
    public boolean isAlertResponseEnabled() {
        return alertResponseEnabled;
    }

    @Command(type = Command.CommandType.ACTION, description = "Enable/disable vacuum monitoring VAT shutting action")
    public void setMonAlertEnabledVATShut(boolean enable) {
        this.monAlertEnabledVATShut = enable;
    }

    @Command(type = Command.CommandType.QUERY, description = "Get true/false if the alert response is enabled")
    public boolean isMonAlertEnabledVATShut() {
        return monAlertEnabledVATShut;
    }

    @Command(type = Command.CommandType.ACTION, description = "Shut off cold cryotels")
    public void shutOffColdCryotels() {

	try {
	    ((AVCCryoDevice)switchDevices[VacSysState.SW_CRYOTEL_COLD1]).setMode("OFF");
	}
	catch (DriverException e) {
	    LOG.log(Level.SEVERE, "Error switching off cold1 cryotel", e);
	}

	Sleep(20.0); // delay between shutoffs to avoid too much vibration

	try {
	    ((AVCCryoDevice)switchDevices[VacSysState.SW_CRYOTEL_COLD2]).setMode("OFF");
	}
	catch (DriverException e) {
	    LOG.log(Level.SEVERE, "Error switching off cold2 cryotel", e);
	}

    }


    /**                                                                                                                                 
     *  Updates the state of the switches.
     *
     *  @return  Whether the state changed
     */
    private boolean updateSwitchState() {
        boolean changed = false;
        changed |= updateCryoGateValve();
        changed |= updateCryoTurboPump();
        changed |= updateCryoIonPump();

	// do a test only
	if (TEST_ONLY) {
	    LOG.info("cryoTurboPr.value = " + cryoTurboPr.value + " cryoFlinePr.value = " + cryoFlinePr.value + " cryoTurboSp.value = " + cryoTurboSp.value);
	    LOG.info("pressTurboLow = " + pressTurboLow + " pressForelineLow = " + pressForelineLow + " speedCryoTurboLow); = " + speedCryoTurboLow);
	    changed = false; 
	    LOG.info("RUNNING IN TEST MODE .... NO CHANGES WILL BE APPLIED!");
	}
        return changed;
    }


    /**                                                                                     
     *  Updates the state of the cryo gate valve.                                           
     *                                                                                      
     *  @return  Whether the state changed                                                  
     */
    private boolean updateCryoGateValve()
    {
        boolean enable = false;
	boolean turnOff = false;

	// When foreline pressure is available, insure that the pressure difference is
	// low enogh to enable opening valve with difference limits depending on the turbo running or not.
	// For ComCam, turboPr == turboPr and the following has no effect.

	boolean cryoTurbAvail = !Double.isNaN(cryoMainPr.value) && !Double.isNaN(cryoTurboPr.value) && !Double.isNaN(cryoTurboSp.value);
        if (cryoTurbAvail) {
	    double prDiff = Math.abs(cryoTurboPr.value - cryoMainPr.value);
	    enable = (prDiff <= pressDiffHigh && cryoTurboSp.value < speedCryoTurboLow)
		|| (prDiff <= pressDiffLow && cryoTurboSp.value > speedCryoTurboHigh);
	}
        
	// to be anded with conditions that either the foreline press be low or the turbo speed be low
	//        boolean enable2 = cryoTurbAvail && cmpAirPr.value > pressCmpAirLow
        boolean enable2 = cryoTurbAvail
	    && (cryoFlinePr.value < pressForelineLow || cryoTurboSp.value <= speedCryoTurboLow);
	enable &= enable2;
                    
	// since ComCam doesn't have a foreline pressure reading, impose the following condition as well
	boolean enable3 = cryoTurboPr.value > COMCAM_PRESS_HIGH || cryoTurboSp.value > speedCryoTurboHigh;
	enable &= enable3;
                    
	// do not close valve just based on a bad reading (added for ComCam)
	if (!Double.isNaN(cryoTurboPr.value)) {
	    turnOff = !enable2;
	}
	boolean forcedVATclosing = turnOff;


	//            enable = (prDiff <= pressDiffHigh && cryoTurboSp.value <= speedCryoTurboLow)
	//                        || (prDiff <= pressDiffLow && cryoTurboSp.value > speedCryoTurboLow);

	//------------------------------------------------------------------
	comCamSharedVacState.setForceValveClosing(forcedVATclosing || isMonAlertEnabledVATShut());
	
	if (forcedVATclosing || isMonAlertEnabledVATShut())
	    vacState.setSwitchState(VacSysState.SW_CRYO_VALVE, SwitchState.OFF);

	if (vacState.getSwitchState(VacSysState.SW_CRYO_VALVE) != SwitchState.OFFLINE) {
	    comCamSharedVacState.setValveOpenStateRequested(vacState.getSwitchState(VacSysState.SW_CRYO_VALVE) == SwitchState.ON);
	} else {
	    comCamSharedVacState.setValveOpenStateRequested(null);
	}
	//------------------------------------------------------------------
	

	
	return updateSwitch(VacSysState.SW_CRYO_VALVE, enable, !enable2, null, ComCamVacuumAlerts.CRYO_GATE_CLOSED);
	//	return true;
    }


    /**
     *  Updates the state of the cryo turbo pump.
     *
     *  @return  Whether the state changed
     */
    private boolean updateCryoTurboPump()
    {
        boolean enable = false;
        if (!Double.isNaN(cryoTurboPr.value) && !Double.isNaN(cryoFlinePr.value) && !Double.isNaN(cryoTurboSp.value)) {
            enable = cryoTurboPr.value < pressTurboLow
                       && (cryoFlinePr.value < pressForelineLow || cryoTurboSp.value < speedCryoTurboLow);
        }
	return updateSwitch(VacSysState.SW_CRYO_TURBO, enable, !enable, cryoTurboDevc.getDeviceState(), ComCamVacuumAlerts.CRYO_TURBO_PUMP_STOPPED);
	//        return updateSwitch(VacSysState.SW_CRYO_TURBO, enable, !enable, vacState.getDeviceState(VacSysState.SW_CRYO_TURBO), ComCamVacuumAlerts.CRYO_TURBO_PUMP_STOPPED);
	//	return true;
    }



    /**
     *  Updates the state of the cryo ion pumps.
     *
     *  @return  Whether the state changed
     */
    private boolean updateCryoIonPump()
    {
        boolean changed = false, enable = false, turnOff = false;
        if (!Double.isNaN(cryoMainPr.value)) {
            enable = cryoMainPr.value < pressIonEnable;
            if (cryoMainPr.value >= pressIonOff) {
                if (cryoIonOverStartTime == 0) {
                    cryoIonOverStartTime = System.currentTimeMillis();
                }
                else {
                    turnOff = System.currentTimeMillis() - cryoIonOverStartTime >= DELAY_ION_OFF;
                }
            }
            else {
                cryoIonOverStartTime = 0;
            }
        }
        else {
            turnOff = true;
        }

	changed |= updateSwitch(VacSysState.SW_CRYO_ION_PUMP1, enable, turnOff, null, ComCamVacuumAlerts.CRYO_ION_PUMPS_STOPPED);

	return changed;
    }




    /**                                                                                                                                                           
     *  Updates the state of the PLC latches.                                                                                                                     
     *                                                                                                                                                            
     *  @return  Whether the state changed                                                                                                                        
     */
    private boolean updateLatchState() {
        boolean changed = false;
        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            Boolean active = plutoDevc.isLatchActive(cond);
            Boolean latched = plutoDevc.isLatchLatched(cond);
            LatchState state = active == null || latched == null ? LatchState.OFFLINE :
                               latched ? LatchState.LATCHED :
                               active ? LatchState.ACTIVE : LatchState.CLEAR;
            LatchState oldState = vacState.getLatch(cond);
            if (state != oldState) {
                vacState.setLatch(cond, state);
                ComCamVacuumAlerts alert = latchAlertMap.get(cond);
                if (state == LatchState.ACTIVE) {
                    raiseAlarm(alert, "Vacuum PLC error condition set");
                }
                else if (state != LatchState.OFFLINE && oldState != LatchState.OFFLINE) {
                    if (state == LatchState.LATCHED && oldState == LatchState.CLEAR) {
                        raiseAlarm(alert, "Vacuum PLC error condition set");
                    }
                    if (state == LatchState.LATCHED || oldState == LatchState.ACTIVE) {
                        lowerAlarm(alert, "Vacuum PLC error condition cleared");
                    }
                }
                changed = true;
            }
        }
        return changed;
    }



    /**                                                                                                                                 
     *  Updates the state of the PLC conditions.                                                                                        
     *                                                                                                                                  
     *  @return  Whether the state changed                                                                                              
     */
    private boolean updateCondState() {
        boolean changed = false;
        for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
            Boolean active = plutoDevc.isConditionActive(cond);
            ConditionState state = active == null ? ConditionState.OFF :
                                   active ? ConditionState.YES : ConditionState.NO;
            if (state != vacState.getCondition(cond)) {
                vacState.setCondition(cond, state);
                changed = true;
            }
        }
        return changed;
    }




   /**                                                                                                                        
     *  Updates the state of a switch.                                                                                         
     *                                                                                                                         
     *  @param sw        The switch ID                                                                                         
     *  @param enable    Whether to enable it                                                                                  
     *  @param turnOff   Whether to turn it off                                                                                
     *  @param devState  The state of the switch's device (turbo pump only)                                                    
     *  @param alert     The alert to raise/lower if on/off state change was forced                                            
     *  @return  Whether the state changed                                                                                     
     */
 
    private boolean updateSwitch(int sw, boolean enable, boolean turnOff, DeviceState devState, ComCamVacuumAlerts alert)
    {
        boolean changed = false;
        SwitchState oldState = vacState.getSwitchState(sw);
        if (turnOff && oldState == SwitchState.ON) {
            try {
                setSwitch(sw, false);
                raiseAlarm(alert, "Switch was forced to open");
            }
            catch (VacuumException e) {
                LOG.log(Level.SEVERE, "Error setting switch: {0}", e);
            }
        }
        Boolean isOn = isSwitchOn(sw);
        SwitchState state = isOn != null ? isOn ? SwitchState.ON : SwitchState.OFF : SwitchState.OFFLINE;
        SwitchEnable enabled = enable ? SwitchEnable.ON : SwitchEnable.OFF;
        if (state != oldState || enabled != vacState.getSwitchEnable(sw)) {
            vacState.setSwitchState(sw, state);
            vacState.setSwitchEnable(sw, enabled);
            changed = true;
            if (enable && alert != null && isAlarmRaised(alert)) {
                lowerAlarm(alert, "Switch has become re-enabled");
            }
        }
        if (devState != vacState.getDeviceState(sw)) {
            vacState.setDeviceState(sw, devState);
            changed = true;
        }
        return changed;
    }



/**                                                                                     
     *  Updates the PLC state                                                               
     *                                                                                      
     *  @return  Whether the state changed                                                  
     */
    private boolean updatePlcState() {
        boolean changed = false;
        Boolean plcActive = plutoDevc.isPLCActive();
        if (plcActive == null) {
            if (vacState.getPlcState() != PLCState.OFFLINE) {
                raiseAlarm(ComCamVacuumAlerts.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC is offline");
                vacState.setPlcState(PLCState.OFFLINE);
                changed = true;
            }
        }
        else if (!plcActive) {
            if (vacState.getPlcState() != PLCState.DEAD) {
                raiseAlarm(ComCamVacuumAlerts.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC has died");
                vacState.setPlcState(PLCState.DEAD);
                changed = true;
            }
        }
        else {
            if (vacState.getPlcState() != PLCState.ALIVE) {
                lowerAlarm(ComCamVacuumAlerts.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC is alive");
                vacState.setPlcState(PLCState.ALIVE);
                changed = true;
            }
        }
        return changed;
    }

    
    /**                                                                                                                                 
     *  Raises an alarm.                                                                                                                
     *                                                                                                                                  
     *  @param  alert  The vacuum alert to raise to alarm state                                                                         
     *  @param  cond   The alert condition                                                                                              
     */
    private void raiseAlarm(ComCamVacuumAlerts alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.ALARM, cond);
        activeAlarmMap.put(alert.getId(), true);
    }


    /**                                                                                                                                                           
     *  Lowers an alarm.                                                                                                                
     *                                                                                                                                  
     *  @param  alert  The vacuum alert to lower to normal state                                                                        
     *  @param  cond   The alert condition                                                                                              
     */
    private void lowerAlarm(ComCamVacuumAlerts alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.NOMINAL, cond);
        activeAlarmMap.put(alert.getId(), false);
    }

    /**                                                                                                                                 
     *  Checks whether an alert is in alarm state.                                                                                      
     *                                                                                                                                  
     *  @param  alert  The vacuum alert to check                                                                                        
     *  @return  Whether it has been raised                                                                                             
     */
    private boolean isAlarmRaised(ComCamVacuumAlerts alert)
    {
        return activeAlarmMap.get(alert.getId()) == Boolean.TRUE;
    }



    /**
     *  Raises an alert.
     *
     *  @param  alert  The vacuum alert to raise
     *  @param  cond   The alert condition
     */
    private void raiseAlert(ComCamVacuumAlerts alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.ALARM, cond);
        activeAlertMap.put(alert.getId(), true);
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The vacuum alert to lower
     *  @param  cond   The alert condition
     */
    private void lowerAlert(ComCamVacuumAlerts alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.NOMINAL, cond);
        activeAlertMap.put(alert.getId(), false);
    }

    /**
     *  Alert event handler. 
     *
     *  Resets PLC latch when corresponding alert is cleared.
     *
     *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event) {
        
        //Handle raised alerts
        if ( event.getType() == AlertEvent.AlertEventType.ALERT_RAISED && isAlertResponseEnabled() ) {
            Alert raisedAlert = event.getAlert();
            String alertId = raisedAlert.getAlertId();
            
            if ( alertId.equals(ComCamVacuumAlerts.PRESSURE_TOO_HIGH.getAlert().getAlertId()) ) {
                if ( event.getLevel() == AlertState.ALARM ) {
		    setMonAlertEnabledVATShut(true);
		    LOG.error("Shutting the vacuum valve due to high pressure!");                    
                }
            }
            
            // shutoff the cold cryotels if the cold plate temperature becomes too low
            if ( alertId.equals(ComCamVacuumAlerts.COLD_PLATE_TEMPERATURE_TOO_LOW.getAlert().getAlertId()) ) {
                if ( event.getLevel() == AlertState.ALARM ) {
                    LOG.error("Shutting off cold cryotels due to low cold plate temperature!");
                    shutOffColdCryotels();
                }
            }
        }
        
        //Handle cleared alerts 
        if (event.getType() != AlertEvent.AlertEventType.ALERT_CLEARED) return;
        for (String id : event.getClearedIds()) {
            Integer cond = revAlertMap.get(id);
            if (cond != null) {
                try {
                    clearLatch(cond);
                }
                catch (VacuumException e) {
                    LOG.log(Level.SEVERE, "Error clearing latched PLC condition ({0}): {1}", new Object[]{cond, e});
                }
            }
        }
    }


    /**
     * Publishes the state of the vacuum system.
     *
     * This is intended to be called whenever any element of the state is
     * changed.
     */
    private void publishState() {
	        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(VacSysState.KEY, getVacuumState()));
	        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(ComCamSharedVacState.KEY, getComCamSharedVacState()));

    }

    /**
     * Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period) {
	        apts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }

    /**
     * Gets the monitoring publishing period.
     */
    private int getTickPeriod() {
	        return (int) apts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

}
