package org.lsst.ccs.subsystem.comcamvacuum;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.auxelex.IonPump;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.comcamvacuum.config.VacuumConfig;
import org.lsst.ccs.subsystem.comcamvacuum.data.VacuumException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Interface to a SLAC ion pump controller.
 *
 *  @author Owen Saxton
 */
public class IonPumpDevice extends Device {

    /**
     *  Constants.
     */
    public static final int
        CHAN_HIP1 = 0,
        CHAN_HIP2 = 1,
        CHAN_OIP  = 3,
        CHAN_CIP1 = 2;
//        CHAN_CIP2 = 4,
//        CHAN_CIP3 = 5,
//        CHAN_CIP4 = 6,
//        CHAN_CIP5 = 7,
//        CHAN_CIP6 = 8;

    private static final int
        MON_TYPE_VOLTAGE = 0,
        MON_TYPE_CURRENT = 1,
        MON_TYPE_POWER   = 2;

    private static final String
        IP_ADDRESS   = "ipAddr";

    private static final Map<String, Integer> mTypeMap = new HashMap<>();
    static {
        mTypeMap.put("VOLTAGE", MON_TYPE_VOLTAGE);
        mTypeMap.put("CURRENT", MON_TYPE_CURRENT);
        mTypeMap.put("POWER", MON_TYPE_POWER);
    }

    /**
     *  Data fields.
     */
    @LookupField (strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    @LookupField (strategy = LookupField.Strategy.CHILDREN)
    private final Map<String, IonPumpControl> ctrlChans = new LinkedHashMap<>();

    @ConfigurationParameter(name=IP_ADDRESS, category=VacuumConfig.CRYO, isFinal=true)
    private String ipAddr;

    private static final Logger LOG = Logger.getLogger(IonPumpDevice.class.getName());
    private final IonPump ipc = new IonPump();


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        if (ipAddr == null) {
            MonitorLogUtils.reportConfigError(LOG, name, IP_ADDRESS, "is missing");
        }
        fullName = "Ion pump controller (" + ipAddr + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            ipc.open(ipAddr);
            for (IonPumpControl ctrl : ctrlChans.values()) {
                ctrl.writeVoltage();
                ctrl.writeCurrent();
                ctrl.writePower();
            }
            initSensors();
            setOnline(true);
            LOG.info("Connected to " + fullName);
        }
        catch (DriverException | VacuumException e) {
            if (!inited) {
                LOG.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            ipc.close();
        }
        catch (DriverException e) {
        }
    }


    /**
     *  Checks a monitoring channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  Two-element array containing the encoded type [0] and subtype [1] 
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer mType = mTypeMap.get(type.toUpperCase());
        if (mType == null) {
            MonitorLogUtils.reportError(LOG, name, "type", type);
        }
        if (hwChan < 0 || hwChan >= IonPump.NUM_CHANNELS) {
            MonitorLogUtils.reportError(LOG, name, "hw channel number", hwChan);
        }
        return new int[]{mType, 0};
    }


    /**
     *  Reads a monitoring channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        String item = null;
        if (isOnline()) {
            try {
                switch (type) {
                case MON_TYPE_VOLTAGE:
                    item = "voltage";
                    value = ipc.readVoltage(hwChan);
                    break;

                case MON_TYPE_CURRENT:
                    item = "current";
                    value = ipc.readCurrent(hwChan);
                    break;

                case MON_TYPE_POWER:
                    item = "power";
                    value = ipc.readPower(hwChan);
                    break;
                }
            }
            catch (DriverException e) {
                LOG.error("Error reading " + fullName + " " + item + ": " + e);
                setOnline(false);
            }
        }
        return value;
    }


    /**
     *  Turns a channel on or off
     *
     *  @param  chan  The hardware channel number
     *  @param  on    Whether to set on or off
     */
    protected void setChannelOn(int chan, boolean on)
    {
        try {
            if (on) {
                ipc.powerOn(chan);
            }
            else {
                ipc.powerOff(chan);
            }
        }
        catch (DriverException e) {
            LOG.error("Error setting channel " + chan + " on " + fullName);
        }
        
    }


    /**
     *  Gets the on state of a channel.
     *
     *  @param  chan  The hardware channel number
     *  @return  Whether the line is on or off
     */
    protected Boolean isChannelOn(int chan)
    {
        if (!isOnline()) return null;
        try {
            return ipc.isPowered(chan);
        }
        catch (DriverException e) {
            LOG.error("Error getting state for channel " + chan + " on " + fullName);
            return null;
        }
    }


    /**
     *  Gets the list of defined channel names.
     *
     *  @return  The channel names
     */
    public List<String> getChannelNames()
    {
        return new ArrayList<>(ctrlChans.keySet());
    }


    /**
     *  Gets the list of defined channel numbers.
     *
     *  @return  The channel numbers
     */
    public List<Integer> getChannelNumbers()
    {
        List<Integer> chans = new ArrayList<>();
        for (IonPumpControl ctrl : ctrlChans.values()) {
            chans.add(ctrl.getHwChan());
        }
        return chans;
    }


    /**
     *  Sets the voltage for a channel.
     *
     *  @param  chan   The channel number
     *  @param  value  The voltage
     *  @throws  VacuumException
     */
    void setVoltage(int chan, double value) throws VacuumException
    {
        try {
            ipc.setVoltage(chan, value);
        }
        catch (DriverException e) {
            throw new VacuumException(e);
        }
    }


    /**
     *  Sets the current for a channel.
     *
     *  @param  chan   The channel number
     *  @param  value  The current
     *  @throws  VacuumException
     */
    void setCurrent(int chan, double value) throws VacuumException
    {
        try {
            ipc.setCurrent(chan, value);
        }
        catch (DriverException e) {
            throw new VacuumException(e);
        }
    }


    /**
     *  Sets the power for a channel.
     *
     *  @param  chan   The channel number
     *  @param  value  The power
     *  @throws  VacuumException
     */
    void setPower(int chan, double value) throws VacuumException
    {
        try {
            ipc.setPower(chan, value);
        }
        catch (DriverException e) {
            throw new VacuumException(e);
        }
    }


    /**
     *  Powers on a channel.
     *
     *  @param  chan  The channel number
     *  @throws  VacuumException
     */
    void powerOn(int chan) throws VacuumException
    {
        try {
            ipc.powerOn(chan);
        }
        catch (DriverException e) {
            throw new VacuumException(e);
        }
    }


    /**
     *  Powers off a channel.
     *
     *  @param  chan  The channel number
     *  @throws  VacuumException
     */
    void powerOff(int chan) throws VacuumException
    {
        try {
            ipc.powerOff(chan);
        }
        catch (DriverException e) {
            throw new VacuumException(e);
        }
    }


    /**
     *  Gets named controller.
     *
     *  @param  name  The channel name
     *  @throws  VacuumException
     */
    private IonPumpControl getControl(String name) throws VacuumException
    {
        IonPumpControl ctrl = ctrlChans.get(name);
        if (ctrl == null) {
            throw new VacuumException("Unrecognized ion pump name: " + name);
        }
        return ctrl;
    }

}
