package org.lsst.ccs.subsystem.comcamvacuum;

import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.monitor.Alarm;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.comcamvacuum.alerts.ComCamVacuumAlerts;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * A class with information about ComCamVacuum alarms
 *
 * @author The LSST CCS Team
 */
public class ComCamVacuumAlertHandler implements HasLifecycle, Monitor.AlarmHandler, ClearAlertHandler {

    private static final Logger LOG = Logger.getLogger("org.lsst.ccs.subsystem.comcamvacuum.alarms");

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    //The ComCam Vacuum subsystem
    @LookupField(strategy = LookupField.Strategy.TREE)
    private ComCamVacuumSubsystem comcam;


    @Override
    public boolean processAlarm(int event, int parm, String cause, String alarmName) {

	LOG.info("processing alarms .... alert response enable = "+comcam.isAlertResponseEnabled());

        if ( (parm == ComCamVacuumAlerts.PRESSURE_TOO_HIGH.ordinal()) && comcam.isAlertResponseEnabled() ) {
            alertService.raiseAlert(ComCamVacuumAlerts.PRESSURE_TOO_HIGH.getAlert(), getAlarmStateFromEvent(event), alarmName + ": " + cause);            
	    switch (event) {
	    case Alarm.EVENT_TRIP:                    
		//Alarm triggered
		//		try { //Shut the valve, then fork off the rest of the response
		    comcam.setMonAlertEnabledVATShut(true);
		    LOG.error("Shutting the vacuum valve due to high pressure!");
		/*
		} catch (DriverException de) {
                    LOG.severe("Failed to shut VACUUMVALVE outlet -- manual intervention needed" + de);
		}
		*/
		Thread t = new Thread(new VacuumAlarmResponse());
		t.start();
		break;
	    case Alarm.EVENT_LIMBO_FROM_ERROR:
		//In Warning band from Alarm
		break;
	    case Alarm.EVENT_LIMBO_FROM_GOOD:
		//In Warning band from Nominal
		break;
	    case Alarm.EVENT_RESET:
		//Back to Nominal
		break;
            }
            return true;
	}

	// shutoff the cold cryotels if the cold plate temperature becomes too low
        if ( (parm == ComCamVacuumAlerts.COLD_PLATE_TEMPERATURE_TOO_LOW.ordinal()) && comcam.isAlertResponseEnabled() ) {
            alertService.raiseAlert(ComCamVacuumAlerts.COLD_PLATE_TEMPERATURE_TOO_LOW.getAlert(), getAlarmStateFromEvent(event), alarmName + ": " + cause);            
	    switch (event) {
	    case Alarm.EVENT_TRIP:                    
		LOG.error("Shutting off cold cryotels due to low cold plate temperature!");

		comcam.shutOffColdCryotels();


		Thread t = new Thread(new VacuumAlarmResponse());
		t.start();
		break;
	    case Alarm.EVENT_LIMBO_FROM_ERROR:
		//In Warning band from Alarm
		break;
	    case Alarm.EVENT_LIMBO_FROM_GOOD:
		//In Warning band from Nominal
		break;
	    case Alarm.EVENT_RESET:
		//Back to Nominal
		break;
            }
            return true;
	}




        return false;
    }


    class VacuumAlarmResponse implements Runnable {
        @Override
        public void run() {
	    // place holder
        }
    }


    private static AlertState getAlarmStateFromEvent(int event) {
        switch (event) {
            //The High Alarm treshold has been crossed.
	case Alarm.EVENT_TRIP:
	    return AlertState.ALARM;
	case Alarm.EVENT_LIMBO_FROM_ERROR:
	    return AlertState.WARNING;
	case Alarm.EVENT_LIMBO_FROM_GOOD:
	    return AlertState.WARNING;
	case Alarm.EVENT_RESET:
	    return AlertState.NOMINAL;
        }
        throw new IllegalArgumentException("Unknown event: id " + event);
    }
 
    
    @Override
    public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState state) {

        for ( ComCamVacuumAlerts comcamAlert : ComCamVacuumAlerts.values() ) {
            if ( comcamAlert.getId().equals(alert.getAlertId() ) ) {
                Alarm a = getAlarmById(comcamAlert.ordinal());
                if ( a != null ) {
                    a.clearState();
                    return ClearAlertCode.CLEAR_ALERT;
                }
            }
        }

        return ClearAlertCode.UNKNOWN_ALERT;
    }
    
    
}
