package org.lsst.ccs.subsystem.comcamvacuum.ui.jas;

import java.awt.BorderLayout;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import javax.swing.JComponent;
import javax.swing.JPanel;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.EmbeddedObjectDeserializationException;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.base.OptionalPage;
import org.lsst.ccs.gconsole.plugins.monitor.SectionedTableView;
import org.lsst.ccs.gconsole.plugins.monitor.TabbedView;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.subsystem.comcamvacuum.data.ComCamVacuumAgentProperties;
import org.lsst.ccs.subsystem.comcamvacuum.data.VacSysState;
import org.lsst.ccs.subsystem.comcamvacuum.ui.ComCamVacControlPanel;
import org.lsst.ccs.subsystem.comcamvacuum.ui.ComCamVacuumPage;

/**
 *
 * @author tonyj
 */
@Plugin(name = "ComCam Vacuum Control Plugin", description = "ComCam Vacuum Control Plugin")
public class ComCamVacControlPlugin extends ConsolePlugin {

    @Override
    public void initialize() {
        OptionalPage.getService().add(new ComCamVacImageOptionalPage());
        OptionalPage.getService().add(new ComCamVacControlOptionalPage());
    }
    
    static class ComCamVacControlOptionalPage implements OptionalPage {

        private final CommandSender cmdSender = new CommandSender();
        private final Map<String,BusListener> busListeners = new ConcurrentHashMap<>();
        
        @Override
        public boolean isAutoOpen() {
            return false;
        }

        @Override
        public String getPage(AgentInfo agent) {
            return agent != null && agent.hasAgentProperty(ComCamVacuumAgentProperties.COMCAMVACUUM_TYPE_AGENT_PROPERTY)
                     ? getPageName(agent) : null;
        }

        @Override
        public boolean agentDisconnected(AgentInfo agent) {
            BusListener listener = busListeners.get(getPageName(agent));
            if (listener == null) return true;
            listener.controls.disableSystem();
            return false;
        }
        
        @Override
        public JComponent open(AgentInfo agent, JComponent existingComponent) {
            String pageName = getPageName(agent);
            String agentName = agent.getName();
            BusListener listener = busListeners.get(pageName);
            if (listener != null) {
                initControls(agentName, listener.controls);
                return existingComponent;
            }

            JPanel panel = new JPanel();
            panel.setLayout(new java.awt.BorderLayout());
            panel.setName(agentName + " Control");

            //Create the monitoring panel
            TabbedView view = new TabbedView();
            view.setViewFactory(name -> new SectionedTableView());            
            ComCamVacControlFilter monitorFilter = new ComCamVacControlFilter(agent.getName());
            view.setFilter(monitorFilter);
            view.install();
            panel.add(view.getPanel(), BorderLayout.CENTER);

            ComCamVacControlPanel controls = new ComCamVacControlPanel(cmdSender);
            controls.setSubsystem(agent.getName());
            panel.add(controls, BorderLayout.SOUTH);
            
            listener = new BusListener(controls, agent.getName());
            AgentMessagingLayer aml = Console.getConsole().getMessagingAccess();
            aml.addStatusMessageListener(listener, (msg) -> msg.getOriginAgentInfo().getName().equals(agentName)
                                                              && msg instanceof StatusSubsystemData);
            busListeners.put(pageName, listener);

            initControls(agentName, controls);
            return panel;
        }

        @Override
        public void closed(String pageName) {
            BusListener listener = busListeners.remove(pageName);
            if (listener == null) {
                throw new RuntimeException("No page exists with the name " + pageName);
            }
            Console.getConsole().getMessagingAccess().removeStatusMessageListener(listener);
        }
        
        private String getPageName(AgentInfo agent) {
            return "CCS Subsystems/" + agent.getName() + "/Control";
        }
        
        private void initControls(String agentName, ComCamVacControlPanel controls) {
            Object response = cmdSender.sendCommand(agentName, null, "getVacuumState");
            if (response instanceof VacSysState) {
                controls.updateControlPanel((VacSysState)response);
            }
        }

    }
    
    static class ComCamVacImageOptionalPage implements OptionalPage {
        
        private ComCamVacuumPage vacuumPage;

        @Override
        public boolean isAutoOpen() {
            return false;
        }

        @Override
        public String getPage(AgentInfo agent) {
            return agent != null && agent.hasAgentProperty(ComCamVacuumAgentProperties.COMCAMVACUUM_TYPE_AGENT_PROPERTY)
                     ? "CCS Subsystems/" + agent.getName() + "/Schematic" : null;
        }
        
        @Override
        public JComponent open(AgentInfo agent, JComponent existingComponent) {
            vacuumPage = new ComCamVacuumPage(agent);
            JComponent panel = vacuumPage.getView().getPanel();
            panel.setName(agent.getName() + " Schematic");
            return panel;
        }

        @Override
        public void opened(String page) {
            vacuumPage.getView().install();
        }

        @Override
        public void closed(String page) {
            vacuumPage.getView().uninstall();
            vacuumPage = null;
        }
    }

    static class BusListener implements StatusMessageListener {
        
        private final ComCamVacControlPanel controls;
        private final String agentName;
        
        BusListener(ComCamVacControlPanel controls, String agentName) {
            this.controls = controls;
            this.agentName = agentName;
        }
        
        @Override
        public void onStatusMessage(StatusMessage msg) {
            try {
                StatusSubsystemData sd = (StatusSubsystemData) msg;
                if (sd.getDataKey().equals(VacSysState.KEY)) {
                    controls.updateControlPanel((VacSysState)((KeyValueData) sd.getSubsystemData()).getValue());
                }
            }
            catch (EmbeddedObjectDeserializationException x) {
            }
        }

    }

}
