package org.lsst.ccs.subsystem.cluster.monitor;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;
import org.influxdb.dto.Point;
import org.influxdb.dto.Point.Builder;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandMessage;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.localdb.utils.InfluxDbClientService;
import org.lsst.ccs.messaging.CommandMessageListener;
import org.lsst.ccs.messaging.LogMessageListener;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;
import org.python.netty.util.internal.ConcurrentSet;

/**
 * A Subsystem that tracks the bus traffic.
 *
 * @author The LSST CCS Team
 */
public class BusTrafficMonitor implements HasLifecycle {

    private static final Object counterLock = new Object();

    private static final Logger LOG = Logger.getLogger(BusTrafficMonitor.class.getName());

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    InfluxDbClientService influxDbClientService;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService agentPropertiesService;

    
    private static BlockingQueue<Runnable> statusQueue = null;
    private static BlockingQueue<Runnable> commandQueue = null;
    private static BlockingQueue<Runnable> logQueue = null;

    private Thread statusThread;
    private Thread commandThread;
    private Thread logThread;

    @Override
    public void build() {

        agentPropertiesService.setAgentProperty("org.lsst.ccs.use.full.paths", "true");

        AgentPeriodicTask resetCounterAndPublish = new AgentPeriodicTask("busTrafficStatsUpdate",
                () -> {
                    resetCounterAndPublish();
                }).withPeriod(Duration.ofSeconds(60));

        agent.getAgentService(AgentPeriodicTaskService.class).scheduleAgentPeriodicTask(resetCounterAndPublish);

    }

    private void resetCounters() {
        Thread oldStatusThread = statusThread;
        Thread oldCommandThread = commandThread;
        Thread oldLogThread = logThread;
        synchronized (counterLock) {
            agentCountersMap = new ConcurrentHashMap<>();
            tagsAccumMap = new ConcurrentHashMap<>();

            //Last element of the queue to terminate the thread.
            if (statusQueue != null) {
                statusQueue.add(new DoneWork());
                commandQueue.add(new DoneWork());
                logQueue.add(new DoneWork());
            }

            statusQueue = new LinkedBlockingQueue<>();
            commandQueue = new LinkedBlockingQueue<>();
            logQueue = new LinkedBlockingQueue<>();

            statusThread = new MyQueueWorker(statusQueue);
            commandThread = new MyQueueWorker(commandQueue);
            logThread = new MyQueueWorker(logQueue);

            statusThread.start();
            commandThread.start();
            logThread.start();

        }
        try {
            if (oldStatusThread != null) {
                oldStatusThread.join();
            }
            if (oldCommandThread != null) {
                oldCommandThread.join();
            }
            if (oldLogThread != null) {
                oldLogThread.join();
            }
        } catch (InterruptedException ie) {
            throw new RuntimeException(ie);
        }

    }

    @Override
    public void start() {
        resetCounters();
        agent.getMessagingAccess().addCommandMessageListener(new CommandBusCounterListener());
        agent.getMessagingAccess().addStatusMessageListener(new StatusBusCounterListener());
        agent.getMessagingAccess().addLogMessageListener(new LogBusCounterListener());
    }


    private void resetCounterAndPublish() {
        
        Map<Tags, Integer> agentCountersMapOld;
        Map<Tags, Map<String,Accumulate>> agentTimeAccumulatorsMapOld;

        synchronized (counterLock) {
            agentCountersMapOld = agentCountersMap;
            agentTimeAccumulatorsMapOld = tagsAccumMap;
        }
        resetCounters();

        if (influxDbClientService!= null && influxDbClientService.getInfluxDbClient() != null) {
            Set<Tags> allTags = new ConcurrentSet<>();
            allTags.addAll(agentCountersMapOld.keySet());
            allTags.addAll(agentTimeAccumulatorsMapOld.keySet());

            long time = System.currentTimeMillis();
            for (Tags tag : allTags) {
                Builder pointBuilder = Point.measurement("cluster_metrics")
                        .time(time, TimeUnit.MILLISECONDS);
                Integer counter = agentCountersMapOld.get(tag);
                if (counter != null) {
                    pointBuilder = pointBuilder.addField("traffic", counter);
                }

                Map<String,Accumulate> accumMap = agentTimeAccumulatorsMapOld.get(tag);                
                for (String key : accumMap.keySet()) {
                    Accumulate accum = accumMap.get(key);

                    if (accum.getCounts() > 0) {
                        pointBuilder = pointBuilder.addField(key + "_avg", (double) accum.getAverageValue()).
                                addField(key + "_max", (double) accum.getMaxValue()).
                                addField(key + "_min", (double) accum.getMinValue());

                    }
                }
                
                if (pointBuilder.hasFields()) {
                    AgentType agentType = tag.agentInfo.getType();
                    String agentName = agentType == AgentType.CONSOLE ? "console" : tag.agentInfo.getName();
                    Point point = pointBuilder.tag("bus", tag.bus.name()).tag("agent", agentName).
                            tag("host", tag.agentInfo.getAgentProperty("org.lsst.ccs.agent.hostname", "")).
                            tag("user", tag.agentInfo.getAgentProperty("org.lsst.ccs.agent.username", "")).
                            tag("type", agentType.name()).build();                    
                    LOG.fine("Writing to influxDb " + point);
                    influxDbClientService.getInfluxDbClient().write(point);
                }
            }

        } else {
            Set<Tags> allTags = new ConcurrentSet<>();
            allTags.addAll(agentCountersMapOld.keySet());
            allTags.addAll(agentTimeAccumulatorsMapOld.keySet());

            KeyValueDataList dataList = new KeyValueDataList("cluster_metrics");
            HashMap<String,Integer> allCounters = new HashMap<>();
            
            for (Tags tag : allTags) {
                String busName = tag.bus.name();
                String agentName = tag.agentInfo.getType() == AgentType.CONSOLE ? "console" : tag.agentInfo.getName();

                Integer counter = agentCountersMapOld.get(tag);
                if (counter != null) {
                    dataList.addData("traffic/"+busName+"/"+agentName, counter);
                    allCounters.put(busName, allCounters.getOrDefault(busName, 0)+counter);
                }

                Map<String,Accumulate> accumMap = agentTimeAccumulatorsMapOld.get(tag);                
                for (String key : accumMap.keySet()) {
                    Accumulate accum = accumMap.get(key);

                    if (accum.getCounts() > 0) {
                        dataList.addData(key+"_avg/"+busName+"/"+agentName,accum.getAverageValue());
                        dataList.addData(key+"_max/"+busName+"/"+agentName,accum.getMaxValue());
                        dataList.addData(key+"_min/"+busName+"/"+agentName,accum.getMinValue());                                
                    }
                }
                
            }

            for ( String busName : allCounters.keySet() ) {
                dataList.addData("traffic/"+busName+"/all", allCounters.get(busName));
            }
            
            
            agent.publishSubsystemDataOnStatusBus(dataList);
        }

    }

    private static class Tags {

        private final Bus bus;
        private final AgentInfo agentInfo;

        Tags(Bus bus, AgentInfo agentInfo) {
            this.bus = bus;
            this.agentInfo = agentInfo;
        }

        @Override
        public boolean equals(Object obj) {
            if (obj instanceof Tags) {
                Tags t1 = (Tags) obj;
                return t1.bus.equals(bus) && t1.agentInfo.equals(agentInfo);
            }
            return false;
        }

        @Override
        public int hashCode() {
            return bus.hashCode() + 34 * agentInfo.hashCode();
        }

    }

    private static Map<Tags, Integer> agentCountersMap = new ConcurrentHashMap<>();
    private static Map<Tags, Map<String,Accumulate>> tagsAccumMap = new ConcurrentHashMap<>();

    private static void incrementAgentCounter(Tags tags) {
        Integer agentCounter = agentCountersMap.get(tags);
        if (agentCounter == null) {
            agentCounter = 0;
        }
        agentCountersMap.put(tags, ++agentCounter);
    }

    private static void incrementAgentKeyAccumulator(Tags tags, String key, double value) {
        Map<String,Accumulate> keyAccumMap = tagsAccumMap.get(tags);
        if (keyAccumMap == null) {
            keyAccumMap = new HashMap<>();
            tagsAccumMap.put(tags, keyAccumMap);
        }
        Accumulate accum = keyAccumMap.get(key);
        if ( accum == null ) {
            accum = new Accumulate();
            keyAccumMap.put(key,accum);
        }
        accum.accumulate(value);
    }


    private static class CommandBusCounterListener implements CommandMessageListener {

        @Override
        public void onCommandMessage(CommandMessage msg) {
            commandQueue.add(() -> processBusMessage(Bus.COMMAND,msg, CCSTimeStamp.currentTime()));
        }
    }
    
    private static void processBusMessage(Bus bus, BusMessage msg, CCSTimeStamp received) {
            AgentInfo agentInfo = msg.getOriginAgentInfo();
            Tags tags = new Tags(bus, agentInfo);

            long deltaT = Duration.between(msg.getCCSTimeStamp().getUTCInstant(),received.getUTCInstant()).toMillis();
            
            long transferTime = msg.getTransferDuration() != null ? msg.getTransferDuration().toMillis() : 0;
            long processingOut = msg.getSerializationTime() != null ? Duration.between(msg.getCCSTimeStamp().getUTCInstant(), msg.getSerializationTime().getUTCInstant()).toMillis() : 0;
            long processingIn = 0;
            if ( msg.getDeSerializationTime() != null && received != null ) {           
                processingIn = Duration.between(msg.getDeSerializationTime().getUTCInstant(),received.getUTCInstant()).toMillis();
            } 
            double size = sizeOf(msg);
            synchronized (counterLock) {
                incrementAgentCounter(tags);

                incrementAgentKeyAccumulator(tags, "transfer_total", deltaT);
                incrementAgentKeyAccumulator(tags, "transfer", transferTime);
                incrementAgentKeyAccumulator(tags, "process_out", processingOut);
                incrementAgentKeyAccumulator(tags, "process_in", processingIn);
                incrementAgentKeyAccumulator(tags, "size", size);
            }
        
    }
    

    private static class LogBusCounterListener implements LogMessageListener {

        @Override
        public void onLogMessage(LogMessage msg) {
            logQueue.add(() -> processBusMessage(Bus.LOG,msg, CCSTimeStamp.currentTime()));
        }

    }

    private static class StatusBusCounterListener implements StatusMessageListener {

        @Override
        public void onStatusMessage(StatusMessage msg) {
            statusQueue.add(() -> processBusMessage(Bus.STATUS,msg, CCSTimeStamp.currentTime()));
        }

    }

    public static double sizeOf(Serializable object) {
        ByteArrayOutputStream byteOutputStream = new ByteArrayOutputStream();
        try (ObjectOutputStream objectOutputStream = new ObjectOutputStream(byteOutputStream)) {
            objectOutputStream.writeObject(object);
            objectOutputStream.flush();
        } catch (IOException ieo) {
            return 0;
        }

        return (double)byteOutputStream.toByteArray().length / 1000.;
    }

    //Executes the elements in the queue until a null Runnable is added.    
    private class MyQueueWorker extends Thread {

        private final BlockingQueue<Runnable> queue;

        public MyQueueWorker(BlockingQueue<Runnable> queue) {
            this.queue = queue;
        }

        public void run() {
            try {
                while (true) {
                    Runnable r = queue.take();
                    if (r instanceof DoneWork) {
                        break;
                    }
                    r.run();
                }
            } catch (InterruptedException ie) {
                // just terminate
            }
        }

    }

    private class DoneWork implements Runnable {

        @Override
        public void run() {
            throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
        }

    }

}
