package org.lsst.ccs.subsystem.cluster.monitor;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.localdb.statusdb.server.DataChannel;
import org.lsst.ccs.localdb.statusdb.server.DataChannel.DataChannelList;
import org.lsst.ccs.localdb.utils.TrendingConnectionUtils;
import org.lsst.ccs.messaging.jgroups.ClusterSplitUtils;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.DataProviderDictionaryService;

/**
 * A Service to monitor the cluster activity.
 * 
 * @author The LSST CCS Team
 */
public class ClusterMonitorService extends Subsystem implements HasLifecycle {
        

    @LookupField(strategy = LookupField.Strategy.TREE)
    AgentPropertiesService agentPropertiesService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    DataProviderDictionaryService dataProviderDictionaryService;


    ClusterMonitorService() {
        super("clusterMonitor",AgentInfo.AgentType.SERVICE);
    }
    
    @Override
    public void build() {
        agentPropertiesService.setAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, AgentCategory.CLUSTER_MONITOR.name());
    }

    @Command(type = Command.CommandType.ACTION)
    public void splitCluster(int nSplits) {
        Set<String> allAgents = new HashSet<>();
        for ( AgentInfo ai : getMessagingAccess().getAgentPresenceManager().listConnectedAgents() ) {
            allAgents.add(ai.getName());
        }
        ClusterSplitUtils.splitCluster(nSplits, allAgents);
    }

    
    @Command(type = Command.CommandType.QUERY)
    public String checkDictionariesForTrending() {
        
        StringBuilder sb = new StringBuilder();
        TrendingConnectionUtils connUtils = null;
        for (AgentInfo ai : getMessagingAccess().getAgentPresenceManager().listConnectedAgents()) {
            if ( ai.getAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, "none").equals(AgentCategory.REST_SERVER.name()) ) {
                connUtils = new TrendingConnectionUtils(ai.getAgentProperty("rest-service-addr"), Integer.valueOf(ai.getAgentProperty("rest-service-port")));
            }
        }
        if ( connUtils == null ) {
            sb.append("Could not find rest server\n");
            return sb.toString();                    
        }
        
        List<String> channels = new ArrayList<>();
        
        DataChannelList channelList = connUtils.getChannelList();
        sb.append("Trending Channels: \n");
        for ( DataChannel ch : channelList.list ) {
            String channelPath = ch.getPathAsString();
            if ( channelPath.startsWith("/") ) {
                channelPath = channelPath.substring(1);
            }
            if ( channels.contains(channelPath) ) {
                sb.append("WARNING duplicate channel ").append(channelPath).append("\n");
            }
            channels.add(channelPath);
//            sb.append(ch.getPathAsString()).append("\n");            
        }
        
        
        sb.append("\nChecking Trending data for worker subsystems\n");
        for (AgentInfo ai : getMessagingAccess().getAgentPresenceManager().listConnectedAgents()) {
            if (ai.getType() == AgentInfo.AgentType.WORKER) {
                DataProviderDictionary dataDict = dataProviderDictionaryService.getDataProviderDictionaryForAgent(ai.getName());
                if ( dataDict == null ) {
                    sb.append("WARNING: no dictionary\n");
                    continue;
                }
                sb.append("WORKER: ").append(ai.getName()).append("\n");
                        
                int found = 0;
                int doNotTrend = 0;
                int missing = 0;
                for (DataProviderInfo dpi : dataDict.getDataProviderInfos()) {
                    if (dpi.getAttributeValue(DataProviderInfo.Attribute.DATA_TYPE).equals(DataProviderInfo.Type.MONITORING.name()) || dpi.getAttributeValue(DataProviderInfo.Attribute.DATA_TYPE).equals(DataProviderInfo.Type.TRENDING.name()) ) {
                        if ( dpi.getAttributeValue(DataProviderInfo.Attribute.DO_NOT_TREND) != null ) {
                            doNotTrend++;
                            continue;
                        }
                        String trendingPath = ai.getName()+"/"+dpi.getFullPath();
                        if ( !channels.contains(trendingPath) ) {
                            sb.append(trendingPath).append("\n");
                            missing++;
                        } else {
                            found++;
                        }
                    }
                }
                sb.append("Summary for ").append(ai.getName()).append(" found: ").append(found).append(" doNotTrend: ").append(doNotTrend).append(" missing: ").append(missing).append("\n");
            }
        }
        return sb.toString();        
    }
    
}
