package org.lsst.ccs.subsystem.cluster.monitor;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.influxdb.dto.BatchPoints;
import org.influxdb.dto.Point;
import org.influxdb.dto.Point.Builder;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandMessage;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandReply;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.InfluxDbClientService;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessagePreProcessor;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.imagehandling.data.ImageHeaderKeywords;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * A Subsystem that tracks the bus traffic.
 *
 * @author The LSST CCS Team
 */
public class BusTrafficMonitor implements HasLifecycle, AgentPresenceListener {

    private static final Object counterLock = new Object();

    private static final Logger LOG_INFLUX = Logger.getLogger(BusTrafficMonitor.class.getName()+".influx");
    private static final Logger LOG_MSG = Logger.getLogger(BusTrafficMonitor.class.getName()+".messages");
    private static final Logger LOG_APL = Logger.getLogger(BusTrafficMonitor.class.getName()+".apl");

    private static final Alert CLUSTER_SPLIT_ALERT = new Alert("clusterSplit", "Alert raised when the CCS cluster splits");
    
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    InfluxDbClientService influxDbClientService;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    AlertService alertService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService agentPropertiesService;

    private static final HashMap<UUID,CommandMsg> commands = new HashMap<>();
    
    private static BlockingQueue<Runnable> statusQueue = null;
    private static BlockingQueue<Runnable> commandQueue = null;
    private static BlockingQueue<Runnable> logQueue = null;

    private Thread statusThread;
    private Thread commandThread;
    private Thread logThread;

    private final CommandBusCounterListener commandBusListener = new CommandBusCounterListener();
    private final StatusBusCounterListener statusBusListener = new StatusBusCounterListener();
    private final LogBusCounterListener logBusListener = new LogBusCounterListener();
    
    
    @Override
    public void build() {

        AgentPeriodicTask resetCounterAndPublish = new AgentPeriodicTask("busTrafficStatsUpdate",
                () -> {
                    resetCounterAndPublish();
                }).withPeriod(Duration.ofSeconds(60));

        agent.getAgentService(AgentPeriodicTaskService.class).scheduleAgentPeriodicTask(resetCounterAndPublish);


        agent.getMessagingAccess().addBusMessagePreProcessor(commandBusListener);
        agent.getMessagingAccess().addBusMessagePreProcessor(logBusListener);
        agent.getMessagingAccess().addBusMessagePreProcessor(statusBusListener);
        
        alertService.registerAlert(CLUSTER_SPLIT_ALERT);        
    }

    
    
    private void resetCounters() {
        Thread oldStatusThread = statusThread;
        Thread oldCommandThread = commandThread;
        Thread oldLogThread = logThread;
        synchronized (counterLock) {
            agentCountersMap = new ConcurrentHashMap<>();
            tagsAccumMap = new ConcurrentHashMap<>();

            //Last element of the queue to terminate the thread.
            if (statusQueue != null) {
                statusQueue.add(new DoneWork());
                commandQueue.add(new DoneWork());
                logQueue.add(new DoneWork());
            }

            statusQueue = new LinkedBlockingQueue<>();
            commandQueue = new LinkedBlockingQueue<>();
            logQueue = new LinkedBlockingQueue<>();

            statusThread = new MyQueueWorker(statusQueue);
            commandThread = new MyQueueWorker(commandQueue);
            logThread = new MyQueueWorker(logQueue);

            statusThread.start();
            commandThread.start();
            logThread.start();

        }
        try {
            if (oldStatusThread != null) {
                oldStatusThread.join();
            }
            if (oldCommandThread != null) {
                oldCommandThread.join();
            }
            if (oldLogThread != null) {
                oldLogThread.join();
            }
        } catch (InterruptedException ie) {
            throw new RuntimeException(ie);
        }

    }

    @Override
    public void start() {
        resetCounters();
        agent.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);
    }

    @Override
    public void shutdown() {
        agent.getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(this);
    }

    private Map<String, CCSTimeStamp> disconnectedAgents = new ConcurrentHashMap<>();

    
    @Override
    public void connecting(AgentInfo... agents) {
        LOG_APL.log(Level.FINE,"Connecting Agents: {0}",Arrays.asList(agents));        
    }

    @Override
    public void connected(AgentInfo... agents) {
        LOG_APL.log(Level.FINE,"Connected Agents: {0}",Arrays.asList(agents));        
        Map<String, Duration> reconnected = new HashMap<>();
        for (AgentInfo agent : agents) {
            String agentName = agent.getName();
            CCSTimeStamp ts = disconnectedAgents.remove(agentName);
            if ( ts != null ) {
                reconnected.put(agentName, Duration.between(ts.getUTCInstant(), CCSTimeStamp.currentTime().getUTCInstant()));
            }
        }
        if ( ! reconnected.isEmpty() ) {
            String msg = "Reconnected agents: ";
            for ( String agent : reconnected.keySet() ) {
                msg += agent+" ("+reconnected.get(agent).getSeconds()+"s) ";
            }
            LOG_APL.log(Level.WARNING,msg);
            if ( disconnectedAgents.isEmpty() ) {
                alertService.raiseAlert(CLUSTER_SPLIT_ALERT, AlertState.NOMINAL, "All missing agents have rejoined the cluster.");                
            } else {
                alertService.raiseAlert(CLUSTER_SPLIT_ALERT, AlertState.WARNING, "The following agents are still missing: "+disconnectedAgents.keySet());                                
            }
        }
    }

    @Override
    public void disconnected(AgentInfo... agents) {
        LOG_APL.log(agents.length < 2 ? Level.FINE : Level.WARNING ,"Disconnecting Agent: {0}",Arrays.asList(agents));        
        if ( agents.length >= 2 ) {
            alertService.raiseAlert(CLUSTER_SPLIT_ALERT, AlertState.WARNING, "These agents left the cluster: "+Arrays.asList(agents));
            CCSTimeStamp now = CCSTimeStamp.currentTime();
            for (AgentInfo agent : agents) {
                disconnectedAgents.put(agent.getName(), now);
            }
        }
        
    }


    

    private void resetCounterAndPublish() {
        
        Map<Tags, Integer> agentCountersMapOld;
        Map<Tags, Map<String,Accumulate>> agentTimeAccumulatorsMapOld;

        synchronized (counterLock) {
            agentCountersMapOld = agentCountersMap;
            agentTimeAccumulatorsMapOld = tagsAccumMap;
        }
        resetCounters();

        if (influxDbClientService.isEnabled()) {
            Set<Tags> allTags = ConcurrentHashMap.newKeySet();
            allTags.addAll(agentCountersMapOld.keySet());
            allTags.addAll(agentTimeAccumulatorsMapOld.keySet());

            HashMap<String, Integer> typeCount = new HashMap<>();
                       
            BatchPoints bp = influxDbClientService.createBatchPoints();
            long time = System.currentTimeMillis();
            //Now that we have more multiplicity in the Tag class we have to
            //make sure we single count agents.
            List<String> alreadyCountedAgentName = new ArrayList();            
            for (Tags tag : allTags) {
                
                //Count the agents by type but only for the status bus
                if ( tag.bus == Bus.STATUS ) {
                    AgentType type = tag.agentInfo.getType();
                    if ( !alreadyCountedAgentName.contains(tag.agentInfo.getName()) ) {
                        alreadyCountedAgentName.add(tag.agentInfo.getName());                        
                        Integer currentCount = typeCount.getOrDefault(type.name(), 0);
                        currentCount++;
                        typeCount.put(type.name(), currentCount);
                    }
                }
                
                Builder pointBuilder = Point.measurement("cluster_metrics")
                        .time(time, TimeUnit.MILLISECONDS);
                Integer counter = agentCountersMapOld.get(tag);
                if (counter != null) {
                    pointBuilder = pointBuilder.addField("traffic", counter);
                }

                Map<String,Accumulate> accumMap = agentTimeAccumulatorsMapOld.get(tag);                
                for (String key : accumMap.keySet()) {
                    Accumulate accum = accumMap.get(key);

                    if (accum.getCounts() > 0) {
                        pointBuilder = pointBuilder.addField(key + "_avg", (double) accum.getAverageValue()).
                                addField(key + "_max", (double) accum.getMaxValue()).
                                addField(key + "_min", (double) accum.getMinValue()).addField(key + "_tot", (double) accum.getTotalValue());

                    }
                }
                
                if (pointBuilder.hasFields()) {
                    AgentType agentType = tag.agentInfo.getType();
                    String agentName = /*agentType == AgentType.CONSOLE ? "console" :*/ tag.agentInfo.getName();
                    Point point = pointBuilder.tag("bus", tag.bus.name()).tag("agent", agentName).
                            tag("host", tag.agentInfo.getAgentProperty("org.lsst.ccs.agent.hostname", "")).
                            tag("user", tag.agentInfo.getAgentProperty("org.lsst.ccs.agent.username", "")).
                            tag("msgType", tag.messageType).
                            tag("type", agentType.name()).tag(influxDbClientService.getGlobalTags()).build();                    
                    LOG_INFLUX.fine("Writing to influxDb " + point);
                    bp.point(point);
                }
            }
            
            for ( Entry<String,Integer> entry : typeCount.entrySet()  ) {
                Builder pointBuilder = Point.measurement("cluster_metrics")
                        .time(time, TimeUnit.MILLISECONDS);
                pointBuilder = pointBuilder.addField("agent_count", entry.getValue());
                if (pointBuilder.hasFields()) {
                    Point point = pointBuilder.tag("type", entry.getKey()).tag(influxDbClientService.getGlobalTags()).build();
                    bp.point(point);
                }
            }

            if ( !bp.getPoints().isEmpty() ) {
                influxDbClientService.write(bp);
            }

        } 

    }

    private static class Tags {

        private final Bus bus;
        private final AgentInfo agentInfo;
        private final String messageType;

        Tags(Bus bus, AgentInfo agentInfo, BusMessage msg) {
            this.bus = bus;
            this.agentInfo = agentInfo;
            String msgSimpleName = msg.getClass().getSimpleName();
            if ( msg instanceof StatusMessage ) {
                messageType = msgSimpleName;
            } else if ( msg instanceof CommandMessage ) {
                if ( msg instanceof CommandRequest ) {
                    messageType = msgSimpleName+"-"+((CommandRequest)msg).getBasicCommand().getCommand();
                } else {
                    messageType = msgSimpleName;
                }
            } else if ( msg instanceof LogMessage ) {
                messageType = msgSimpleName+"-"+((LogMessage)msg).getLoggerName()+"-"+((LogMessage)msg).getLevel();
            } else {
                throw new RuntimeException("Unknown message "+msg);
            }
        }

        @Override
        public boolean equals(Object obj) {
            if (obj instanceof Tags) {
                Tags t1 = (Tags) obj;
                return t1.bus.equals(bus) && t1.agentInfo.equals(agentInfo) && t1.messageType.equals(messageType);
            }
            return false;
        }

        @Override
        public int hashCode() {
            return bus.hashCode() + 34 * agentInfo.hashCode() + 123*messageType.hashCode();
        }

    }

    private static Map<Tags, Integer> agentCountersMap = new ConcurrentHashMap<>();
    private static Map<Tags, Map<String,Accumulate>> tagsAccumMap = new ConcurrentHashMap<>();

    private static void incrementAgentCounter(Tags tags) {
        Integer agentCounter = agentCountersMap.get(tags);
        if (agentCounter == null) {
            agentCounter = 0;
        }
        agentCountersMap.put(tags, ++agentCounter);
    }

    private static void incrementAgentKeyAccumulator(Tags tags, String key, double value) {
        Map<String,Accumulate> keyAccumMap = tagsAccumMap.get(tags);
        if (keyAccumMap == null) {
            keyAccumMap = new HashMap<>();
            tagsAccumMap.put(tags, keyAccumMap);
        }
        Accumulate accum = keyAccumMap.get(key);
        if ( accum == null ) {
            accum = new Accumulate();
            keyAccumMap.put(key,accum);
        }
        accum.accumulate(value);
    }


    private static class CommandBusCounterListener implements BusMessagePreProcessor {

        @Override
        public BusMessage preProcessMessage(BusMessage msg) {
//            commandQueue.add(() -> processBusMessage(Bus.COMMAND,msg, CCSTimeStamp.currentTime()));
            processBusMessage(Bus.COMMAND,msg, CCSTimeStamp.currentTime());
            return msg;
        }

        @Override
        public Bus getBus() {
            return Bus.COMMAND;
        }
        
    }
    
    private static void processBusMessage(Bus bus, BusMessage msg, CCSTimeStamp received) {
        AgentInfo agentInfo = msg.getOriginAgentInfo();
        Tags tags = new Tags(bus, agentInfo, msg);
        MsgInfo msgInfo = new MsgInfo(msg, received);


        synchronized (counterLock) {
            incrementAgentCounter(tags);
            incrementAgentKeyAccumulator(tags, "process_out", msgInfo.getOutboundProcessingTime());
            incrementAgentKeyAccumulator(tags, "creation_to_queue_out", msgInfo.getCreationToQueueTime());
            incrementAgentKeyAccumulator(tags, "queue_out", msgInfo.getOutgoingQueueTime());
            incrementAgentKeyAccumulator(tags, "queue_out_to_serialization", msgInfo.getQueueToSerializationTime());
            incrementAgentKeyAccumulator(tags, "serialization", msgInfo.getSerializationTime());
            incrementAgentKeyAccumulator(tags, "transfer", msgInfo.getNetworkTransferTime());
            incrementAgentKeyAccumulator(tags, "deserialization", msgInfo.getDeserializationTime());
            incrementAgentKeyAccumulator(tags, "deserialization_to_queue_in", msgInfo.getDeserializationToQueueTime());
            incrementAgentKeyAccumulator(tags, "queue_in", msgInfo.getIncomingQueueTime());
//            incrementAgentKeyAccumulator(tags, "queue_in_to_receive", msgInfo.getQueueToReceiveTime());
            incrementAgentKeyAccumulator(tags, "process_in", msgInfo.getQueueToReceiveTime());
            incrementAgentKeyAccumulator(tags, "transfer_total", msgInfo.getTotalTransferTime());
            incrementAgentKeyAccumulator(tags, "size", msgInfo.getSize());
        }

        if ( msg instanceof StatusMessage ) {
            if ( ! (msg instanceof StatusHeartBeat) ) {
                Level logLevel = msgInfo.getTotalTransferTime() > 50 ? Level.INFO : Level.FINE;
                LOG_MSG.log(logLevel, "{0}\n{1}", new Object[]{getMessageInfoStr(msgInfo),getMessageTimingInformation(msgInfo)});                                
                LOG_MSG.log(Level.FINER, "{0}", msg);
                if ( msg instanceof StatusSubsystemData ) {
                    StatusSubsystemData ssd = (StatusSubsystemData) msg;
                    if ( ImageHeaderKeywords.IMAGE_HEADER_KEYWORDS.equals(ssd.getDataKey()) ) {
                        ImageHeaderKeywords keywords = (ImageHeaderKeywords)ssd.getObject().getValue();
                        Map<String, Serializable> headerMap = keywords.getKeywords();
                        if (headerMap.containsKey("darkTime")) {
                            StringBuilder sb = new StringBuilder();
                            sb.append("DarkTime msg : \n");
                            sb.append(getMessageTimingInformation(msgInfo));
                            LOG_MSG.log(Level.INFO, "{0}", sb.toString());
                        } 
                    }
                }
            }
        } else if ( msg instanceof CommandMessage ) {
            CommandMessage cmdMsg = (CommandMessage)msg;
            UUID corrId = cmdMsg.getCorrelationId();
            if ( cmdMsg instanceof CommandRequest ) {
                commands.put(corrId, new CommandMsg(msgInfo));
            } else if ( msg instanceof CommandAck ) {
                CommandMsg existingMsg = commands.get(corrId);
                if ( existingMsg != null ) {
                    existingMsg.setAck(msgInfo);
                }
            } else if ( msg instanceof CommandNack || msg instanceof CommandResult ) {
                CommandMsg existingMsg = commands.remove(corrId);
                if ( existingMsg != null ) {
                    existingMsg.logCommandMsg(msgInfo);
                }
            }            
        }
    }
    

    private static class LogBusCounterListener implements BusMessagePreProcessor {

        @Override
        public Bus getBus() {
            return Bus.LOG;
        }

        @Override
        public BusMessage preProcessMessage(BusMessage msg) {
//            logQueue.add(() -> processBusMessage(Bus.LOG,msg, CCSTimeStamp.currentTime()));
            processBusMessage(Bus.LOG,msg, CCSTimeStamp.currentTime());
            return msg;
        }

    }

    private static class StatusBusCounterListener implements BusMessagePreProcessor {

        @Override
        public Bus getBus() {
            return Bus.STATUS;
        }

        @Override
        public BusMessage preProcessMessage(BusMessage msg) {
//            statusQueue.add(() -> processBusMessage(Bus.STATUS,msg, CCSTimeStamp.currentTime()));
            processBusMessage(Bus.STATUS,msg, CCSTimeStamp.currentTime());
            return msg;
        }

    }

    public static double sizeOf(Serializable object) {
        ByteArrayOutputStream byteOutputStream = new ByteArrayOutputStream();
        try (ObjectOutputStream objectOutputStream = new ObjectOutputStream(byteOutputStream)) {
            objectOutputStream.writeObject(object);
            objectOutputStream.flush();
        } catch (IOException ieo) {
            return 0;
        }

        return (double)byteOutputStream.toByteArray().length / 1000.;
    }

    //Executes the elements in the queue until a null Runnable is added.    
    private class MyQueueWorker extends Thread {

        private final BlockingQueue<Runnable> queue;

        public MyQueueWorker(BlockingQueue<Runnable> queue) {
            this.queue = queue;
        }

        public void run() {
            try {
                while (true) {
                    Runnable r = queue.take();
                    if (r instanceof DoneWork) {
                        break;
                    }
                    r.run();
                }
            } catch (InterruptedException ie) {
                // just terminate
            }
        }

    }

    private class DoneWork implements Runnable {

        @Override
        public void run() {
            throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
        }

    }

    private static class MsgInfo {
        
        private final BusMessage msg;
        private final long totalTransferTime, networkTransferTime;
        private final long outboundProcessingTime, inboundProcessingTime;
        private final long serializationTime, deserializationTime;
        private final long outgoingQueueTime, incomingQueueTime;
        private final long creationToQueueTime, deserializationToQueueTime;
        private final long queueToSerializationTime , queueToReceiveTime;
        private final CCSTimeStamp received;
        private final double size;
        
        MsgInfo(BusMessage msg, CCSTimeStamp received) {
            this.msg = msg;
            this.received = received;
            
            totalTransferTime = Duration.between(msg.getCCSTimeStamp().getUTCInstant(),received.getUTCInstant()).toMillis();            
            networkTransferTime = msg.getTransferDuration() != null ? msg.getTransferDuration().toMillis() : 0;

            outboundProcessingTime = msg.getSerializationTime() != null ? Duration.between(msg.getCCSTimeStamp().getUTCInstant(), msg.getSerializationTime().getUTCInstant()).toMillis() : 0;
            if ( msg.getDoneDeSerializationTime() != null ) {           
                inboundProcessingTime = Duration.between(msg.getDoneDeSerializationTime().getUTCInstant(),received.getUTCInstant()).toMillis();
            } else {
                inboundProcessingTime = -1;                
            }

            if ( msg.getSerializationDuration()!= null ) {           
                serializationTime = msg.getSerializationDuration().toMillis();
            } else {
                serializationTime = -1;
            }
            
            if ( msg.getDeserializationDuration()!= null ) {           
                deserializationTime = msg.getDeserializationDuration().toMillis();
            } else {
                deserializationTime = -1;
            }
            
            outgoingQueueTime = (msg.getOutgoingQueueInTimeStamp() != null && msg.getOutgoingQueueOutTimeStamp() != null) ?
                    Duration.between(msg.getOutgoingQueueInTimeStamp().getUTCInstant(), msg.getOutgoingQueueOutTimeStamp().getUTCInstant()).toMillis() : 0L;
            
            incomingQueueTime = (msg.getIncomingQueueInTimeStamp() != null && msg.getIncomingQueueOutTimeStamp() != null) ?
                    Duration.between(msg.getIncomingQueueInTimeStamp().getUTCInstant(), msg.getIncomingQueueOutTimeStamp().getUTCInstant()).toMillis() : 0L;

            
        
            creationToQueueTime = Duration.between(msg.getCCSTimeStamp().getUTCInstant(),msg.getOutgoingQueueInTimeStamp().getUTCInstant()).toMillis();            
            queueToSerializationTime = Duration.between(msg.getOutgoingQueueOutTimeStamp().getUTCInstant(),msg.getSerializationTime().getUTCInstant()).toMillis();            

            deserializationToQueueTime = Duration.between(msg.getDoneDeSerializationTime().getUTCInstant(),msg.getIncomingQueueInTimeStamp().getUTCInstant()).toMillis();            
            queueToReceiveTime = Duration.between(msg.getIncomingQueueOutTimeStamp().getUTCInstant(),received.getUTCInstant()).toMillis();            
            
            size = sizeOf(msg);                   
        }

        public long getTotalTransferTime() {
            return totalTransferTime;
        }

        public long getNetworkTransferTime() {
            return networkTransferTime;
        }

        public long getOutboundProcessingTime() {
            return outboundProcessingTime;
        }

        public long getInboundProcessingTime() {
            return inboundProcessingTime;
        }

        public long getSerializationTime() {
            return serializationTime;
        }

        public long getDeserializationTime() {
            return deserializationTime;
        }

        public double getSize() {
            return size;
        }                

        public long getOutgoingQueueTime() {
            return outgoingQueueTime;
        }
        
        public long getIncomingQueueTime() {
            return incomingQueueTime;
        }

        public CCSTimeStamp getReceivedTimestamp() {
            return received;
        }
        
        public long getCreationToQueueTime() {
            return creationToQueueTime;
        }
        
        public long getQueueToSerializationTime() {
            return queueToSerializationTime;
        }
        
        public long getDeserializationToQueueTime() {
            return deserializationToQueueTime;
        }

        public long getQueueToReceiveTime() {
            return queueToReceiveTime;
        }
        
    }
    
    private static class CommandMsg {
        private final MsgInfo msgInfo;
        private MsgInfo ack;
        
        CommandMsg(MsgInfo msgInfo) {
            this.msgInfo = msgInfo;
        }
        
        void setAck(MsgInfo ack) {
            this.ack = ack;
        }
        
        void logCommandMsg(MsgInfo finalReply) {
            long totalCmdExecution = Duration.between(msgInfo.msg.getCCSTimeStamp().getUTCInstant(), finalReply.received.getUTCInstant()).toMillis();
            Level logLevel = totalCmdExecution > 50 ? Level.INFO : Level.FINE;            
            if ( ack != null ) {
                LOG_MSG.log(logLevel, "{0}\n{1}\n{2}\n{3}", new Object[]{getMessageInfoStr(msgInfo),getMessageInfoStr(ack),getMessageInfoStr(finalReply),"Total Command Execution: "+totalCmdExecution+" ms"});                
            } else {
                LOG_MSG.log(logLevel, "{0}\n{1}\n{2}", new Object[]{getMessageInfoStr(msgInfo),getMessageInfoStr(finalReply),"Total Command Execution: "+totalCmdExecution+" ms"});                                
            }
        }        
    }
    
    private static String getMessageInfoStr(MsgInfo msgInfo) {
        BusMessage msg = msgInfo.msg;
        if (msg instanceof CommandRequest) {
            CommandRequest request = (CommandRequest) msg;
            return String.format("Command Request to "+request.getDestination()+" from "+request.getOriginAgentInfo().getName()+": "+request.getBasicCommand().getCommand()+" ("+msgInfo.getTotalTransferTime()+" ms, "+msgInfo.getSize()+" Kb)");
        } else if (msg instanceof CommandReply) {
            String header = "unknown";
            if (msg instanceof CommandAck) {
                header = "Command Ack";
            } else if (msg instanceof CommandNack) {
                header = "Command Nack";
            } else if (msg instanceof CommandResult) {
                header = "Command Result";
            }
            return String.format(header+": ("+msgInfo.getTotalTransferTime()+" ms, "+msgInfo.getSize()+" Kb)");
        } else if (msg instanceof StatusMessage) {
            return String.format("Status "+msg.getClass().getSimpleName()+" from "+msg.getOriginAgentInfo().getName()+": ("+msgInfo.getTotalTransferTime()+" ms, "+msgInfo.getSize()+" Kb)");
        }
        return "";
    }
    
    private static String getMessageTimingInformation(MsgInfo msgInfo) {
        StringBuilder sb = new StringBuilder("Message timing (ms)\n\tTotal transfer time ");
        sb.append(msgInfo.getTotalTransferTime()).append("\n\tOutgoing Processing ");
        sb.append(msgInfo.getOutboundProcessingTime()).append("\n\t\tCreation to queue ");
        sb.append(msgInfo.getCreationToQueueTime()).append("\n\t\tOutgoing queue ");
        sb.append(msgInfo.getOutgoingQueueTime()).append("\n\t\tOutgoing queue to serialization ");
        sb.append(msgInfo.getQueueToSerializationTime()).append("\n\t\tSerialization ");
        sb.append(msgInfo.getSerializationTime()).append("\n\tNetwork Transfer ");
        sb.append(msgInfo.getNetworkTransferTime()).append("\n\tIncoming Processing ");
        sb.append(msgInfo.getInboundProcessingTime()).append("\n\t\tDeserialization ");
        sb.append(msgInfo.getDeserializationTime()).append("\n\t\tDeserialization to incoming queue ");
        sb.append(msgInfo.getDeserializationToQueueTime()).append("\n\t\tIncoming queue ");
        sb.append(msgInfo.getIncomingQueueTime()).append("\n\t\tIncoming queue to processing ");
        sb.append(msgInfo.getQueueToReceiveTime()).append("\n");
        return sb.toString();
    }

}
