package org.lsst.ccs.subsystem.cluster.monitor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.log4j.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.AgentAlerts;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentMBeanService;
import org.lsst.ccs.services.AgentMonitor;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.cluster.monitor.CustomClusterProbe.ClusterInfo;
import org.lsst.ccs.subsystem.cluster.monitor.CustomClusterProbe.ClusterMemberInfo;

/**
 * A Monitor for the JGroups View.
 * It raises an Alert when view splits.
 * 
 * @author The LSST CCS Team
 */
public class JGroupsViewMonitor implements HasLifecycle, AgentMonitor, ClearAlertHandler {
    
    @LookupField(strategy = LookupField.Strategy.TOP)
    Agent agent;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentMBeanService mbeanService;
    
    private final Logger log = Logger.getLogger("org.lsst.ccs.jgroup.monitor");
    
    private String diagnosticAddress = null;

    @Override
    public void init() {
        //Not a particularly elegant way of fetching this property.
        //A better way would be to fetch it from the JGroups stack
        diagnosticAddress = BootstrapResourceUtils.getBootstrapProperties("udp_ccs").getProperty("org.lsst.ccs.jgroups.ALL.UDP.diagnostics_addr", "224.0.75.75");
    }
    
    
    
    @Override
    public String getAgentMonitorDescription() {
        return "JGroups Cluster View Monitor";
    }

    @Override
    public synchronized String getAgentMonitorStatus(boolean useCcsBuses) {
        CustomClusterProbe p = new CustomClusterProbe(diagnosticAddress);        
        p.probeCluster(new String[0]);
        
        Collection<ClusterInfo> clusters = p.getClusters();
        
        StringBuilder probeAlertBuffer = new StringBuilder();
        //First check that all the clusters have the same number of entries.
        //WE COMMENT THIS FEATURE OUT SINCE WE ARE GETTING EXCEPTIONS ON CLIENTS LIKE:
        /*
        
SEVERE: 2018-01-30 08:38:26,292 - failed sending diag rsp to /134.79.209.38:43070

java.io.IOException: Operation not permitted (sendto failed)
at java.net.PlainDatagramSocketImpl.send(Native Method)
at java.net.DatagramSocket.send(DatagramSocket.java:693)
at org.jgroups.stack.DiagnosticsHandler.sendResponse(DiagnosticsHandler.java:207)
at org.jgroups.stack.DiagnosticsHandler.handleDiagnosticProbe(DiagnosticsHandler.java:169)
at org.jgroups.stack.DiagnosticsHandler.run(DiagnosticsHandler.java:126)
at java.lang.Thread.run(Thread.java:745)
        
        */
        //This exception seems to indicate that there are threading issues on
        //the client side. They might be solved/addressed in newer JGroups versions.
        //Because of these exceptions the number of member in each JGroups channel
        //is unreliable.
        /*
        boolean clustersSizeCheckOk = true;        
        if ( clusters.size() > 1 ) {
            log.debug("Checking cluster compatibility for "+clusters.size()+" clusters");
            StringBuilder sb = new StringBuilder();
            int clusterSize = -1;
            for ( ClusterInfo cluster: clusters ) {
                if ( clusterSize < 0 ) {
                    clusterSize = cluster.size();
                } else {
                    if ( clusterSize != cluster.size() ) {
                        clustersSizeCheckOk = false;                        
                    }
                }
                sb.append(cluster.clusterName).append(" has ").append(cluster.size()).append(" members\n");                
            }
            if ( ! clustersSizeCheckOk ) {
                sb = new StringBuilder("Cluster Size mismatch!!\n").append(sb);
                log.warn(sb);                
                probeAlertBuffer.append(sb);                
            } else {
                log.debug(sb.toString());            
            }
        }
        */
        //Now check that all members within a cluster have the same view.
        boolean clusterViewOk = true;
        for ( ClusterInfo cluster: clusters ) {
            log.debug("Checking view for "+cluster.clusterName);
            //Build a map of all the views in the cluster. If things are ok
            //there should be only one view in the cluster.
            HashMap<String, List<ClusterMemberInfo>> viewsInCluster = new HashMap<>();
            StringBuilder sb = new StringBuilder();
            for ( CustomClusterProbe.ClusterMemberInfo member : cluster ) {
                //Build a unique view identifier for each member: viewName|viewId (viewSize);            
                String clusterUniqueId = member.viewName+"|"+member.viewId+" ("+member.viewSize+")";
                if ( !viewsInCluster.containsKey(clusterUniqueId) ) {
                    viewsInCluster.put(clusterUniqueId, new ArrayList<>());
                }
                viewsInCluster.get(clusterUniqueId).add(member);                
            }

            for ( Map.Entry<String,List<ClusterMemberInfo>> entry : viewsInCluster.entrySet() ) {
                sb.append("View: ").append(entry.getKey()).append(" [");
                for ( ClusterMemberInfo member : entry.getValue() ) {
                    sb.append(member.getMemberName()).append(" ");
                }
                sb.append("]\n");
            }
            
            //Raise a problem if there is more than one view
            if ( viewsInCluster.size() > 1 ) {
                clusterViewOk = false;
                sb = new StringBuilder("Too many views in Cluster: ").append(cluster.clusterName).append("\n").append(sb);
                log.warn(sb);                
                probeAlertBuffer.append(sb);                
            } else {
                log.debug(sb);
            }
        }
        

        if ( useCcsBuses ) {
            if ( (!clusterViewOk ) ) {                    
                agent.getAgentService(AlertService.class).raiseAlert(AgentAlerts.ClusterViewAlert.getAlert(null), AlertState.ALARM, probeAlertBuffer.toString());
            } else {
                //Check if there is any outstanding alert
                RaisedAlertHistory history = agent.getAgentService(AlertService.class).getRaisedAlertSummary().getRaisedAlert(AgentAlerts.ClusterViewAlert.getAlertId());
                if ( history != null && history.getLatestAlertInstance().getAlertState() != AlertState.NOMINAL ) {                
                    agent.getAgentService(AlertService.class).raiseAlert(AgentAlerts.ClusterViewAlert.getAlert(null), AlertState.NOMINAL, "Cluster is back to normal.");
                }
            }
        }
        
        return probeAlertBuffer.toString().isEmpty() ? "OK" : probeAlertBuffer.toString();
    }
        
    
    @Override
    public ClearAlertCode canClearAlert(Alert alert, AlertState state) {
        String alertId = alert.getAlertId();
        if ( alertId.equals(AgentAlerts.ClusterViewAlert.getAlertId()) ) {
            return getAgentMonitorStatus(false).equals("OK") ? ClearAlertCode.CLEAR_ALERT : ClearAlertCode.DONT_CLEAR_ALERT;
        } 
        return ClearAlertCode.UNKNOWN_ALERT;
    }
    
    public static void main(String[] argv) {
                
        JGroupsViewMonitor monitor = new JGroupsViewMonitor();
        String result = monitor.getAgentMonitorStatus(false);
        System.out.println("\nResult of probing the cluster: ");
        System.out.println(result);

    }
    
    
}
