package org.lsst.ccs.subsystem.cluster.monitor;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandMessage;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusClearedAlert;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusDataProviderDictionary;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusRaisedAlert;
import org.lsst.ccs.bus.messages.StatusRaisedAlertSummary;
import org.lsst.ccs.bus.messages.StatusRuntimeInfo;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.localdb.statusdb.server.DataChannel;
import org.lsst.ccs.localdb.statusdb.server.DataChannel.DataChannelList;
import org.lsst.ccs.localdb.utils.ClientRequestUtils;
import org.lsst.ccs.messaging.CommandMessageListener;
import org.lsst.ccs.messaging.LogMessageListener;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.subsystem.cluster.monitor.BusTrafficMonitor.DataType;

/**
 * A Subsystem that tracks the bus traffic.
 *
 * @author The LSST CCS Team
 */
public class BusTrafficMonitor implements HasLifecycle {

    private static final Object counterLock = new Object();
    private static Counter counter;
    private static Accumulate sizeAccumulator;
    private static Accumulate timeAccumulator;

    private static ClientRequestUtils restUtils;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;

    public static String restServerName = "";

    public static int restServerPort = 8080;

    private static final List<String> recentSubsystems = new ArrayList<>();
    private static final List<String> STATUS_MESSAGES = new ArrayList<>();
    private static final List<String> COMMAND_MESSAGES = new ArrayList<>();
    private static final List<String> LOG_MESSAGES = new ArrayList<>();

    private static Map<Bus, List<String>> BUS_MAP = new HashMap<>();
    private static Map<DataType, String> DATA_UNITS = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final List<BusTrafficDevice> devices = new ArrayList<>();

    public enum DataType {
        TRAFFIC,
        SIZE,
        TIME;
    }

    static {
        STATUS_MESSAGES.add(StatusHeartBeat.class.getSimpleName());
        STATUS_MESSAGES.add(StatusStateChangeNotification.class.getSimpleName());
        STATUS_MESSAGES.add(StatusRuntimeInfo.class.getSimpleName());
        STATUS_MESSAGES.add(StatusDataProviderDictionary.class.getSimpleName());
        STATUS_MESSAGES.add(StatusRaisedAlert.class.getSimpleName());
        STATUS_MESSAGES.add(StatusClearedAlert.class.getSimpleName());
        STATUS_MESSAGES.add(StatusSubsystemData.class.getSimpleName());
        STATUS_MESSAGES.add(StatusRaisedAlertSummary.class.getSimpleName());
        STATUS_MESSAGES.add(StatusConfigurationInfo.class.getSimpleName());
        STATUS_MESSAGES.add("all");
        STATUS_MESSAGES.add("other");

        COMMAND_MESSAGES.add(CommandResult.class.getSimpleName());
        COMMAND_MESSAGES.add(CommandRequest.class.getSimpleName());
        COMMAND_MESSAGES.add(CommandNack.class.getSimpleName());
        COMMAND_MESSAGES.add(CommandAck.class.getSimpleName());
        COMMAND_MESSAGES.add("all");

        LOG_MESSAGES.add("all");

        BUS_MAP.put(Bus.STATUS, STATUS_MESSAGES);
        BUS_MAP.put(Bus.COMMAND, COMMAND_MESSAGES);
        BUS_MAP.put(Bus.LOG, LOG_MESSAGES);

        DATA_UNITS.put(DataType.TRAFFIC, "#/min");
        DATA_UNITS.put(DataType.SIZE, "Kb");
        DATA_UNITS.put(DataType.TIME, "ms");
                
    }
    
    public static double getHighLimitFor(DataType type, String dataName) {
        switch (type) {
            case TRAFFIC:
                if ( dataName.contains("/all") ) {
                    return recentSubsystems.size()*700;
                } else {
                    return 700;
                }
            case TIME:
                for (Bus b : Bus.values() ) {
                    if ( dataName.endsWith(b.name().toLowerCase()+"/all") ) {
                        return 4;
                    } 
                }
                return 70;
            case SIZE:
                for (Bus b : Bus.values() ) {
                    if ( dataName.endsWith(b.name().toLowerCase()+"/all") ) {
                        return 4;
                    } 
                }
        }
        return 0;
    }

    public static double getDeadBandFor(DataType type, String dataName) {
        switch (type) {
            case TRAFFIC:
                if ( dataName.contains("/all") ) {
                    return recentSubsystems.size()*100;
                } else {
                    return 200;
                }
            case TIME:
                return 20;
            case SIZE:
                return 100;
        }
        return 0;
    }

    public static boolean getCheckHi(DataType type, String dataName) {
        switch (type) {
            case TRAFFIC:
                return true;
            case TIME:
                for (Bus b : Bus.values() ) {
                    if ( dataName.endsWith(b.name().toLowerCase()+"/all") ) {
                        return true;
                    } 
                }
                return false;
            case SIZE:
                for (Bus b : Bus.values() ) {
                    if ( dataName.endsWith(b.name().toLowerCase()+"/all") ) {
                        return true;
                    } 
                }
                return false;
        }
        return false;
    }
    
    
    public static List<String> getSubsystemsList() {
        if (recentSubsystems.isEmpty()) {
            restUtils = new ClientRequestUtils(restServerName, restServerPort);

            recentSubsystems.add("");
            if (restServerName.isEmpty()) {
                recentSubsystems.add("demo");
                recentSubsystems.add("demo1");
                recentSubsystems.add("client");
                recentSubsystems.add("other");
            } else {
                DataChannelList list = restUtils.getChannelList();
                for (DataChannel dc : list.list) {
                    String subsystemName = dc.getPath()[0];
                    if (!recentSubsystems.contains(subsystemName)) {
                        recentSubsystems.add(subsystemName);
                    }
                }

                recentSubsystems.add("client");
                recentSubsystems.add("other");
            }
        }
        return recentSubsystems;
    }

    @Override
    public void build() {

        agent.setAgentProperty("org.lsst.ccs.use.full.paths", "true");

        AgentPeriodicTask resetCounterAndPublish = new AgentPeriodicTask("busTrafficStatsUpdate",
                () -> {
                    resetCounterAndPublish();
                }).withPeriod(Duration.ofSeconds(60));

        agent.getAgentPeriodicTaskService().scheduleAgentPeriodicTask(resetCounterAndPublish);

    }
    
    private void resetCounters() {
        synchronized(counterLock) {
            for (BusTrafficDevice d : devices) {
                switch (d.getType()) {
                    case TRAFFIC:
                        counter = new Counter(d.getDataList());
                        break;
                    case TIME:
                        timeAccumulator = new Accumulate(d.getDataList());
                        break;
                    case SIZE:
                        sizeAccumulator = new Accumulate(d.getDataList());
                        break;
                }
            }
        }
    }

    @Override
    public void start() {
        resetCounters();        
        agent.getMessagingAccess().addCommandMessageListener(new CommandBusCounterListener());
        agent.getMessagingAccess().addStatusMessageListener(new StatusBusCounterListener());
        agent.getMessagingAccess().addLogMessageListener(new LogBusCounterListener());
    }

    private void resetCounterAndPublish() {
        Counter oldCounter;
        Accumulate oldSizeAccumulator;
        Accumulate oldTimeAccumulator;
        synchronized (counterLock) {
            oldCounter = counter;
            oldSizeAccumulator = sizeAccumulator;
            oldTimeAccumulator = timeAccumulator;
            resetCounters();
        }

        for ( BusTrafficDevice device : devices ) {
            Map dataMap = device.getType() == DataType.TRAFFIC ? oldCounter.getCountersMap() : device.getType() == DataType.SIZE ? oldSizeAccumulator.getAverageMap() : oldTimeAccumulator.getAverageMap();
            device.setDataMap(dataMap);
        }
//        KeyValueData counterData = new KeyValueData("traffic", (Serializable) oldCounter.getCountersMap());
//        agent.publishSubsystemDataOnStatusBus(counterData);
//        KeyValueData sizeData = new KeyValueData("size", (Serializable) oldSizeAccumulator.getAverageMap());
//        agent.publishSubsystemDataOnStatusBus(sizeData);
//        KeyValueData timeData = new KeyValueData("time", (Serializable) oldTimeAccumulator.getAverageMap());
//        agent.publishSubsystemDataOnStatusBus(timeData);
    }

    private static class CommandBusCounterListener implements CommandMessageListener {

        String busName = Bus.COMMAND.name();

        @Override
        public void onCommandMessage(CommandMessage msg) {
            long deltaT = System.currentTimeMillis() - msg.getTimeStamp();
            String busKey = msg.getOriginAgentInfo().getType().ordinal() >= AgentInfo.AgentType.WORKER.ordinal() ? msg.getOriginAgentInfo().getName() : "client";
            if ( !getSubsystemsList().contains(busKey) ) {
                busKey = "other";
            }
            busKey += "/" + busName;
            String busMessageTypeKey = busName + "/" + msg.getClass().getSimpleName();
            String messageKey = busKey + "/" + msg.getClass().getSimpleName();
            double size = sizeOf(msg);
            synchronized (counterLock) {
                counter.increment("traffic/" + busName + "/all");
                counter.increment("traffic/" + busMessageTypeKey);
                counter.increment("traffic/" + busKey + "/all");
                counter.increment("traffic/" + messageKey);
                sizeAccumulator.accumulate(size, "size/" + busName + "/all");
                sizeAccumulator.accumulate(size, "size/" + busMessageTypeKey);
                sizeAccumulator.accumulate(size, "size/" + busKey + "/all");
                sizeAccumulator.accumulate(size, "size/" + messageKey);
                timeAccumulator.accumulate(deltaT, "time/" + busName + "/all");
                timeAccumulator.accumulate(deltaT, "time/" + busMessageTypeKey);
                timeAccumulator.accumulate(deltaT, "time/" + busKey + "/all");
                timeAccumulator.accumulate(deltaT, "time/" + messageKey);
            }
        }

    }

    private static class LogBusCounterListener implements LogMessageListener {

        String busName = Bus.LOG.name();

        @Override
        public void onLogMessage(LogMessage msg) {
            long deltaT = System.currentTimeMillis() - msg.getTimeStamp();
            String busKey = msg.getOriginAgentInfo().getType().ordinal() >= AgentInfo.AgentType.WORKER.ordinal() ? msg.getOriginAgentInfo().getName() : "client";
            if ( !getSubsystemsList().contains(busKey) ) {
                busKey = "other";
            }
            busKey += "/" + busName;
            double size = sizeOf(msg);
//            String loggerName = msg.getLoggerName();
            synchronized (counterLock) {
                counter.increment("traffic/" + busName + "/all");
                counter.increment("traffic/" + busKey+"/all");
                sizeAccumulator.accumulate(size, "size/" + busName + "/all");
                sizeAccumulator.accumulate(size, "size/" + busKey+"/all");
                timeAccumulator.accumulate(deltaT, "time/" + busName + "/all");
                timeAccumulator.accumulate(deltaT, "time/" + busKey+ "/all");
//                String[] tmpLoggerNames = loggerName.split(".");
//                String agentTmpLoggerName = busKey;
//                String logBusTmpLoggerName = busKey+"/";
//                for ( String token : tmpLoggerNames ) {
//                    agentTmpLoggerName += token;
//                    logBusTmpLoggerName += token;
//                    counter.increment(agentTmpLoggerName);
//                    sizeAccumulator.accumulate(size, agentTmpLoggerName);
//                    timeAccumulator.accumulate(deltaT, agentTmpLoggerName);
//                    counter.increment(logBusTmpLoggerName);
//                    sizeAccumulator.accumulate(size, logBusTmpLoggerName);
//                    timeAccumulator.accumulate(deltaT, logBusTmpLoggerName);
//                    agentTmpLoggerName += "/";
//                    logBusTmpLoggerName += "/";
//                }
//                
            }
        }
    }

    private static class StatusBusCounterListener implements StatusMessageListener {

        String busName = Bus.STATUS.name();

        @Override
        public void onStatusMessage(StatusMessage msg) {
            long deltaT = System.currentTimeMillis() - msg.getTimeStamp();
            String busKey = msg.getOriginAgentInfo().getType().ordinal() >= AgentInfo.AgentType.WORKER.ordinal() ? msg.getOriginAgentInfo().getName() : "client";
            if ( !getSubsystemsList().contains(busKey) ) {
                busKey = "other";
            }
            busKey += "/" + busName;
            
            String classKey = msg.getClass().getSimpleName();
            if ( !BUS_MAP.get(Bus.STATUS).contains(classKey) ) {
                classKey = "other";
            }
            
            String messageKey = busKey + "/" + classKey;
            String busMessageTypeKey = busName + "/" + classKey;
            double size = sizeOf(msg);
            synchronized (counterLock) {
                counter.increment("traffic/" + busName + "/all");
                counter.increment("traffic/" + busKey + "/all");
                counter.increment("traffic/" + busMessageTypeKey);
                counter.increment("traffic/" + messageKey);
                sizeAccumulator.accumulate(size, "size/" + busName + "/all");
                sizeAccumulator.accumulate(size, "size/" + busMessageTypeKey);
                sizeAccumulator.accumulate(size, "size/" + busKey + "/all");
                sizeAccumulator.accumulate(size, "size/" + messageKey);
                timeAccumulator.accumulate(deltaT, "time/" + busName + "/all");
                timeAccumulator.accumulate(deltaT, "time/" + busMessageTypeKey);
                timeAccumulator.accumulate(deltaT, "time/" + busKey + "/all");
                timeAccumulator.accumulate(deltaT, "time/" + messageKey);
            }
        }

    }

    public static double sizeOf(Serializable object) {
        ByteArrayOutputStream byteOutputStream = new ByteArrayOutputStream();
        try (ObjectOutputStream objectOutputStream = new ObjectOutputStream(byteOutputStream)) {
            objectOutputStream.writeObject(object);
            objectOutputStream.flush();
        } catch (IOException ieo) {
            return 0;
        }

        return byteOutputStream.toByteArray().length/1000;
    }
}
