package org.lsst.ccs.subsystem.ccob.thin;

import java.nio.CharBuffer;
import java.util.ArrayList;
import static java.util.Collections.unmodifiableList;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Parses replies to commands such as {@code X?} which contain a line of key-value pairs where the key is
 * one or more space-separated alphanumeric words and the values are decimal numbers compatible with
 * Java doubles. A key and its value are separated by an equals sign or a colon. A value may be followed
 * by one of the unit specifiers "mm", "deg", "A" or "u" with or without separating whitespace. The units
 * are checked but discarded.
 * <p>
 * Example input: {@code X=1.5e-1, Y=0.0mm, U=0.0, B: 0.0 deg, P=0.0deg, foo   bar    =22.5}
 * In the last pair the key treated as "foo bar"; leading and trailing whitespace is pruned while
 * whitespace between words is reduced to s single space.
 * <p>
 * The result of a successful parse is a {@code Map<String, Double>}.

 * @author tether
 */
public class ValueParser {
    // The parser is implemented using recursive descent.
    // EBNF syntax, omitting the skipping of whitespace:
    // line = keyValuePair {"," keyValuePair}.
    // keyValuePair = WORD {WORD} ("="|":") NUMBER [UNITS] [COMMENTARY].
    
    // Regular expressions for token definitions. The capture group determines the token value.
    // WORD = ^\s*(\w+)
    // NUMBER = ^\s*([\d\+-\.eE]+)
    // UNITS = ^\s*(mm|deg|A|u)
    // COMMENTARY = ^([^,]+)
   
    // A CharBuffer is used to hold the input. Characters are said to be consumed when the
    // buffer's position is advanced past them. Parsing methods always look at the
    // input starting at the buffer's current position.
    
    // The pattern for numbers is actually much too forgiving but will accept an entire valid
    // NUMBER if one is present. Final checking is done by Double::valueOf.
    // We anchor all the patterns so that any matches must occur at the start.
    private static final Pattern WORD = Pattern.compile("^\\s*(\\w+)");
    private static final Pattern NUMBER = Pattern.compile("^\\s*([eE+.\\-\\d]+)");
    private static final Pattern COMMA = Pattern.compile("^\\s*,");
    private static final Pattern EQUALS = Pattern.compile("^\\s*(=|:)");
    private static final Pattern UNITS = Pattern.compile("^\\s*(mm|deg|A|u)");
    private static final Pattern COMMENTARY = Pattern.compile("^([^,]+)");
    private static final Pattern EOI = Pattern.compile("^\\s*($)");
    
    private static class Kvp {
        public final List<String> key;  // The list of words before the EQUALS sign.
        public final double value;      // The value of the NUMBER after the EQUALS sign, stripped of UNITS.
        public Kvp(final List<String> key, final double value) {
            this.key = unmodifiableList(new ArrayList<>(key));
            this.value = value;
        }
    }

    // Precondition: The input is a series of key-value pairs separated by commas with optional whitespace.
    // Postcondition: The entire series has been consumed and nothing else remains except possibly whitespace.
    /**
     * 
     * @param tbReply A reply from the TB Server, cut down to a single line containing key-value pairs as
     * described in the class Javadoc.
     * @return A map of keys to values.
     * @throws ParseError if the line can't be parsed.
     */
    public static Map<String, Double> parse(final String tbReply)
    {
        final Map<String, Double> result = new TreeMap<>();
        final CharBuffer input = CharBuffer.wrap(tbReply);
        Matcher m = WORD.matcher(input);
        if (!m.find()) {
            throw new ParseError("Expected at least one key-value pair.", input.toString());
        }
        boolean morePairs = true;
        while (morePairs) {
            // Consume a key-value pair.
            final Kvp pair = keyValuePair(input);
            final String key = String.join(" ", pair.key);
            result.put(key, pair.value);
            // Test for a COMMA, signifying more pairs to follow.
            m = COMMA.matcher(input);
            if (m.find()) {
                consume(m, input);
                morePairs = true;
            }
            else {
                morePairs = false;
            }
        }
        m = EOI.matcher(input);
        if (!m.find()) {
            throw new ParseError("Expected end of line or whitespace.", input.toString());
        }
        return result;
    }
    
    // Precondition: The input starts with a key-value pair.
    // Postcondition: The pair has been consumed.
    private static Kvp keyValuePair(final CharBuffer input) {
        final List<String> words = wordList(input);
        Matcher m = EQUALS.matcher(input);
        if (!m.find()) {
            throw new ParseError("Expected an equals sign.", input.toString());
        }
        consume(m, input);
        final double val = value(input);
        return new Kvp(words, val);
    }
    
    // Precondition: The input starts with a non-empty set of words separated by whitespace.
    // Postcondition: The words have been consumed.
    private static List<String> wordList(final CharBuffer input) {
        final List<String> result = new LinkedList<>();
        Matcher m = WORD.matcher(input);
        boolean wordFound = m.find();
        if (!wordFound) {
            throw new ParseError("Expected one or more words.", input.toString());
        }
        while (wordFound) {
            result.add(m.group(1));
            consume(m, input);
            m = WORD.matcher(input);
            wordFound = m.find();
        }
        return result;
    }
    
    // Precondition: The input starts with numberical value.
    // Postcondition: The value has been consumed along with any associated UNITS.
    private static double value(final CharBuffer input) {
        double result = 0.0;
        Matcher m = NUMBER.matcher(input);
        if (!m.find()) {
            throw new ParseError("Expected a decimal number.", input.toString());
        }
        try {
            result = Double.valueOf(m.group(1));
        }
        catch (final NumberFormatException exc) {
            System.out.println(exc);
            throw new ParseError("Expected a decimal number.", input.toString());
        }
        consume(m, input);
        // Skip over units, if any.
        m = UNITS.matcher(input);
        if (m.find()) {
            consume(m, input);
        }
        // Skip over commentary, anything up to but not including a comma.
        m = COMMENTARY.matcher(input);
        if (m.find()) {
            consume(m, input);
        }
        return result;
    }
    
    // Move the current input position past whatever was matched by m.
    private static void consume(final Matcher m, final CharBuffer input) {
        input.position(input.position() + m.group().length());
    }
    
    /**
     * Represents a parsing error. The message string returned by {@code getMessage()} describes
     * what was expected followed by the first 30 characters of the input starting at the point at which
     * the error was detected followed by ellipses if needed.
     */
    public static class ParseError extends java.lang.RuntimeException {
        private static final long serialVersionUID = 1;
        /**
         * @param errMsg a short statement of what was expected that we didn't find.
         * @param remainingInput the unparsed input starting from the point at which the error occurred.
         */
        public ParseError(final String errMsg, final String remainingInput) {
            super(errMsg + " Input remaining: " + truncate(remainingInput));
        }
        
        private static String truncate(final String input) {
            if (input.length() <= 30) {
                return input;
            }
            else {
                return input.substring(0, 30) + " ...";
            }
        }
    }
    
    

    public static void main(final String[] arg) {
        final List<String> tests = new LinkedList<>();
        tests.add("X=1.5, X2=1.3mm, Z=38 deg");
        tests.add("X=1.5e-1, X2=1.3E+2mm, Z=38. deg");
        tests.add("X =1.5,X2  =  1.3mm,    Z=    38 deg");
        tests.add("X=1.5, X2:1.3mm, Z=38 deg");
        tests.add("X=1.5, X2 : 1.3mm, Z=38 deg");
        tests.add("X=1.5, X2=1.3mm, Z=38 deg 8%");
        tests.add("x=22,");
        tests.add("");
        
        for (final String t: tests) {
            try {
                System.out.println(t);
                final Map<String, Double> result = parse(t);
                System.out.println(result);
            }
            catch (final ParseError exc) {
                System.out.println(exc.getMessage());
            }
            System.out.println();
        }
    }
}
