package org.lsst.ccs.subsystem.ccob.thin;

import java.io.Serializable;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import static java.util.Objects.requireNonNull;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.RunMode;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.ccob.thin.buses.TBServerException;
import org.lsst.ccs.subsystem.motorplatform.bus.AxisStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.ControllerStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.motorplatform.bus.PlatformConfig;
import org.lsst.ccs.subsystem.motorplatform.bus.SendAxisStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.SendConfiguration;
import org.lsst.ccs.subsystem.motorplatform.bus.SendControllerStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.StopAllMotion;

/**
 * Communicates with the hardware. Translates CCS commands into commands for the TBServer. Provides
 * hardware configuration and status information. Thread-safe.
 * @author tether
 */
public class Controller implements HasLifecycle {
    
    // This class executes commands coming from several different client threads. It achieves
    // thread safety by handing off all commands to an execution thread which owns most of
    // the instance variables. The exceptions to this scheme are:
    //     Final fields which are static or set in the constructor. These are
    //     automatically safe-published.
    //
    //     CCS configuration parameters and lookup fields which are perforce declared volatile.
    //
    //     The lastReply field which is an atomic reference so that its target can be fetched
    //     at any time.
    //
    //     The isSimulation field which is a Boolean flag set once before the execution
    //     thread is started and which is therefore safe-published to that thread.
    
    // Each time the execution thread issues a command to the TB server it waits for the
    // TB server's reply. The server itself is synchronous; the reply comes back
    // only when the command is completed, even if it's a motion that takes a 
    // minute or more to finish.
    
    // The TB server can only handle one connection at a time, and appears to have an
    // open file leak due to improperly closed connections. Therefore this subsystem
    // opens a single connection which it uses for all commands and keeps that
    // connection open indefinitely; the subsystem must be shut down
    // in order for others to use the TB server.
    
    // The incoming commands conform to the motorplatform "standard" we've been using
    // in CCS. Some incoming commands will need to be translated into several TB
    // server commands. For example, there is no motorplatform command to set speed
    // or acceleration. Instead we will have to set them before every motion.
    // This means that there may be several replies from the TB server for a single
    // incoming command. If all goes well then the reply returned as "the" reply for
    // the incoming command will normally be the reply gotten for the last TB server
    // command issued in order to implement the incoming command. For motion this will be
    // the command that actually starts the motion. If however a command returns a reply
    // starting with "ERROR" then the sequence of commands is cut short and an exception
    // is generated with that reply as its message.
    
    // Client threads wait for the completion of incoming commands using the Future
    // mechanism, which will either return the last reply from the TB server or generate
    // an exception, which may be a time-out. Motorplatform commands return no values so any
    // reply returned is saved and can be retrieved with a special command. Exceptions
    // are returned to the command issuer through the usual CCS mechanism in which case the
    // saved reply will be set to a string starting with "ERROR:", which may be an
    // actual TB server reply which caused the exception to be raised or
    // a string made from the exception message if the exception didn't result from a TB server
    // error.
    
    // The TB server's reply to each command sent to it ends with "\n\n" so the subsystem will
    // collect reply text from the server until an empty line is seen. Leading and trailing
    // whitespace will be trimmed from all TB server replies.
    
    // Since the TB server is itself synchronous, we use a synchronous mechanism to pass
    // commands from the client threads to the execution thread. This means that the client
    // threads must wait until the execution thread has finished with the current command
    // and is ready to pick up a new one. We use fair scheduling so that no client will be
    // delayed indefinitely.
    
    // One client thread is created here. It runs a periodic task which upon each execution
    // submits a command to update the device newStatus information. It will block if the
    // execution thread is busy, so it's scheduled as fixed delay rather than fixed rate
    // so that the scheduler won't schedule extra executions to make up for delays.
    
    // There is no single hardware controller for the CCOB thin beam. Instead we treat
    // the TB server as the controller. We send an updated ControllerStatus
    // message whenever we try to connect to the TB server. The link newStatus becomes "UP"
    // if we succeed or "DOWN" if we fail. There is no global motion inhibit line
    // so the motion-enabled flag in the controller newStatus is always true. The
    // last-command-newStatus is either "RUNNING", "OK" or "ERROR".

    private static final Logger LOG = Logger.getLogger(Controller.class.getName());
        
    private final ScheduledExecutorService scheduler;
    
    private final BlockingQueue<CannedCommand> execQueue;
    
    private static final class CannedCommand {
        public final CompletableFuture<String> reply;
        public final Callable<String> command;
        public CannedCommand(CompletableFuture<String> reply, Callable<String> command) {
            this.reply = reply;
            this.command = command;
        }
    }
    
    private final AtomicReference<String> lastReply;
    
    private ControllerStatus ctrlStatus;
    
    private Map<String, AxisStatus> axisStati;
    
    
    ////////// Lookup fields (references to other subsystem components) //////////
    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    AlertService alertSvc;
    
        
    ////////// Configuration parameters //////////
    @ConfigurationParameter(
        description = "The interval between automatic status updates and publishing.",
        units = "s"
    )
    private volatile Duration statusUpdateInterval = Duration.ofSeconds(30);
    
    @ConfigurationParameter(
        description = "How long to wait for the TB server to respond to a command.",
        units = "s"
    )
    private volatile Duration responseTimeout = Duration.ofSeconds(120);
    
    @ConfigurationParameter(
        description = "The IPv4 address (or hostname) of the TB server."
    )
    private volatile String tbIpv4Address = "127.0.0.1";
    
    @ConfigurationParameter(
        description = "The TCP port on which the TB server is listening."
    )
    private volatile int tbTcpPort = 1557;
    
    @ConfigurationParameter(
        description = "The frequency of the power main to which the CCOB is connected. 50 or 60.",
        units = "Hz"
    )
    private volatile double powerHz = 60.0;
    
    ////////// Lifecycle methods //////////

    public Controller() {
        this.scheduler = Executors.newScheduledThreadPool(2);
        this.execQueue = new SynchronousQueue<>();
        this.lastReply = new AtomicReference<>("");
        this.ctrlStatus = new ControllerStatus(
            "UP", // Comm link newStatus.
            true,      // Is motion enabled?
            "OK"       // Final status of last command to controller. RUNNING, OK or ERROR.
        );
        this.axisStati = config.getAxisNames().stream()
            .map( axname -> new AxisStatus(
                axname,
                true,   // Enabled.
                false,  // Moving.
                false,  // Homed.
                false,  // At low limit.
                false,  // At high limit.
                Collections.emptyList(),  // Axis faults (errors).
                0.0))   // Position.
            .collect(Collectors.toMap( stat -> stat.getAxisName(), stat -> stat));
            
    }

    @Override
    public void init() {
        // Register newStatus bus messages that need to be trended. The PlatformConfig message 
        // does not need to be trended.
        final DataProviderDictionaryService dictService =
            subsys.getAgentService(DataProviderDictionaryService.class);
        dictService.registerData(new KeyValueData(kvdKey(ctrlStatus), ctrlStatus));
        for (final AxisStatus status: axisStati.values()) {
            final KeyValueData kvd = new KeyValueData(kvdKey(status), status);
            dictService.registerData(kvd);
        }
        Alerts.registerAllAlerts(alertSvc);
    }
    
    private boolean isSimulation = false;

    /**
     * Starts two tasks. One is long running and executes commands. The other acts as
     * an internal client and periodically submits a status-update command.
     */
    @Override
    public void start() {
        isSimulation = RunMode.isSimulation();
        scheduler.submit(this::commandLoop);
        submit( () -> {
            try (final AsciiAuto driver = getOpenDriver()) {
                // Don't want escape sequences mixed in with the text.
                return sendCommandAndCheck(driver, "COLOR OFF");
            }
        } );
        scheduler.scheduleWithFixedDelay(() -> submitUpdateStatus(),
            0,
            statusUpdateInterval.toMillis(),
            TimeUnit.MILLISECONDS);
    }

    /**
     * Shuts down all tasks.
     */
    @Override
    public void postShutdown() {
        scheduler.shutdownNow();
    }
    
    ////////// Command scheduling //////////

    public void moveAxisRelative(final MoveAxisRelative cmd) {
        submit( () -> doMoveAxisRelative(cmd) );    
    }

    public void moveAxisAbsolute(final MoveAxisAbsolute cmd) {
        submit( () -> doMoveAxisAbsolute(cmd) );
    }

    public void sendAxisStatus(final SendAxisStatus cmd) {
        submit( () -> doSendAxisStatus(cmd) );
    }

    private static final PlatformConfig config;
    static {
        final List<String> axes  = Arrays.asList("X",  "Y",  "U",   "B",   "P");
        final List<String> units = Arrays.asList("mm", "mm", "deg", "deg", "deg");
        config = new PlatformConfig(
            axes,
            Collections.emptyList(), // Motion captures.
            Collections.emptyList(), // Digital inputs.
            Collections.emptyList(), // Digital outputs.
            Collections.emptyList(), // Analog inputs.
            "CCOB Thin",             // Configuration name.
            units
        );
    }

    public void sendConfiguration(final SendConfiguration cmd) {
        // The configuration never changes so we can just send it directly.
        subsys.publishSubsystemDataOnStatusBus(
            new KeyValueData("PlatformConfig", config)
        );
        
        // However, let's send the current newStatus out so that the GUI (if used)
        // can record the state upon connection to the subsystem.
        submit( () -> doRegularStatusUpdate() );
    }

    public void sendControllerStatus(final SendControllerStatus cmd) {
        submit( () -> doSendControllerStatus(cmd) );
    }

    public void stopAllMotion(final StopAllMotion cmd) {
        submit( () -> doStopAllMotion(cmd) );
    }

    public String status(final String axisName) {
        final CompletableFuture<String> result = new CompletableFuture<>();
        submit( () -> completeFuture(result, doStatus(axisName)) );
        return getFuture(result);
    }
    
    public void switchDiode(final String newState) {
        submit( () -> doSwitchDiode(newState) );
    }

    public double picoReadCurrent() {
        final CompletableFuture<Double> result = new CompletableFuture<>();
        submit( () -> doPicoReadCurrent(result) );
        return getFuture(result);
    }

    public void picoSetTime(double millis) {
        submit( () -> doPicoSetTime(millis) );
    }

    public void picoSetRange(int range) {
        submit( () -> doPicoSetRange(range) );
    }

    public double picoGetRange() {
        final CompletableFuture<Double> result = new CompletableFuture<>();
        submit( () -> doPicoGetRange(result) );
        return getFuture(result);
    }

    public void hyperSetWavelength(final String wavelength) {
        submit( () -> doHyperSetWavelength(wavelength) );
    }

    public void hyperSwitchFastShutter(final String newState) {
        submit( () -> doHyperSwitchFastShutter(newState) );
    }

    public void hyperStartFastExposure(final int millis) {
        submit( () -> doHyperStartFastExposure(millis) );
    }

    public void hyperSwitchMainShutter(final String newState) {
        submit( () -> doHyperSwitchMainShutter(newState) );
    }

    public void hyperSwitchLamp(final String newState) {
        submit( () -> doHyperSwitchLamp(newState) );
    }

    public void hyperZOP(String gratingName) {
        submit( () -> doHyperZOP(gratingName) );
    }

    public double getDiodeAttenuation() {
        final CompletableFuture<Double> result = new CompletableFuture<>();
        submit( () -> doGetDiodeAttenuation(result) );
        return getFuture(result);
    }

    public double readThenIlluminate(final int millis) {
        final CompletableFuture<Double> result = new CompletableFuture<>();
        submit( () -> doReadThenIlluminate(result, millis) );
        return getFuture(result);
    }

    public double illuminateThenRead(final int millis) {
        final CompletableFuture<Double> result = new CompletableFuture<>();
        submit( () -> doIlluminateThenRead(result, millis) );
        return getFuture(result);
    }

    public String getTarget() {
        final CompletableFuture<String> result = new CompletableFuture<>();
        submit( () -> doGetTarget(result) );
        return getFuture(result);
    }

    void setTargetTo(double x, double y) {
        submit( () -> doSetTargetTo(x, y) );
    }

    void setTargetHere() {
        submit( this::doSetTargetHere );
    }

    void aimAgainXY() {
        submit( this::doAimAgainXY );
    }

    void aimAgainUB() {
        submit(  this::doAimAgainUB );
    }

    public String getLastReply() {return lastReply.get();}

    ////////// Command execution //////////
    
    // All the "do" methods are meant to be called only from the command loop thread.
    
    // Takes Callable<String> objects from a queue. Each returns a string which is the response of
    // the TB server to the last command sent to it by the callable. The string is used to
    // complete the Future sent along with the command. Since that will unblock the client,
    // the completion should be done only after all state changes have been made.
    private void commandLoop() {
        LOG.info("Starting the command execution task.");
        while (!Thread.currentThread().isInterrupted()) {
            CannedCommand canned;

            try {
                canned = execQueue.take();
            } catch (InterruptedException exc) {
                break;
            }

            try {
                final String result = canned.command.call();
                setCommandStatus("OK");
                canned.reply.complete(result);
            }
            catch (final InterruptedException exc) {
                canned.reply.completeExceptionally(exc);
                break;
            }
            catch (final TBServerException exc) {
                LOG.log(Level.SEVERE, "Error from TB server.", exc);
                Alerts.raiseServerAlert(alertSvc, exc);
                canned.reply.completeExceptionally(exc);
            }
            catch (final DriverException exc) {
                LOG.log(Level.SEVERE, "Error communicating with TB_Server.", exc);
                Alerts.raiseCommAlert(alertSvc, exc);
                setConnectionStatus("DOWN");
                discardCurrentDriver();
                setCommandStatus("ERROR");
                canned.reply.completeExceptionally(exc);
            } catch (final Exception exc) {
                LOG.log(Level.SEVERE, "Internal error.", exc);
                Alerts.raiseInternalAlert(alertSvc, exc);
                setCommandStatus("ERROR");
                canned.reply.completeExceptionally(exc);
            }
        }
        LOG.info("Normal termination of the command execution task.");
    }
    
    // Called by methods that want to send commands to TB server via the command loop.
    // The {@code lastReply} reference will always be given a new string to point to.
    // Will set the thread interrupt flag if {@code InterruptedException} is caught.
    // All exceptions resulting from the command execution are caught and re-thrown
    // wrapped in instances of {@code RuntimeException}.
    private void submit(final Callable<String> code) {
        final CannedCommand canned = new CannedCommand(new CompletableFuture<String>(), code);
        try {
            execQueue.put(canned);
        } catch (InterruptedException ex) {
            LOG.warning("Subsystem was interrupted.");
            lastReply.set("WARNING: Subsystem was interrupted.");
            Thread.currentThread().interrupt();
            return;
        }
        try {
            lastReply.set(canned.reply.get());
        } catch (InterruptedException exc) {
            LOG.warning("Subsystem was interrupted.");
            Thread.currentThread().interrupt();
            lastReply.set("WARNING: Subsystem was interrupted.");
        } catch (ExecutionException exc) {
            final Throwable cause = exc.getCause();
            throw new RuntimeException(cause);
        }
    }

    // The TB server's V command, e.g., "XY V=<speed>", takes a zero or positive value for <speed>. A value
    // of zero is interpreted as a request to leave the current value unchanged. The command sets the target
    // speed for the next motion, that is, the speed to attain after the initial acceleration.
    
    // The TB server's ACC command, e.g., "XY ACC=<accel>", takes a zero or positive value for <accel>.
    // A value of zero for <accel> leaves the current value unchanged. The command  sets the absolute
    // values of the initial acceleration and final deceleration for the next motion.
    
    // Note that the acceleration and speed values for the X and Y axes are always set jointly;
    // the axis names in the V and ACC commands is always "XY", never just "X" or "Y". Also, the P
    // axis can't have its speed or acceleration changed. 

    // The motorplatform convention for relative motion is to supply a signed change in coordinate
    // together with the amount of time the motion should take. For absolute motion the desired
    // final position together with an unsigned target speed are specified.
    
    private String doMoveAxisRelative(final MoveAxisRelative cmd) throws DriverException {
        final String axis = checkAxisName(cmd.getAxisName());
        if (axis.equals("P")) {
            throw new IllegalArgumentException("The P axis doesn't implement relative motion commands.");
        }
        try (final AsciiAuto driver = getOpenDriver()) {
            // Solve for the parameters of a motion with a symmetrical trapezoidal velocity profile.
            final double dt = 0.1; // Time allowed to start and stop motion.
            final double D = cmd.getDistance();
            final double T = 1e-3 * cmd.getTime().toMillis();
            final double a = Math.abs(D) / (dt * T - dt * dt);
            final double v = a * dt;
            final String tbAxis = "XY".contains(axis) ? "XY" : axis;
            sendCommandAndCheck(driver, String.format("%s V=%.1f", tbAxis, v));
            sendCommandAndCheck(driver, String.format("%s ACC=%.1f", tbAxis, a));
            publishMotionStatus(axis, true);
            final String motionReply  =
                sendCommandAndCheck(driver, String.format("%s+=%.1f", axis, cmd.getDistance()));
            return motionReply;
        }
        finally {
            publishMotionStatus(axis, false);
        }
    }

    private String doMoveAxisAbsolute(final MoveAxisAbsolute cmd) throws DriverException {
        final String axis = checkAxisName(cmd.getAxisName());
        try (final AsciiAuto driver = getOpenDriver()) {
            if (!axis.equals("P")) {
                // All the axes except for P implement V= and ACC= .
                final double dt = 0.1; // Time allowed to start and stop motion.
                final double v = cmd.getSpeed();
                final double a = v / dt;
                final String tbAxis = "XY".contains(axis) ? "XY" : axis;
                sendCommandAndCheck(driver, String.format("%s V=%.1f", tbAxis, v));
                sendCommandAndCheck(driver, String.format("%s ACC=%.1f", tbAxis, a));
            }
            else {
                LOG.warning("For most work use diodeOn() and diodeOff() rather than explicit P-axis motion.");
            }
            publishMotionStatus(axis, true);
            final String motionReply  =
                sendCommandAndCheck(driver, String.format("%s=%.1f", axis, cmd.getPosition()));
            return motionReply;
        }
        finally {
            publishMotionStatus(axis, false);
        }
    }

    // A no-op. We'll wait for the regular newStatus updates.
    private String doSendAxisStatus(final SendAxisStatus cmd) throws DriverException {
        return "OK";
    }

    // A no-op. We'll wait for the regular updates.
    private String doSendControllerStatus(SendControllerStatus cmd) {
        return "OK";
    }

    // Since the TB server won't respond to a new command until it has finished the one before,
    // there's simply no way to use the TB server to stop a motion in progress.
    private String doStopAllMotion(final StopAllMotion cmd) throws DriverException {
        throw new RuntimeException("There is no way to use the TB server to stop a motion in progress.");
    }

    // For now, just get all the axis positions since the command responses are easy to parse.
    // Later on we'll see if we can reliably parse the output of newStatus commands such as
    // "XY STATUS", "UB STATUS", etc.
    private String doRegularStatusUpdate() throws DriverException {
        String reply = ERROR_PREFIX;
        for (final String axisName : config.getAxisNames()) {
            try (final AsciiAuto driver = getOpenDriver()) {
                reply = sendCommandAndCheck(driver, axisName + "?");
                final Map<String, Double> values = ValueParser.parse(reply);
                final AxisStatus newStatus = new AxisStatus(
                    axisName,
                    true, // Enabled.
                    false, // Moving.
                    false, // Homed.
                    false, // At low limit.
                    false, // At high limit.
                    Collections.emptyList(), // Axis fault (error) list.
                    requireNonNull(values.get(axisName), "Can't find info for axis " + axisName)); // Position.
                axisStati.put(axisName, newStatus);
                publish(newStatus);
            }
            catch (ValueParser.ParseError exc) {
                LOG.log(Level.WARNING, "Could not parse reply to {0}?\n{1}",
                    new Object[]{axisName, exc.getMessage()});
                throw new DriverException("The server's reply was un-parseable.");
            }
        }
        return reply;
    }
    
    private String doStatus(String controllerName) throws DriverException {
        controllerName = controllerName.toUpperCase();
        try (final AsciiAuto driver = getOpenDriver()) {
            if (" XY UB P H K ".contains(controllerName)) {
                return sendCommandAndCheck(driver, controllerName + " STATUS");
            }
            else if (controllerName.startsWith("SUMM")) {
                return sendCommandAndCheck(driver, "STATUS");
            }
            else {
                throw new IllegalArgumentException("Valid controller names are XY, UB, P, H, K or SUMMARY.");
            }
        }
    }
    
    private String doSwitchDiode(final String newState) throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            return sendCommandAndCheck(driver, "P DIODE " + newState);
        }
    }
    
    private String doPicoReadCurrent(final CompletableFuture<Double> result) throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            final String reply = sendCommandAndCheck(driver, "K?");
            completeFuture(result, ValueParser.parse(reply).get("K"));
            return reply;
        }
    }

    private String doPicoSetTime(final double millis)  throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            final String cmd = String.format("K PLC=%.2f", 1e-3 * millis * powerHz);
            return sendCommandAndCheck(driver, cmd);
        }
    }

    private String doPicoSetRange(final int range)  throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            final String cmd = String.format("K RANGE=%d", range);
            return sendCommandAndCheck(driver, cmd);
        }
    }

    private String doPicoGetRange(final CompletableFuture<Double> result) throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            String reply = sendCommandAndCheck(driver, "K RANGE?");
            // Format of reply:
            //     Range: 123.45A (Table of valid ranges)
            // Drop the range table from the reply.
            reply = reply.substring(0, reply.indexOf('('));
            completeFuture(result, ValueParser.parse(reply).get("Range"));
            return reply;
        }
    }

    private String doHyperSetWavelength(final String wavelength) throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            return sendCommandAndCheck(driver, "H=" + wavelength.toUpperCase());
        }
    }

    private String doHyperSwitchFastShutter(final String newState) throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            return sendCommandAndCheck(driver, "H HSS " + newState);
        }
    }

    private String doHyperStartFastExposure(final int millis) throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            return sendCommandAndCheck(driver, "H HSS=" + millis);
        }
    }

    private String doHyperSwitchMainShutter(final String newState) throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            return sendCommandAndCheck(driver, "H SHUTTER " + newState);
        }
    }

    private String doHyperSwitchLamp(final String newState) throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            return sendCommandAndCheck(driver, "H LAMP " + newState);
        }
    }

    private String doHyperZOP(final String gratingName) throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            return sendCommandAndCheck(driver, "H=ZOP" + gratingName.toUpperCase());
        }
    }

    private String doGetDiodeAttenuation(final CompletableFuture<Double> result) throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            final String reply = sendCommandAndCheck(driver, "KC?");
            // KC? returns just a number with no key or units.
            completeFuture(result, ValueParser.parse("KC=" + reply + "u").get("KC"));
            return reply;
        }
    }

    private String doReadThenIlluminate(final CompletableFuture<Double> result, final int millis)
    throws DriverException
    {
        try (final AsciiAuto driver = getOpenDriver()) {
            final String reply = sendCommandAndCheck(driver, "KH HSS=" + millis);
            completeFuture(result, ValueParser.parse(reply).get("K"));
            return reply;
        }
    }

    private String doIlluminateThenRead(final CompletableFuture<Double> result, final int millis)
    throws DriverException
    {
        try (final AsciiAuto driver = getOpenDriver()) {
            final String reply = sendCommandAndCheck(
                driver,
                String.format("HK PLC=%.1f", 1e-3 * millis * powerHz));
            completeFuture(result, ValueParser.parse(reply).get("K"));
            return reply;
        }
    }

    private String doGetTarget(final CompletableFuture<String> result) throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            final String reply = sendCommandAndCheck(driver, "TARGET?");
            completeFuture(result, reply);
            return reply;
        }
    }

    private String doSetTargetTo(final double x, final double y) throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            return sendCommandAndCheck(driver, String.format("TARGET=%.1f,%.1f", x, y));
        }
    }

    private String doSetTargetHere() throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            return sendCommandAndCheck(driver, "TARGET");
        }
    }

    private String doAimAgainXY() throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            return sendCommandAndCheck(driver, "AAXY");
        }
    }

    private String doAimAgainUB() throws DriverException {
        try (final AsciiAuto driver = getOpenDriver()) {
            return sendCommandAndCheck(driver, "AAUB");
        }
    }
        

    ////////// Internal client //////////
    
    //Executed at regular intervals in order to publish hardware newStatus.
    // Suppresses all exceptions since the logging and alert raising are handled
    // elsewhere.
    private void submitUpdateStatus() {
        try {
            submit( () -> doRegularStatusUpdate() );
        }
        catch (final Throwable exc) {
        }
    }

    ////////// Utility /////////

    // Supplies an open AsciiAuto driver upon request. Must be called only from the command loop thread.
    // Opens a new connection if required and thereafter supplies that connection. The driver is stored
    // in a field owned by the command thread. The variable is null if a new connection is needed.
    // If we're simulating then always return a new MockThin instance.
    private AsciiAuto getOpenDriver() throws DriverException {
        // Assume that the controller is OK until an error occurs.
        setConnectionStatus("UP");
        setCommandStatus("RUNNING");
        if (isSimulation) {
            return new MockThin();
        }
        if (driverRef == null) {
            final AsciiAuto driver = new AsciiAuto();
            driver.setResponseTerm(Ascii.Terminator.LF);
            driver.setTimeout(1e-3 * responseTimeout.toMillis());
            driver.openNet(tbIpv4Address, tbTcpPort);
            driverRef = driver;
        }
        return driverRef;
    }
    
    // Used after the current driver has thrown an error and a new one is needed. Must be called
    // only from the command thread.
    private void discardCurrentDriver() {
        if (driverRef != null) {
            driverRef.closeForReal();
        }
        driverRef = null;
    }

    private AsciiAuto driverRef = null;

    private String checkAxisName(final String in) throws IllegalArgumentException {
        final String out = in.toUpperCase();
        if (!config.getAxisNames().contains(out)) {throw new IllegalArgumentException("Bad axis name " + out);}
        return out;
    }
    
    private static final String ERROR_PREFIX = "ERROR:";
    
    private static final String WARNING_PREFIX = "Warning:";

    private String sendCommandAndCheck(final Ascii driver, final String tbserverCommand) 
    throws DriverException
    {
        // Send the command then collect the reply stripped of white space. If the reply
        // is an error message then throw a TBServerException.
        driver.write(tbserverCommand);
        final String reply = collectReply(driver);
        if (reply.startsWith(ERROR_PREFIX) || reply.startsWith(WARNING_PREFIX)) {
            throw new TBServerException(reply);
        }
        return reply;
    }
    
    private String collectReply(final Ascii driver) throws DriverException {
        final List<String> lines = new LinkedList<>();
        String s;
        do  {
            s = driver.read();
            lines.add(s);
        }
        while (!s.isEmpty());
        return String.join("\n", lines).trim();
    }

    private void setConnectionStatus(final String stat) {
        ctrlStatus = new ControllerStatus(
            stat,
            ctrlStatus.isMotionEnabled(),
            ctrlStatus.getLastCommandStatus()
        );
        publish(ctrlStatus);
    }
    
    private void setCommandStatus(final String stat) {
        ctrlStatus = new ControllerStatus(
            ctrlStatus.getCommLinkStatus(),
            ctrlStatus.isMotionEnabled(),
            stat
        );
        publish(ctrlStatus);
    }
    
    // Most of our newStatus bus messages have keys that are just the simple class name.
    private String kvdKey(final Serializable value) {
        return value.getClass().getSimpleName();
    }
    
    private void publish(final Serializable value) {
        // Value must be immutable so taht no defensive copying is needed.
        final KeyValueData kvd = new KeyValueData(kvdKey(value), value);
        subsys.publishSubsystemDataOnStatusBus(kvd);
    }
    
    // AxisStatus messages have the axis name added to the key. That way each axis
    // will have its own heading in trending.
    private String kvdKey(final AxisStatus value) {
        return kvdKey((Serializable)value) + "/" + value.getAxisName();
    }

    final void publish(final AxisStatus value) {
        // Value must be immutable.
        final KeyValueData kvd = new KeyValueData(kvdKey(value), value);
        subsys.publishSubsystemDataOnStatusBus(kvd);
    }
    
    private void publishMotionStatus(final String axisName, final boolean isMoving) {
        final AxisStatus oldstat = axisStati.get(axisName);
        final AxisStatus newstat = new AxisStatus(
            axisName,
            oldstat.isEnabled(),
            isMoving,
            oldstat.isAtHome(),
            oldstat.isAtLowLimit(),
            oldstat.isAtHighLimit(),
            oldstat.getFaults(),
            oldstat.getPosition()
        );
        axisStati.put(axisName, newstat);
        publish(newstat);
    }
    
    // Completes a future other than the one in the CannedCommand instance. Returns the supplied value.
    // Use only inside code sent to submit().
    private <T> T completeFuture(final CompletableFuture<T> fut, final T value) {
        fut.complete(value);
        return value;
    }
    
    // Gets a value from a future other than the one in the CannedCommand instance. Returns
    // the completed value or throws an exception. Use only in the thread that called submit().
    // Exceptions are wrapped inside RuntimeException so that they make it back to the subsystem framework
    // without needing to be declared.
    private <T> T getFuture(final CompletableFuture<T> fut) {
         try {
            return fut.get();
        } catch (final ExecutionException exc) {
            final Throwable cause = exc.getCause();
            throw new RuntimeException(cause);
        } catch (final Throwable exc) {
            throw new RuntimeException(exc);
        }
         
    }
}
