package org.lsst.ccs.subsystem.ccob.thin;

import java.util.LinkedList;
import java.util.List;
import static java.util.concurrent.TimeUnit.SECONDS;
import java.util.regex.Pattern;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Provides a dummy implementation of the thin-beam CCOB API. Logs CCOB commands at level INFO and
 * provides suitable made-up responses to callers. Thread-safe.
 * <p>
 * The only overridden ASCII driver methods that actually do anything are {@link #read(java.lang.String) },
 * {@link #read() } and {@link #write(java.lang.String) }.
 * @author tether
 */
public class MockThin extends AsciiAuto {

    // The ASCII driver's read() method returns a single line, but the TB server's response to a command
    // always has at least two lines: the actual reponse plus an empty line acting as a terminator.
    // Store replies in this queue, one line per element, no line terminators.
    private final List<String> replyQueue;

    public MockThin() {
        this.replyQueue = new LinkedList<>();
    }

    private static final Logger LOG = Logger.getLogger(MockThin.class.getName());

    /** Does nothing. */
    @Override
    public void setTimeout(int time) throws DriverException {
    }

    /** Does nothing. */
    @Override
    public void setTimeout(double time) throws DriverException {
    }

    /** Does nothing. */
    @Override
    public void setTerminator(Terminator term) {
    }

    /** Does nothing. */
    @Override
    public void setCommandTerm(Terminator term) {
    }

    /** Does nothing. */
    @Override
    public void setResponseTerm(Terminator term) {
    }

    /** Does nothing. */
    @Override
    public void flush() throws DriverException {
    }

    /** 
     * @throws UnsupportedOperationException every time. 
     */
    @Override
    public int readBytes(byte[] buff, int offset, int leng) throws DriverException {
        throw new UnsupportedOperationException("Not implemented");
    }

    /** 
     * @throws UnsupportedOperationException every time. 
     */
    @Override
    public int readBytes(byte[] buff, int offset) throws DriverException {
        throw new UnsupportedOperationException("Not implemented");
    }
    /** Does nothing. */

    /** 
     * @throws UnsupportedOperationException every time. 
     */
    @Override
    public void writeBytes(byte[] command, int offset, int leng) throws DriverException {
        throw new UnsupportedOperationException("Not implemented");
    }

    /** 
     * @throws UnsupportedOperationException every time. 
     */
    @Override
    public void writeBytes(byte[] command) throws DriverException {
        throw new UnsupportedOperationException("Not implemented");
    }


    @Override
    public String read(String command) throws DriverException {
        write(command);
        return read();
    }

    @Override
    public String read() throws DriverException {
        try {
            return replyQueue.remove(0);
        }
        catch (final IndexOutOfBoundsException exc) {
            throw new DriverException("The response is missing or incomplete, e.g., unterminated.", exc);
        }
    }

    @Override
    public void write(final String command) throws DriverException {
        final boolean alwaysFail = true;
        LOG.info(command);
        if (alwaysFail && !command.startsWith("COLOR")) {
            replyQueue.add("ERROR: The simulation is in always-fail mode.");
            replyQueue.add("");
            return;
        }
        // Add a delay for motion commands.
        if (isMotionCommand(command)) {
            try {
                SECONDS.sleep(3);
            } catch (InterruptedException exc) {
                LOG.info("Command execution was interrupted!");
                Thread.currentThread().interrupt();
            }
        }
        if (command.startsWith("KC?")) {
            replyQueue.add("15.0");
        }
        else {
            replyQueue.add("X=1.5e-1mm, Y=0.0mm, U=0.0deg, B=0.0deg, P=0.0deg, Range: 42.0A (blah blah), K=3.14A");
        }
        replyQueue.add("");
    }

    /** Does nothing but return {@code true}. */
    @Override
    public boolean closeSilent() {
        return true;
    }

    /** Does nothing. */
    @Override
    public void close() throws DriverException {
    }

    /** Does nothing. */
    @Override
    public void openFtdi(String serialNo, int baudRate) throws DriverException {
    }

    /** Does nothing. */
    @Override
    public void openFtdi(String serialNo, int baudRate, int dataChar) throws DriverException {
    }

    /** Does nothing. */
    @Override
    public void openSerial(String devcName, int baudRate) throws DriverException {
    }

    /** Does nothing. */
    @Override
    public void openSerial(String devcName, int baudRate, int dataChar) throws DriverException {
    }

    /** Does nothing. */
    @Override
    public void openNet(String host, int port) throws DriverException {
    }

    /** Does nothing. */
    @Override
    public void open(ConnType type, String ident, int parm) throws DriverException {
    }

    /** Does nothing. */
    @Override
    public void open(ConnType type, String ident, int parm1, int parm2) throws DriverException {
    }
    
    private static final Pattern COMMAND_PREFIX = Pattern.compile("[XYUBP]\\+?=");
    
    private boolean isMotionCommand(final String command) {
        return COMMAND_PREFIX.matcher(command).lookingAt();
    }
}
