package org.lsst.ccs.subsystem.ccob;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.drivers.ccob.CCOBCommands;
import org.lsst.ccs.drivers.ccob.CCOBDataRead;
import org.lsst.ccs.drivers.ccob.CCOBInterface;
import org.lsst.ccs.drivers.ccob.CCOBSocketDriver;
import org.lsst.ccs.drivers.ccob.CCOBUsb;
import org.lsst.ccs.drivers.ccob.jni.CcobJNI;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * A device around a CCOB driver
 * @author LSST CCS Team
 */
public class CCOBDevice extends Device {

    public static final int
        CHAN_LED_VOLTAGE = CCOBDataRead.ADC_LED_VOLTAGE,
        CHAN_LED_CURRENT = CCOBDataRead.ADC_LED_CURRENT,
        CHAN_LED_VREF    = CCOBDataRead.ADC_LED_VREF,
        CHAN_LED_TEMP1   = CCOBDataRead.ADC_LED_TEMP1,
        CHAN_LED_TEMP2   = CCOBDataRead.ADC_LED_TEMP2,
        CHAN_PD_CURRENT  = CCOBDataRead.ADC_PD_CURRENT,
        CHAN_SPHERE_TEMP = CCOBDataRead.ADC_SPHERE_TEMP,
        CHAN_BOARD_TEMP  = CCOBDataRead.ADC_BOARD_TEMP,
        CHAN_PD_ADC      = CCOBDataRead.NUM_ADCS,
        NUM_CHANNELS     = CCOBDataRead.NUM_ADCS + 1;

    private static final int
        CONN_TYPE_USB = 0,
        CONN_TYPE_SOCKET = 1,
        CONN_TYPE_JNI = 2,
        CONN_TYPE_SIM = 3;

    private static final Map<String, Integer> connTypeMap = new HashMap<>();
    static {
        connTypeMap.put("usb", CONN_TYPE_USB);
        connTypeMap.put("socket", CONN_TYPE_SOCKET);
        connTypeMap.put("jni", CONN_TYPE_JNI);
    }

    @LookupField(strategy=Strategy.TREE)
    private CCOBCommands cmnds;
    
    private String connType = "usb";
    private String node;
    private int index = 0;
    private int port = 1557;

    private static final Logger LOG = Logger.getLogger(CCOBDevice.class.getName());
    private CCOBInterface driver;
    private CCOBDataRead readValues;

    /**
     *  Checks that CCOB is specified correctly
     */
    @Override
    protected void initDevice() {
        LOG.info("Initializing CCOBDevice");
        Integer cType = connTypeMap.get(connType.toLowerCase());
        if (cType == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "connType", "is invalid");
        }
        if (cType == CONN_TYPE_USB) {
            driver = new CCOBUsb(index);
            fullName = "CCOB (USB)";
        }
        else if (cType == CONN_TYPE_SOCKET) {
            if (node == null) {
                MonitorLogUtils.reportConfigError(LOG, name, "node", "is not specified");
            }
            driver = new CCOBSocketDriver(node, port);
            fullName = "CCOB (Socket:" + node + ":" + port + ")";
        }
        else {
            driver = new CcobJNI();
            fullName = "CCOB (JNI)";
        }
        cmnds.setDriver(driver);
    }

    /**
     *  Initializes (opens a connection to) the CCOB.
     */
    @Override
    protected void initialize() {
        try {
            driver.init();
            setOnline(true);
            LOG.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                LOG.severe("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }

    /**
     *  Closes the connection to the CCOB.
     */
    @Override
    protected void close() {
        try {
            driver.stop();
        }
        catch (DriverException e) {
            // Just ignore any problems here
        }
    }

    /**
     * Checks a channel definition for validity.
     * 
     * @param name
     * @param hwChan
     * @param type
     * @param subtype
     * @return
     * @throws Exception 
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception {
        if (hwChan < 0 || hwChan >= NUM_CHANNELS) {
            MonitorLogUtils.reportError(LOG, name, "hw channel number", hwChan);
        }
        return new int[]{0, 0};
    }

    /**
     * Reads all the grouped data channels.
     */
    @Override
    public void readChannelGroup() {
        if (!isOnline()) return;
        try {
            readValues = driver.getAdcValues();
        } catch (DriverException ex) {
            LOG.error("Error reading " + fullName + " ADC group: " + ex);
            setOnline(false);
            readValues = null;
        }
    }

    /**
     * Returns a channel value.
     * 
     * @param hwChan
     * @param type
     * @return 
     */
    @Override
    public double readChannel(int hwChan, int type) {
        double value = super.readChannel(hwChan, type);
        if (isOnline()) {
            if (hwChan != CHAN_PD_ADC) {
                value = readValues != null ? readValues.getDoubleArray()[hwChan] : Double.NaN;
            }
            else {
                try {
                    value = driver.getAdcPhotoDiode();
                }
                catch (DriverException ex) {
                    LOG.error("Error reading " + fullName + " phodiode ADC: " + ex);
                    setOnline(false);
                }
            }
        }
        return value;
    }
    
}
