package org.lsst.ccs.subsystem.ccob;

import java.time.Duration;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.ccob.CCOBDataRead;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.ccob.constants.CCOBAgentProperties;
import org.lsst.ccs.subsystem.ccob.data.CCOBState;
import org.lsst.ccs.utilities.logging.Logger;

public class CCOBMainModule implements HasLifecycle {
    
    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService agentPropertiesService;
    
    private static final Logger LOG = Logger.getLogger(CCOBMainModule.class.getName());
    
    // -- HasLifecycle callbacks -----------------------------------------------
    @Override
    public void postInit() {
        agentPropertiesService.setAgentProperty(CCOBAgentProperties.CCOB_AGENT, CCOBMainModule.class.getCanonicalName());
    }
    
    @Override
    public void postStart() {
        LOG.info("CCOB subsystem started");
    }
    
    /**
     *  Gets the full state of the CCOB module.
     *
     *  This is intended to be called by GUIs during initialization
     *  @return  The full Raft state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the CCOB state")
    public CCOBState getFullState()
    {
        return getState();
    }    

    /**
     *  Sets the update period.
     *
     *  @param  value  The update period (milliseconds) to set.
     */
    @Command(type=Command.CommandType.ACTION, description="Set the update interval")
    public void setUpdatePeriod(@Argument(description="The tick period (ms)") int value)
    {
        setTickPeriod(value);
        publishState();
    }

    /**
     *  Gets the ADC values.
     *
     *  @return  The ADC values
     *  @throws  DriverException
     */
    @Command
    public CCOBDataRead getADCValues() throws DriverException {
        // Placeholder for now
        return null;
    }
    
    /**
     * Turn On the led. 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
             description="Turn On the led.")
    public void turnOnLed() {
        // Placeholder for now
    }

    /**
     * Turn Off the led. 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
             description="Turn Off the led.")   
    public void turnOffLed() {
        // Placeholder for now
    }

    /**
     *  Gets the state of the CCOB module.
     *
     *  @return  The CCOB state
     */
    private CCOBState getState() {
        return new CCOBState(getTickPeriod());
    }    

    /**
     *  Publishes the state of the CCOB module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(CCOBState.KEY, getState()));
    }

    /**
     *  Sets the monitoring publishing period
     */
    private void setTickPeriod(long period)
    {
        pts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    
    /**
     *  Gets the monitoring publishing period
     */
    private int getTickPeriod()
    {
        return (int)pts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

}
