package org.lsst.ccs.subsystem.ccob;

import java.time.Duration;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.drivers.ccob.CCOBDataRead;
import org.lsst.ccs.drivers.ccob.CCOBCommands;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.ccob.data.CCOBAgentProperties;
import org.lsst.ccs.subsystem.ccob.data.CCOBFullState;
import org.lsst.ccs.subsystem.ccob.data.CCOBState;
import org.lsst.ccs.utilities.logging.Logger;

public class CCOBMainModule implements HasLifecycle {
    
    /* A Logger for all the subsystem CCOB.*/
    public static final Logger ccobLog = Logger.getLogger("org.lsst.ccs.subsystem.ccob");
    
    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;
    
    @LookupField(strategy=Strategy.CHILDREN)
    private CCOBCommands driver;
    
    @LookupField(strategy=Strategy.CHILDREN)
    Led led;
    
    @LookupField(strategy=Strategy.CHILDREN)
    Shutter shutter;
    Thermometer thermometer;

    // -- HasLifecycle callbacks -----------------------------------------------
    @Override
    public void postInit() {
        subsys.setAgentProperty(CCOBAgentProperties.CCOB_TYPE_AGENT_PROPERTY,
                                CCOBMainModule.class.getCanonicalName());
    }
    
    @Override
    public void postStart() {
        ccobLog.info("CCOB subsystem started");
    }
    
    /**
     *  Gets the full state of the CCOB module.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The full Raft state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the full AuxTest state")
    public CCOBFullState getFullState()
    {
        return new CCOBFullState(getState(), subsys.getMonitor().getFullState());
    }    

    /**
     *  Sets the update period.
     *
     *  @param  value  The update period (milliseconds) to set.
     */
    @Command(type=Command.CommandType.ACTION, description="Set the update interval")
    public void setUpdatePeriod(@Argument(description="The tick period (ms)") int value)
    {
        setTickPeriod(value);
        publishState();
    }

    /**
     *  Gets the ADC values.
     */
    @Command
    public CCOBDataRead getADCValues() throws DriverException {
        return driver.getAdcValues();
    }
    
    /**
     * Turn On the led. 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Turn On the led.")
    public void turnOnLed() {
        led.turnOn();
    }

    /**
     * Turn Off the led. 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Turn Off the led.")   
    public void turnOffLed() {
        led.turnOff();
    }

    /**
     * Open the shutter. 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Open the shutter.")
    public void openShutter() {
        shutter.open();
    }

    /**
     * Close the shutter. 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Close the shutter.")
    public void closeShutter() {
        shutter.close();
    }
 
    /**
     * Take an image. 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Take an image.")
    public void takeImage() {
        
    }
    
    /**
     * Read temperature on the thermometer and returns its values in decimal format.
     * @return temperature
     */
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Read temperature on the gauge and returns its values "
                    + "in decimal format.")
    public double readTemperature()  {
        return this.thermometer.readTemperature();
    }

    /**
     *  Gets the state of the CCOB module.
     *
     *  @return  The CCOB state
     */
    private CCOBState getState() {
        return new CCOBState(getTickPeriod());
    }    

    /**
     *  Publishes the state of the CCOB module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(CCOBState.KEY, getState()));
    }

    /**
     *  Sets the monitoring publishing period
     */
    private void setTickPeriod(long period)
    {
        pts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    
    /**
     *  Gets the monitoring publishing period
     */
    private int getTickPeriod()
    {
        return (int)pts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

}
