package org.lsst.ccs.subsystem.ccob.ui.jas;

import java.awt.BorderLayout;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import javax.swing.JComponent;
import javax.swing.JPanel;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.services.optpage.OptionalPage;
import org.lsst.ccs.gconsole.plugins.monitor.SectionedTableView;
import org.lsst.ccs.gconsole.plugins.monitor.TabbedView;
import org.lsst.ccs.messaging.AgentMessagingLayer;

/**
 * @author tonyj
 */
public class CCOBOptionalPage implements OptionalPage {

    private final CommandSender cmdSender = new CommandSender();
    private final Map<String, Control> controls = new ConcurrentHashMap<>();
    private final String agentProperty;
    private final Class controlClass;

    public CCOBOptionalPage(String agentProperty, Class controlClass) {
        this.agentProperty = agentProperty;
        this.controlClass = controlClass;
    }

    @Override
    public boolean isAutoOpen() {
        return false;
    }

    @Override
    public String getPage(AgentInfo agent) {
        return agent != null && agent.hasAgentProperty(agentProperty)
                 ? getPageName(agent) : null;
    }

    @Override
    public boolean agentDisconnected(AgentInfo agent) {
        //System.out.println("agentDisconnected called");
        Control control = controls.get(getPageName(agent));
        if (control == null) return true;
        control.disableSystem();
        return false;
    }

    @Override
    public JComponent open(AgentInfo agent, JComponent existingComponent) {
        //System.out.println("open called");
        String pageName = getPageName(agent);
        String agentName = agent.getName();
        Control control = controls.get(pageName);
        if (control != null) {
            initControls(agentName, control);
            return existingComponent;
        }

        // Create the control object
        try {
            control = (Control)controlClass.newInstance();
            controls.put(pageName, control);
        }
        catch (Exception e) {
            return null;
        }

        // Create the monitoring panel
        TabbedView view = new TabbedView();
        view.setViewFactory(name -> new SectionedTableView());            
        MonitorFilter monitorFilter = new MonitorFilter(agent.getName());
        view.setFilter(monitorFilter);
        view.install();

        // Create the control panel
        JPanel cPanel = control.getControlPanel(cmdSender);
        control.setSubsystems(agent.getName());

        // Create the whole panel
        JPanel panel = new JPanel();
        panel.setLayout(new java.awt.BorderLayout());
        panel.add(view.getPanel(), BorderLayout.CENTER);
        panel.add(cPanel, BorderLayout.SOUTH);

        // Set up the bus listener
        AgentMessagingLayer aml = Console.getConsole().getMessagingAccess();
        aml.addStatusMessageListener(control.getListener(),
                                     (msg) -> msg.getOriginAgentInfo().getName().equals(agentName)
                                                && msg instanceof StatusSubsystemData);

        // Initialize the control panel
        initControls(agentName, control);
        return panel;
    }

    @Override
    public void closed(String pageName) {
        //System.out.println("closed called");
        Control control = controls.remove(pageName);
        if (control == null) {
            throw new RuntimeException("No page exists with the name " + pageName);
        }
        Console.getConsole().getMessagingAccess().removeStatusMessageListener(control.getListener());
    }

    private String getPageName(AgentInfo agent) {
        return "CCS Subsystems/" + agent.getName();
    }

    private void initControls(String agentName, Control control) {
        control.updateControlPanel(cmdSender.sendCommand(agentName, null, "getFullState"));
    }

}
