package org.lsst.ccs.subsystem.camera.rotator;

import java.util.logging.Logger;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.rotator.RotatorState;
import org.lsst.ccs.drivers.rotator.RotatorState.EnabledState;
import org.lsst.ccs.drivers.rotator.RotatorState.RotatorSummaryState;
import org.lsst.ccs.drivers.rotator.RotatorStateChangedEvent;
import org.lsst.ccs.drivers.rotator.RotatorStateChangeListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.camera.rotator.alerts.CameraRotatorAlerts;
import org.lsst.ccs.subsystem.camera.rotator.states.CameraRotatorState;

/**
 * Receives callback of Rotator State change, sets subsystem state, and
 * raises Alerts if indicated.
 * 
 * @author The CCS Team
 */
public class RotatorSubsystemSetState implements RotatorStateChangeListener {

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private CameraRotatorSubsystem subsys;

    private static final Logger LOG = Logger.getLogger(RotatorSubsystemSetState.class.getName());

    // Prior State values (unknown at startup, use convenient initial values)
    private RotatorSummaryState summaryState = RotatorSummaryState.ENABLED;
    private EnabledState enabledState = EnabledState.STATIONARY;

    @Override
    public void stateChanged(RotatorStateChangedEvent event) {
        RotatorState newState = event.getRotatorState();
        LOG.info("RotatorSubsystem received state change: " + newState.toString());
        RotatorSummaryState newSummary = newState.getRotatorSummaryState();
        EnabledState newEnabled = newState.getEnabledState();

        if (newSummary != summaryState) {
	    boolean fault = (summaryState.equals(RotatorSummaryState.FAULT));
            boolean not_en = !fault && !(summaryState.equals(RotatorSummaryState.ENABLED));
            summaryState = newSummary;
            switch (newSummary) {
	    case  FAULT:
                if (!fault) {
                    alertService.raiseAlert(CameraRotatorAlerts.ROTATOR_FAULT.newAlert(),
                                            AlertState.ALARM, 
                                            "StateChange to FAULT");
                    subsys.publishState(CameraRotatorState.FAULT);
                    if (not_en) {
                        restore (CameraRotatorAlerts.NOT_ENABLED);
                    }
                }
                return;
            case ENABLED:
                if (not_en) {
                    restore (CameraRotatorAlerts.NOT_ENABLED);
                }
                if (fault) {
                    restore (CameraRotatorAlerts.ROTATOR_FAULT);
                }
                break;
            default:             // neither ENABLED nor FAULT
                if (!not_en) {
                    alertService.raiseAlert(CameraRotatorAlerts.NOT_ENABLED.newAlert(),
                                            AlertState.ALARM, 
                                            "StateChange, not enabled");
                    subsys.publishState(CameraRotatorState.NOT_ENABLED);
    	        }
                if (fault) {
                    restore (CameraRotatorAlerts.ROTATOR_FAULT);
                }
                return;
	    }
        }

        /* Next check substate of Enabled state */

        if (!newEnabled.equals(enabledState)) {
            switch (newEnabled) {
            case STATIONARY:
                subsys.publishState(CameraRotatorState.STOPPED);
                subsys.publishData(subsys.getPosition());
                break;
            case MOVING_POINT_TO_POINT:
	    case CONTROLLED_STOP:
                subsys.publishState(CameraRotatorState.ROTATING);
                break;
	    default:
                subsys.publishState(CameraRotatorState.OTHER);
	    }
            enabledState = newEnabled;
        }
	return;
    }

    /* Method to restore an ALARM to NOMINAL */

    private void restore(CameraRotatorAlerts alert) {
        alertService.raiseAlert(alert.newAlert(), AlertState.NOMINAL, 
                                "StateChange, reset Alert");
    }

}
