package org.lsst.ccs.subsystem.camera.rotator;

import java.util.concurrent.TimeoutException;
import java.util.logging.Logger;
import java.util.Map;
import java.util.Set;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.rotator.Position;
import org.lsst.ccs.drivers.rotator.RotatorDriver;
import org.lsst.ccs.drivers.rotator.RotatorState;
import org.lsst.ccs.drivers.rotator.RotatorStateChangedEvent;
import org.lsst.ccs.drivers.rotator.RotatorStateChangeListener;
import org.lsst.ccs.drivers.commons.DriverException;
import static org.lsst.ccs.command.annotations.Command.CommandType.ACTION;
import static org.lsst.ccs.command.annotations.Command.CommandType.QUERY;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING1;
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.subsystem.camera.rotator.alerts.CameraRotatorAlerts;
import org.lsst.ccs.subsystem.camera.rotator.data.CameraRotatorData;
import org.lsst.ccs.subsystem.camera.rotator.states.CameraRotatorState;

/**
 * A subsystem for controlling the camera rotator.
 * 
 * @author The CCS Team
 */
public class CameraRotatorSubsystem extends Subsystem implements HasLifecycle, ConfigurationBulkChangeHandler {

    private static final Logger LOG = Logger.getLogger(CameraRotatorSubsystem.class.getName());
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    DataProviderDictionaryService dictionaryService;

    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    RotatorSubsystemSetState setState;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    RotatorSubsystemPosition posListen;

    @ConfigurationParameter(description = "maximum rotator acceleration",
                            range = "0.0..1.0", units = "deg/s**2")
    protected volatile double maxAcceleration;

    @ConfigurationParameter(description = "maximu rotator velocity",
                            range = "0.0..3.5", units = "deg/s")
    protected volatile double maxVelocity;

    private final RotatorDriver driver = new RotatorDriver();
    private boolean initialized = false;
    private static final String dataKey = "positionData";
    private CameraRotatorData lastData;
    
    /**
     *  Constructor
     */
    public CameraRotatorSubsystem() {
        super("camera-rotator", AgentInfo.AgentType.WORKER);
    }

    @Override
    public void init() {
        agentStateService.registerState(CameraRotatorState.class, 
                                        "State of the camera rotator", this); 
        dictionaryService.registerClass(CameraRotatorData.class, dataKey);
        writeConfigurableSettings();
        initialized = true;
    }

    /*
     * Before starting, execute "Stop" to put subsystem in a known
     * initial state.  An exception will cause subsystem to fail to start.
     */
    @Override
    public void start() {
        driver.addStateChangeListener(setState);
        driver.addPositionListener(posListen);
        try {
                driver.stop();
        } catch (DriverException|TimeoutException ex) {
	    throw new RuntimeException(name + ": Driver or Timeout Exception on initializing Stop, " + ex);
        }
        publishState(CameraRotatorState.STOPPED);
    }

    /**
     *  Publish subsystem state-change data
     */
    void publishState(CameraRotatorState state) {
        stateService.updateAgentComponentState(this, state);
        LOG.info(name + ": state set to " + state.toString());
    }

   /**
    *  Publiwh trending data
    */
    void publishData(Position position) {
        CameraRotatorData data = new CameraRotatorData(position.getPosition(),
                                                       position.getError());
        publishSubsystemDataOnStatusBus(new KeyValueData(dataKey, data));
        lastData = data;
    }

   /**
    *  Checks validity of a proposed set of ConfigurationParameters
    *
    *  @param Map of ConfigurationParameters from name to value
    */
    @Override
    public void validateBulkChange(Map<String,Object> params) {
        Set<String> keys = params.keySet();
        // Potential use to impose requirements on set of parameters
    }

   /**
    *  Set configuration parameters to new values.
    *  Checks input Map for those ConfigurationParameters which correspond
    *  to hardware settings; the others are left for framework to deal with.
    *  A DriverException occuring while writing to hardware is
    * rethrown as a RuntimeException.
    *
    *  @param  Map of parameters for which a change is requested,
    *          keyed by names
    */
    @Override
    public void setParameterBulk(Map<String,Object> params)  {
        Set<String> keys = params.keySet();
        if (keys.contains("maxVelocity")) {
            maxVelocity = (double) params.get("maxVelocity");
        }
        if (keys.contains("maxAcceleration")) {
            maxAcceleration = (double) params.get("maxAcceleration");
        }
        if (initialized) {
            writeConfigurableSettings();
        }
    }

   /**
    * boolean isRotating() is true if state is ROTATING
    */
    boolean isRotating() {
        return agentStateService.isInState(CameraRotatorState.ROTATING);
    }

   /**
    *  Applies hardware (EUI) settings for all configuration parameters.
    *  Parameters are assumed to have been validatad.
    *  A Driver or Timeout Exception results ina RuntimeException
    */
    private void writeConfigurableSettings() {
        try {
            driver.configureVelocity(maxVelocity);
            driver.configureAcceleration(maxAcceleration);
        } catch (DriverException|TimeoutException ex) {
            throw new RuntimeException(name + ": Driver or Timeout Exception during Configure of EUI:  " + ex);
        }
    }

    /* Commands */

   /**
    * Rotate camera to specified absolute position in degrees.
    * Before move, publish most recent position data.
    * Any Exception is caught, and subsystem put into fault state.
    *
    * @param  double position (in degreed)
    */
    @Command(type = ACTION, level = NORMAL, name = "rotate",
             description = "Rotate the camera", autoAck = false)
    public void rotate(@Argument(name="position", description="absolute rotator position in degrees") double position) {
        helper()
        .precondition(agentStateService.isInState(CameraRotatorState.STOPPED), "Shutter is not stopped")
        .enterFaultOnException(true)
        .action(() -> {
	    Position pos = getPosition();
            if (pos != null) {
                LOG.info("Last position as of rotate command: "+pos.toString());
                publishData(pos);
            }
            driver.move(position);
	});
    }

   /**
    * Stoo any rotation in progress
    */
    @Command(type = ACTION, level = NORMAL, name = "stopRotation",
             description = "Stop the rotator", autoAck = false)
    public void stopRotation() {
        helper()
        .enterFaultOnException(true)
        .action(() -> {
            driver.stop();
        });
    }

   /**
    *  Get RotatorState as of mos-recent rotator event
    *
    *  @return RotatorState
    */
    @Command(type = QUERY, level = NORMAL, name = "getRotatorState",
             description = "return RotatorState")
    public RotatorState getRotatorState() {
        return driver.getRotatorState();
    }

   /**
    *  Show/print RotatorState
    *
    *  @return String
    */
    @Command(type = QUERY, level = NORMAL, name = "showRotatorState", 
             description = "return RotatorState as String")
    public String showRotatorState() {
        return getRotatorState().toString();
    }

   /**
    *  Get Rotator position information
    *
    *  @return Position  rotator position information
    */
    @Command(type = QUERY, level = NORMAL, name = "getPosition", 
             description = "get Rotator Position object")
    public Position getPosition() {
        return driver.getPosition();
    }

   /**
    *  Show Rotator position information
    *
    *  @return String position and requested position in degrees
    */
    @Command(type = QUERY, level = NORMAL, name = "showPosition", 
             description = "show Rotator position information")
    public String showPosition() {
        return getPosition().toString();
    }

    /**
     * Enable Rotator
     *
     * @throws DriverException, TimeoutException
     */
    @Command(type = ACTION, level = ENGINEERING1, name = "enable", 
             description = "Enable Rotator")
    public void enable() throws DriverException, TimeoutException {
        driver.enable();
    }

    /**
     * Disable Rotator
     *
     * @throws DriverException, TimeoutException
     */
    @Command(type = ACTION, level = ENGINEERING1, name = "disable", 
            description = "Disable Rotator")
    public void disable() throws DriverException, TimeoutException {
        driver.disable();
    }

}
