package org.lsst.ccs.subsystem.cablerotator;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.motorplatform.bus.AxisStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;

/**
 * Performs operations specific to a particular axis.
 * @author tether
 */
public class Axis {

    private enum Status {
        ENABLED(0), HOMED(1), MOVING(3), BRAKED(8), HOMING(14), CW_EOT(22), CCW_EOT(23), ESTOP(31);
        private final long mask;
        Status(int bitPos) {this.mask = 1L << bitPos;}
        public boolean isSetIn(long statusMask) {return 0 != (statusMask & mask);}
    }

    private enum Fault {
        POSITION_ERROR(0, "Position error"),
        OVER_CURRENT(1, "Too much current"),
        HARD_EOT_HIGH(2, "Hard EOT high"),
        HARD_EOT_LOW(3, "Hard EOT low"),
        SOFT_EOT_HIGH(4, "Soft EOT high"),
        SOFT_EOT_LOW(5, "Soft EOT low"),
        AMPLIFIER(6, "Amplifier"),
        POSITION_FEEDBACK(7, "Position feedback"),
        VELOCITY_FEEDBACK(8, "Velocity feedback"),
        HALL(9, "Hall sensor"),
        MAX_VELOCITY(10, "Maximum velocity"),
        ESTOP(11, "ESTOP"),
        VELOCITY_ERROR(12, "Velocity error"),
        EXTERNAL(15, "External"),
        MOTOR_TEMP(17, "Motor temperature"),
        AMP_TEMP(18, "Amplifier temperature"),
        ENCODER(19, "Encoder"),
        COMM_LOST(20, "Lost comm"),
        FEEDBACK_SCALING(23, "Feedback scaling"),
        MARK_SEARCH(24, "Mark search"),
        VOLTAGE_CLAMP(27, "Voltage clamp"),
        POWER_SUPPLY(28, "Power supply");
        private final long mask;
        private final String text;
        Fault(int bitPos, String text) {this.mask = 1L << bitPos; this.text = text;}
        public boolean isSetIn(long faultMask) {return 0 != (faultMask & mask);}
        @Override public String toString() {return text;}
    }

    @ConfigurationParameter(description="The axis name in the Aerotech controller.", isFinal=true)
    private volatile String axisName;

    @ConfigurationParameter(description="The axis index in the Aerotech controller.", isFinal=true)
    private volatile int index;

    @ConfigurationParameter(description="The axis position units.", isFinal=true)
    private volatile String units;

    @ConfigurationParameter(description="The speed limit in axis units/sec.", isFinal=true)
    private volatile double maxSpeed;

    @ConfigurationParameter(description="The furthest the axis can travel, end to end, in axis units.", isFinal=true)
    private volatile double maxTravel;

    /**
     * Constructs a proxy object for one axis.
     */
    public Axis() {}


    public String getName() {
        return axisName;
    }

    public int getIndex() {
        return index;
    }

    public String getUnits() {
        return units;
    }

    public double getMaxSpeed() {
        return maxSpeed;
    }

    public double getMaxTravel() {
        return maxTravel;
    }

    public boolean isAtHome(final Aerotech aero) throws DriverException {
        return Status.HOMED.isSetIn(getStatusMask(aero));
    }

    /**
     * Produces a status report for the axis.
     * @param aero Used to communicate with the motor controller.
     * @return  The report.
     */
    public AxisStatus getStatus(final Aerotech aero) throws DriverException {
        final long statusMask = getStatusMask(aero);
        final long faultMask = getFaultMask(aero);
        final boolean enabled = Status.ENABLED.isSetIn(statusMask);
        final boolean moving = Status.MOVING.isSetIn(statusMask) || Status.HOMING.isSetIn(statusMask);
        final boolean atLow = Fault.HARD_EOT_LOW.isSetIn(faultMask);
        final boolean atHigh = Fault.HARD_EOT_HIGH.isSetIn(faultMask);
        final boolean homed = Status.HOMED.isSetIn(statusMask);
        final double position = getPosition(aero);
        final List<String> faults = getFaultList(statusMask, faultMask);
        final AxisStatus status = new AxisStatus
                (axisName, enabled, moving, homed, atLow, atHigh, faults, position);
        return status;
    }

    private double getPosition(final Aerotech aero) throws DriverException {
        final Aerotech.Reply reply = aero.readCalibratedPosition(getName());
        return reply.asDouble();
    }

    private List<String> getFaultList(final long statusMask, final long faultMask) {
        final List<String> faults = new ArrayList<>();
        faults.addAll(
            Arrays
            .stream(Fault.values())
            .filter(f -> f.isSetIn(faultMask))
            .map(Fault::toString)
            .collect(Collectors.toList()));
        return faults;
   }

    private long getStatusMask(final Aerotech aero) throws DriverException {
        final Aerotech.Reply reply = aero.readAxisStatus(getName());
        return reply.asInt();
    }

    private long getFaultMask(final Aerotech aero) throws DriverException {
        final Aerotech.Reply reply = aero.readAxisFaults(getName());
        return reply.asInt();
    }

    public void moveRelative(final Aerotech aero, final MoveAxisRelative cmd) throws DriverException {
        final double d = cmd.getDistance();
        final double t = cmd.getTime().toMillis() / 1000.0;
        // Allow 100 msec for acceleration and another for deceleration.
        double v = Math.abs(d) / (t - 0.2);
        // Adjust top speed for safety.
        v = Math.min(v, maxSpeed);
        // Accelerate/decelerate in 100 msec.
        final double a = 10.0 * v;
        // Set parameters for motion then start the motion program.
        aero.writeDglobal(0, index);
        aero.writeDglobal(1, d);
        aero.writeDglobal(2, v);
        aero.writeDglobal(3, a);
        aero.writeDglobal(4, Math.abs(d)/v + 0.2);
        aero.writeDglobal(5, 0); // Relative motion.
        aero.startProgram(1, "Move.bcx");
    }

    // This is the time this subsystem should wait for the controller to become idle again.
    public Duration getRelativeMoveTimeout(final MoveAxisRelative cmd) {
        final double d = Math.abs(cmd.getDistance());
        final double t = cmd.getTime().toMillis() / 1000.0;
        // Allow 100 msec for acceleration and another for deceleration.
        double v = d / (t - 0.2);
        // Adjust top speed for safety.
        v = Math.min(v, maxSpeed);
        // Adjust timeout based on limited value of v.
        return Duration.ofMillis(Math.round(1e3*(d/v + 0.2)));
    }

    public void moveAbsolute(final Aerotech aero, final MoveAxisAbsolute cmd) throws DriverException {
        // Since we don't know how far the axis has to go, we'll have to wait
        // long enough for the longest possible move to the target to finish.
        final double d = cmd.getPosition();
        final double v = Math.min(cmd.getSpeed(), maxSpeed);
        // Accelerate/decelerate in 100 msec.
        final double a = 10.0 * v;
        // How long to wait for motion to complete.
        final double t = getAbsoluteMoveTimeout(cmd).toMillis() / 1000.0;
        // Set parameters for motion then start the motion program.
        aero.writeDglobal(0, index);
        aero.writeDglobal(1, d);
        aero.writeDglobal(2, v);
        aero.writeDglobal(3, a);
        aero.writeDglobal(4, t);
        aero.writeDglobal(5, 1); // Absolute motion.
        aero.startProgram(1, "Move.bcx");
    }

    // This is the time this subsystem should wait for the controller to become idle again.
    public Duration getAbsoluteMoveTimeout(final MoveAxisAbsolute cmd) {
        final double d = Math.max(cmd.getPosition(), maxTravel - cmd.getPosition());
        final double v = Math.min(cmd.getSpeed(), maxSpeed);
        return Duration.ofMillis( Math.round(1e3*(d/v + 2.0)) );
    }

    public int getMoveCompletionCode(final Aerotech aero) throws DriverException {
        return (int)aero.readDglobal(0).asDouble();
    }

    public void clearFaults(final Aerotech aero) throws DriverException {
        aero.clearAxisFaults(getName());
    }

    public boolean hasFaults(final Aerotech aero) throws DriverException {
        return 0L != getFaultMask(aero);
    }

    public Duration getHomingTimeout() {
        return Duration.ofMinutes(5); // Arbitrary.
    }

    public int getHomingCompletionCode(final Aerotech aero) {
        return 0;
    }

    public void home(final Aerotech aero) throws DriverException {
        // Note that the HOME command doesn't respect wait mode,
        // it always blocks until the operation finishes. we can make it
        // asynchrous by running the actual command in an AeroBasic program.
        aero.writeIglobal(32, 1); // No. of axis indexes to follow in IGLOBAL.
        aero.writeIglobal(33, getIndex());
        aero.startProgram(1, "HomeAsync.bcx");
    }

    public void enable(final Aerotech aero) throws DriverException {
        aero.enableAxes(getName());
    }

    public void disable(final Aerotech aero) throws DriverException {
        aero.disableAxes(getName());
    }

    public boolean hasESTOP(final Aerotech aero) throws DriverException {
        return Fault.ESTOP.isSetIn(getFaultMask(aero));
    }

}