package org.lsst.ccs.subsystem.cablerotator;

import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RunMode;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.motorplatform.bus.AxisStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeAxisEnable;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAllFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAxisFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ControllerStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.DisableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.EnableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.HomeAxis;
import org.lsst.ccs.subsystem.motorplatform.bus.MotorCommandListener;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.motorplatform.bus.PlatformConfig;
import org.lsst.ccs.subsystem.motorplatform.bus.SendAxisStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.SendConfiguration;
import org.lsst.ccs.subsystem.motorplatform.bus.SendControllerStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.StopAllMotion;
/**
 * Communicates with the Aerotech motor controller, publishes status and configuration.
 *
 * @author tether
 *
 */
public class Controller implements HasLifecycle, HardwareController, MotorCommandListener {

    private static final Logger LOG = Logger.getLogger(Controller.class.getName());

    // Contact the controller at this port.
    private static final int AEROTECH_TCP_PORT = 8000;

    // The Move controller program runs in this controller task.
    private static final int AEROTECH_MOTION_TASK_NUMBER = 1;

    ////////// References to other components //////////

    @LookupField(strategy=LookupField.Strategy.TREE)
    private Subsystem subsys;

    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    private volatile Axis axisComponent;

    ////////// Configuration parameters //////////
    @ConfigurationParameter(description="The name of the configuration used.", isFinal=true)
    private volatile String configurationName = "<none>";

    @ConfigurationParameter(
        description="How long to wait for the Aerotech controller to responds to a command.", isFinal=true)
    private volatile Duration commandTimeout = Duration.ofSeconds(10);

    @ConfigurationParameter(description="The IP address of the motor controller.", isFinal=true)
    private volatile String IPv4Address = "192.168.10.40";

    ////////// Other fields //////////
    // Link to the controller.
    private volatile Aerotech aeroLink;

    // Latest controller status.
    private volatile ControllerStatus controllerStat;

    // Latest axis status.
    private volatile AxisStatus axisStat;

    // Motorplatform configuration info.
    private volatile PlatformConfig motorConfig;

    // Is the controller link open?
    private volatile ConnectionStatus connectionStat;

    // GUARDED by the instance lock.
    // END GUARDED

    // GUARDED by the constructor.
    // END GUARDED

    ////////// Lifecycle methods //////////

    public Controller() {
        // Set initial values for messages that will go out on the status bus since we connect to
        // the buses a little before checkHardware() is called so they might be sent
        // before we contact the Aerotech controller.
        motorConfig = new PlatformConfig(
            Arrays.asList("unknown"),  // Axis name is a configuration parameter.
            Collections.emptyList(), // No known capturing in effect.
            Collections.emptyList(), // No known digital inputs.
            Collections.emptyList(), // No known digital outputs.
            Collections.emptyList(), // No known analog inputs.
            "unknown",
            Arrays.asList("unknown") // Axis position units.
        );
        connectionStat = ConnectionStatus.OFFLINE;
        controllerStat = new ControllerStatus("DOWN", false, "unknown");
        axisStat = new AxisStatus("unknown", false, false, false, false, false, Arrays.asList("Controller not yet contacted"), -9999.0);
    }

    @Override
    public void build() {
       final AgentPeriodicTaskService tasksrv = subsys.getAgentService(AgentPeriodicTaskService.class);
       tasksrv.createScheduler("statusReader", 1);
       final AgentPeriodicTask task = new AgentPeriodicTask("readerTask", this::updateStatus)
           .withIsFixedRate(false);
       tasksrv.scheduleAgentPeriodicTask(task, "statusReader");
    }

    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        // Now that the safe configuration is loaded we can construct the right motorplatform configuration message.
        motorConfig = new PlatformConfig(
            Arrays.asList(axisComponent.getName()),
            motorConfig.getCapture(),
            motorConfig.getDigitalInputNames(),
            motorConfig.getDigitalOutputNames(),
            motorConfig.getAnalogInputDescriptions(),
            configurationName,
            Arrays.asList(axisComponent.getUnits())
        );
        try {
            contactController();
        }
        catch (DriverException exc) {
            throw new HardwareException("Can't contact the Aerotech controller.", exc, null);
        }
        return TreeWalkerDiag.GO;
    }

    ////////// End of lifecycle methods //////////

    ////////// Implementation of MotorCommandListener //////////

    @Override
    public void stopAllMotion(StopAllMotion cmd) {
        runAerotechCommands( () -> {
                aeroLink.stopProgram(AEROTECH_MOTION_TASK_NUMBER);
                aeroLink.abortAxes(axisComponent.getName());
            }
        );
    }

    @Override
    public void sendControllerStatus(SendControllerStatus cmd) {
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData("ControllerStatus", controllerStat));
    }

    @Override
    public void sendConfiguration(SendConfiguration cmd) {
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData("PlatformConfig", motorConfig));
    }

    @Override
    public void sendAxisStatus(SendAxisStatus cmd) {
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData("AxisStatus", axisStat));
    }

    @Override
    public void homeAxis(HomeAxis cmd) {
        final Axis axis = getAxis(cmd.getAxisName());
        runAerotechCommands( () -> {
                axis.home(aeroLink);
            }
        );
    }

    @Override
    public void moveAxisAbsolute(MoveAxisAbsolute cmd) {
        final Axis axis = getAxis(cmd.getAxisName());
        runAerotechCommands( () -> {
                    axis.moveAbsolute(aeroLink, cmd);
                }
        );
    }

    @Override
    public void moveAxisRelative(MoveAxisRelative cmd) {
        final Axis axis = getAxis(cmd.getAxisName());
        runAerotechCommands( () -> {
                    axis.moveRelative(aeroLink, cmd);
                }
        );
    }

    @Override
    public void disableAllAxes(DisableAllAxes cmd) {
        runAerotechCommands( () -> {
                axisComponent.disable(aeroLink);
            }
        );
    }

    @Override
    public void enableAllAxes(EnableAllAxes cmd) {
        runAerotechCommands( () -> {
                axisComponent.enable(aeroLink);
            }
        );
    }

    @Override
    public void clearAxisFaults(ClearAxisFaults cmd) {
        final Axis axis = getAxis(cmd.getAxisName());
        runAerotechCommands( () -> {
                axis.clearFaults(aeroLink);
            }
        );
    }

    @Override
    public void clearAllFaults(ClearAllFaults cmd) {
        runAerotechCommands( () -> {
                aeroLink.clearAllFaults();
            }
        );
    }

    @Override
    public void changeAxisEnable(ChangeAxisEnable cmd) {
        final Axis axis = getAxis(cmd.getAxisName());
        runAerotechCommands( () -> {
                if (cmd.isEnabled()) {axis.enable(aeroLink);}
                else {axis.disable(aeroLink);}
            }
        );
    }

    AxisStatus getAxisStatus() {
        return axisStat;
    }

    ////////// Utility  //////////

    // Like Callable but throws DriverException instead of the generic Exception.
    @FunctionalInterface
    private interface DriverCallable {
        void call() throws DriverException;
    }

    private final static Alert ROTATOR_ALERT = new Alert("CABLE_ROTATOR", "Trouble with the cable rotator.");

    // Catch DriverException and wrap it in an unchecked exception that can be thrown by the methods
    // overriding those in MotorCommandListener. All Aerotech commands should come through here
    // so that one thread's command sequence isn't interrupted by another thread's.
    private synchronized void runAerotechCommands(final DriverCallable cmds) {
        try {
            contactController();
            cmds.call();
            controllerStat = new ControllerStatus("UP", controllerStat.isMotionEnabled(), "OK");
        }
        catch (Aerotech.CommandFailureException exc) {
            LOG.log(Level.SEVERE, "Aerotech command failed.", exc);
            subsys.getAgentService(AlertService.class)
                .raiseAlert(ROTATOR_ALERT, AlertState.ALARM, exc.getMessage());
            controllerStat = new ControllerStatus(
                connectionStat == ConnectionStatus.ONLINE ? "UP" : "DOWN",
                controllerStat.isMotionEnabled(),
                "FAILED");
            LOG.severe(String.format("The failing command: %s", exc.reply.command));
            LOG.severe(String.format("The class of failure: %s", exc.reply.status));
            throw new RuntimeException(exc);
        }
        catch (DriverException exc) {
            LOG.log(Level.SEVERE, "ASCII driver exception.", exc);
            subsys.getAgentService(AlertService.class)
                .raiseAlert(ROTATOR_ALERT, AlertState.ALARM, exc.getMessage());
            controllerStat = new ControllerStatus(
                connectionStat == ConnectionStatus.ONLINE ? "UP" : "DOWN",
                controllerStat.isMotionEnabled(),
                "FAILED");
            throw new RuntimeException(exc);
        }
        finally {
            if (!aeroLink.isOpen()) {
                aeroLink.close();
                connectionStat = ConnectionStatus.OFFLINE;
            }
        }
    }

    private Axis getAxis(final String axisName) {
        assert(axisComponent != null);
        assert(axisName != null);
        if (!axisName.equals(axisComponent.getName())) {
            throw new IllegalArgumentException(String.format("Axis name '%s' should be '%s'.", axisName, axisComponent.getName()));
        }
        return axisComponent;
    }

    // Attempt to update the ControllerStatus and AxisStatus messages and then send them out on the status bus.
    public void updateStatus() {
        runAerotechCommands( () -> {
                final AxisStatus axstat = axisComponent.getStatus(aeroLink);
                final boolean estop = axisComponent.hasESTOP(aeroLink);
                axisStat = axstat;
                // The new controller status will reflect the status of the last command from the outside, not the status query.
                controllerStat = new ControllerStatus("UP", !estop, controllerStat.getLastCommandStatus());
            }
        );
        sendAxisStatus(new SendAxisStatus(axisComponent.getName()));
        sendControllerStatus(new SendControllerStatus());
    }


    private static enum ConnectionStatus {
        ONLINE, OFFLINE;
    }

    private void contactController() throws DriverException {
        if (connectionStat == ConnectionStatus.OFFLINE) {
            if (aeroLink != null) {aeroLink.close();}
            // Contact and set up the Aerotech controller, using a simple simulated controller if in simulation mode.
            final Ascii link = RunMode.isSimulation() ? new MockAscii() : new Ascii();
            link.setTimeout(commandTimeout.toMillis());
            link.openNet(IPv4Address, AEROTECH_TCP_PORT);
            aeroLink = new Aerotech(link, Aerotech.ErrorHandling.THROW_EXCEPTIONS);
            aeroLink.setWaitMode(Aerotech.WaitMode.NOWAIT);
            connectionStat = ConnectionStatus.ONLINE;
            LOG.info("Connected to the Aerotech controller.");
        }
    }
}
