package org.lsst.ccs.subsystem.cablerotator;

import java.time.Duration;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.command.annotations.Command.CommandType.ACTION;
import static org.lsst.ccs.command.annotations.Command.CommandType.QUERY;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING1;
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.motorplatform.bus.AxisStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeAxisEnable;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAllFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAxisFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.DisableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.EnableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.HomeAxis;
import org.lsst.ccs.subsystem.motorplatform.bus.MotorCommandListener;
import org.lsst.ccs.subsystem.motorplatform.bus.MotorplatformType;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.motorplatform.bus.SendAxisStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.SendConfiguration;
import org.lsst.ccs.subsystem.motorplatform.bus.SendControllerStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.ShellCommandListener;
import org.lsst.ccs.subsystem.motorplatform.bus.StopAllMotion;

/**
 * Commands subsystem component, accepts commands.
 *
 * @author tether
 */
public class Commands  extends Subsystem implements HasLifecycle, MotorCommandListener, ShellCommandListener {
    private static final Logger LOG = Logger.getLogger(Commands.class.getName());

    private final static int COMMAND_TIMEOUT = 30; // seconds

    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Subsystem subsys;

    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Controller controller;

    public Commands() {
        super("commands", AgentInfo.AgentType.WORKER);
    }

    ////////// Implement HasLifecycle //////////

    /**
     * Sets the Agent property marking this as a motorplatform subsystem.
     * Must be done before we connect to the busses.
     */
    @Override
    public void postInit() {
        subsys
            .getAgentService(AgentPropertiesService.class)
            .setAgentProperty(
                MotorplatformType.MOTORPLATFORM_TYPE_AGENT_PROPERTY,
                "CABLE_ROTATOR");
    }

    ////////// End HasLifecycle //////////

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Start an axis homing.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void homeAxis(final HomeAxis req) {
        controller.homeAxis(req);
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION,
        description="(Scripting) Start a move to an absolute position.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void moveAxisAbsolute(final MoveAxisAbsolute req) {
        controller.moveAxisAbsolute(req);
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION,
        description="(Scripting) Start a position change.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void moveAxisRelative(final MoveAxisRelative req) {
        controller.moveAxisRelative(req);
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION,
        description="(Scripting) Clear faults on both axes.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void clearAllFaults(final ClearAllFaults req) {
        controller.clearAllFaults(req);
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Enable/disable axis.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void changeAxisEnable(final ChangeAxisEnable req) {
        controller.changeAxisEnable(req);
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION,
        description="(Scripting) Clear faults for one axis.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void clearAxisFaults(final ClearAxisFaults req) {
        controller.clearAxisFaults(req);
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Enable both axes.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void enableAllAxes(final EnableAllAxes req) {
        controller.enableAllAxes(req);
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Disable both axes.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void disableAllAxes(final DisableAllAxes req) {
        controller.disableAllAxes(req);
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Emergency stop motion.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void stopAllMotion(final StopAllMotion req) {
        controller.stopAllMotion(req);
    }

    @Command(level= NORMAL, type=QUERY, description="(Scripting) Publish controller status ASAP.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void sendControllerStatus(SendControllerStatus cmd) {
        controller.sendControllerStatus(cmd);
    }

    @Command(level=NORMAL, type=QUERY, description="(Scripting) Publish controller configuration ASAP.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void sendConfiguration(SendConfiguration cmd) {
        controller.sendConfiguration(cmd);
    }

    @Command(level=NORMAL, type=QUERY, description="(Scripting) Publish axis status ASAP.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void sendAxisStatus(SendAxisStatus cmd) {
        controller.sendAxisStatus(cmd);
    }

    ////////// End of MotorCommandListener implementation //////////

    ////////// ShellCommandListener implementation //////////
    // Motorplatform command aliases meant for manual use from consoles.

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION,
        description="(Console) Clear faults on both axes.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void clearFaults() {
        clearAllFaults(new ClearAllFaults());
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Console) Emergeny stop motion.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void stopAll() {
        stopAllMotion(new StopAllMotion());
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Console) Enable an axis.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void enable(String axisName) {
        changeAxisEnable(new ChangeAxisEnable(axisName, true));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Console) Disable an axis.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void disable(String axisName) {
        changeAxisEnable(new ChangeAxisEnable(axisName, false));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Console) Start an axis homing.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void home(String axisName) {
        homeAxis(new HomeAxis(axisName));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION,
        description="(Console) Start a move to an absolute position.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void moveTo(String axisName, double newPositionMm, double speedMmSec) {
        moveAxisAbsolute(new MoveAxisAbsolute(axisName, newPositionMm, speedMmSec));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION,
        description="(Console) Start a position change.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public void moveBy(String axisName, double positionChangeMm, double seconds) {
        moveAxisRelative(new MoveAxisRelative(axisName, positionChangeMm,
            Duration.ofMillis((long)Math.rint(seconds * 1e3))));
    }

    @Command(level=ENGINEERING1, type=QUERY, description="(Console) Show status info for an axis.",
        timeout=COMMAND_TIMEOUT)
    @Override
    public String status(String axisName) {
        final StringBuilder buff = new StringBuilder(1024);
        buff.append("Axis %s%n");
        buff.append("Enabled %s, Moving %s,  Is Homed %s, At Low %s, At High %s%n");
        buff.append("Position %6.1f%n");
        final AxisStatus axstat = controller.getAxisStatus();
        if (axstat == null) {
            return "Not available at present.";
        }
        for (String fault: axstat.getFaults()) {
            buff.append(fault);
            buff.append("%n");
        }
        return String.format(buff.toString(),
                axstat.getAxisName(),
                axstat.isEnabled(),
                axstat.isMoving(),
                axstat.isAtHome(),
                axstat.isAtLowLimit(),
                axstat.isAtHighLimit(),
                axstat.getPosition());
    }

    ////////// End of ShellCommandListener implementation //////////

}
