package org.lsst.ccs.subsystem.cablerotator;
import java.util.Arrays;
import java.util.stream.Collectors;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Uses the Ascii driver to send commands to an Aerotech Ensemble controller and receive
 * replies. Every method, if it doesn't throw an exception, returns a reply object with information
 * on the command, the text of the controller's reply (less prefix) and information about what if anything
 * went wrong. For command failures the driver can either throw a CommandFailureException or log the error
 * and return the reply object. Which is done depends on a constructor argument. Exceptions from the Ascii
 * driver are not caught.
 * <p>
 * This class is immutable. Thread safety depends on the safety of the Ascii driver.
 * @see CommandFailureException
 * @author tether
 */
public class Aerotech {
    private static final Logger LOG = Logger.getLogger(Aerotech.class.getName());

    /**
     * Gets thrown when a command is received by the controller but fails for some reason such as
     * invalid command parameters, syntax error, execution fault, etc.
     */
    public static class CommandFailureException extends DriverException {
        /** The reply that would have been returned by the driver call. */
        public final Reply reply;
        CommandFailureException(final String msg, final Reply reply) {
        super(msg);
        this.reply = reply;
    }
}

    /** The result of examining the controller's reply to a command. */
    public static enum Status {
        /** The command worked. */
        SUCCESS("Success"),
        /** The command was invalid, e.g., had wrong parameter values or a wrong syntax. */
        INVALID_COMMAND("Invalid command"),
        /** Attempting to execute the command resulted in a fault. */
        COMMAND_FAULTED("Command caused a fault"),
        /** Too much delay between two successive characters of a command. */
        COMMAND_TIMEOUT("Command reception timeout"),
        /** No reply from controller. */
        MISSING_REPOSNSE("Missing reply string"),
        /** The controller reply is an empty string or begins with an invalid prefix character. */
        INVALID_PREFIX("Reply prefix character is missing or invalid");

        /** The error message corresponding to the enum element. */
        public final String message;
        private Status(final String message) {this.message = message;}
    }

    /**
     * How the driver is supposed to handle command failure errors.
     */
    public static enum ErrorHandling {
        LOG_ERRORS,
        THROW_EXCEPTIONS;
    }

    /**
     * Information on the command issued and the reply from the controller.
     */
    public static class Reply {
        /** The status of the attempted execution of the command. */
        public Status status;

        /** The reply text stripped of the prefix and the newline; never null but may
         be empty.
         */
        public final String text;

        /** The command that generated this reply. */
        public final String command;

        /**
         * Converts the text string to a double.
         * @return the result of Double.parseDouble(), or NaN.
         */
        public double asDouble() {
            double result = Double.NaN;
            try {
                if (!text.equals("")) result = Double.parseDouble(text);
            }
            catch (NumberFormatException exc) {
                LOG.warning(
                    String.format("Aerotech error. Can't convert the reply text '%s' to double.", text));
            }
            return result;
        }

        /**
         * Converts the text string to an int.
         * @return the result of Integer.parseInt(), or Integer.MAX_VALUE.
         */
        public int asInt() {
            int result = Integer.MAX_VALUE;
            try {
                if (!text.equals("")) result = Integer.parseInt(text);
            }
            catch (NumberFormatException exc) {
                LOG.warning(
                    String.format("Aerotech error. Can't convert the reply text '%s' to int.", text));
            }
            return result;
        }

        Reply(final Status status, final String text, final String command) {
            this.status = status;
            this.text = text;
            this.command = command;
        }
    }

    // Every text begins with a one-character stat prefix. We assume that the default
    // set of characters is in use.
    private static final String STATUS_PREFIXES = "%!#$";

    final private Ascii driver;
    final private ErrorHandling errorMode;

    /**
     * Saves the Ascii driver instance and the error handling mode.
     * @param driver The Ascii driver instance.
     * @param errorMode How to handle Aerotech command failures.
     */
    public Aerotech(final Ascii driver, final ErrorHandling errorMode) {
        this.driver = driver;
        this.errorMode = errorMode;
    }

    /**
     * Attempts to close the driver.
     */
    public void close() {
        driver.closeSilent();
    }

    /**
     * Is the underlying ASCII driver open?
     * @return true if the driver is open, else false.
     */
    public boolean isOpen() {
        return driver.isOpen();
    }

    /**
     * Sends a command to the device and returns the response string with no post-processing.
     * @param cmd The command to send.
     * @return The response from the controller, including prefix.
     * @throws DriverException
     */
    public String sendCommandRaw(final String cmd) throws DriverException {
        return driver.read(cmd);
    }

    /**
     * Sets the timeout for replies from the controller.
     * @param millis The timeout in milliseconds.
     * @throws DriverException
     */
    public void setReplyTimeout(final int millis) throws DriverException {
        driver.setTimeout(millis);
    }

    /**
     * Sends a command to the controller, receives and examines its reply.
     * @param cmd The command to send.
     * @return The Reply object.
     * @throws DriverException something went wrong with command execution.
     */
    public Reply sendCommand(final String cmd) throws DriverException {
        LOG.fine(cmd);
        String text = driver.read(cmd);
        Status stat = Status.SUCCESS;
        if (text == null) {
            stat = Status.MISSING_REPOSNSE;
        }
        else if (text.length() < 1) {
            stat = Status.INVALID_PREFIX;
        }
        else {
            final String prefix = text.substring(0, 1);
            final int pos = STATUS_PREFIXES.indexOf(prefix);
            if (pos < 0) {
                stat = Status.INVALID_PREFIX;
            }
            else {
                stat = Status.values()[pos];
                text = text.substring(1);
            }
        }
        final Reply result = new Reply(stat, text, cmd);
        if (stat != Status.SUCCESS) {
            final String msg = String.format("Aerotech error: %s. Reply text = '%s'.", stat.message, text);
            if (errorMode == ErrorHandling.LOG_ERRORS) {
                LOG.error(msg);
            }
            else {
                throw new CommandFailureException(msg, result);
            }
        }
        return result;
    }

    /**
     * Reads an analog input channel using {@code AIN(axisName,channel)}.
     * @param axisName The name of the axis which owns the channel.
     * @param channel The index of the channel.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply readAnalogInput(final String axisName, final int channel)
    throws DriverException
    {
        return sendCommand(String.format("AIN(%s, %d)", axisName, channel));
    }

    /**
     * Reads a digital input port using {@code DIN(axisName,port,bit)}.
     * @param axisName The name of the axis which owns the channel.
     * @param port The port index.
     * @param bit The bit index.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply readDigitalInput(final String axisName, final int port, final int bit)
    throws DriverException
    {
        return sendCommand(String.format("DIN(%s,%d,%d)", axisName, port, bit));
    }

    /**
     * Sets the state of a digital output channel using {@code DOUT axisName,port,bit:state}.
     * @param axisName The name of the axis owning the channel.
     * @param port The port index.
     * @param bit The bit number.
     * @param state The desired state; true for 1 and false for 0.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply writeDigitalOutput(final String axisName, final int port, final int bit, final boolean state)
        throws DriverException
    {
        return sendCommand(String.format("DOUT %s,%d,%d:%d", axisName, port, bit, state ? 1 : 0));
    }

    /**
     * Reads the calibrated position of an axis using {@code PFBKCAL(axisName)}.
     * @param axisName The name of the axis.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply readCalibratedPosition(final String axisName)
    throws DriverException
    {
        return sendCommand("PFBKCAL(" + axisName + ")");
    }


    /**
     * Reads the "program" position of an axis using {@code PFBKPROG(axisName)}.
     * @param axisName The name of the axis.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply readProgramPosition(final String axisName)
    throws DriverException
    {
        return sendCommand("PFBKPROG(" + axisName + ")");
    }


    /**
     * Reads the raw position of an axis using {@PFBK(axisName)}.
     * @param axisName The name of the axis.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply readRawPosition(final String axisName)
    throws DriverException
    {
        return sendCommand("PFBK(" + axisName + ")");
    }

    /**
     * Reads the status bits of an axis using {@code AXISSTATUS(axisName)}.
     * @param axisName The name of the axis.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply readAxisStatus(final String axisName) throws DriverException {
        return sendCommand("AXISSTATUS(" + axisName + ")");
    }


    /**
     * Reads the fault bits of an axis using {@code AXISFAULT(axisName)}.
     * @param axisName The name of the axis.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply readAxisFaults(final String axisName)  throws DriverException {
        return sendCommand("AXISFAULT(" + axisName + ")");
    }

    /**
     * Writes the value of a DGLOBAL array element using {@code DGLOBAL(index)=value}.
     * @param index The element index.
     * @param value The new value.
     * @return The reply object.
     * @throws DriverException
     */
    public Reply writeDglobal(final int index, final double value) throws DriverException {
        return sendCommand(String.format("DGLOBAL(%d)=%g", index, value));
    }


    /**
     * Reads the value of a DGLOBAL array element using {@code DGLOBAL(index)}.
     * @param index The element index.
     * @return The reply object.
     * @throws DriverException
     */
    public Reply readDglobal(final int index) throws DriverException {
        return sendCommand(String.format("DGLOBAL(%d)", index));
    }

    /**
     * Writes the value of an IGLOBAL array element using {@code IGLOBAL(index)=value}.
     * @param index The element index.
     * @param value The new value.
     * @return The reply object.
     * @throws DriverException
     */
    public Reply writeIglobal(final int index, final int value) throws DriverException {
        return sendCommand(String.format("IGLOBAL(%d)=%d", index, value));
    }


    /**
     * Reads the value of an IGLOBAL array element using {@code IGLOBAL(index)}.
     * @param index The element index.
     * @return The reply object.
     * @throws DriverException
     */
    public Reply readIglobal(final int index) throws DriverException {
        return sendCommand(String.format("IGLOBAL(%d)", index));
    }

    /**
     * Starts an AeroBasic program in a given task slot using {@code PROGRAM RUN taskNum,"progName"}.
     * @param taskNum The task slot number.
     * @param progName The name of the compiled program file in the controller filesystem, e.g., foo.bcx.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply startProgram(final int taskNum, final String progName) throws DriverException {
        return sendCommand(String.format("PROGRAM RUN %d,\"%s\"", taskNum, progName));
    }

    /**
     * Stops the AeroBasic program running in a given task slot using {@code PROGRAM STOP taskNum}.
     * @param taskNum The task slot number.
     * @param progName The name of the compiled program file in the controller filesystem, e.g., foo.bcx.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply stopProgram(final int taskNum) throws DriverException {
        return sendCommand(String.format("PROGRAM STOP %d", taskNum));
    }

    /**
     * Reads the state of a task slot using {@code TASKSTATE(taskNum)}. The state
     * is a small integer: 0=inactive, 1=idle, 2=ready, 3=running, 4=paused, 5=complete, 6=task error,
     * 10-running a task plugin.
     * @param taskNum The task number.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply readTaskState(final int taskNum) throws DriverException {
        return sendCommand(String.format("TASKSTATE(%d)", taskNum));
    }

    /**
     * Attempts to clear existing faults on axes using {@code FAULTACK axisName}.
     * @param axisNames The array of axis names.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply clearAxisFaults(final String... axisNames) throws DriverException {
        return sendCommand(
            Arrays.stream(axisNames)
            .collect(Collectors.joining(" ", "FAULTACK ", ""))
        );
    }

    /**
     * Homes a set of axes using {@code HOME axisName1 axisName2...}.
     * @param axisNames The array of axis names.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply homeAxes(final String... axisNames) throws DriverException {
        return sendCommand(
            Arrays.stream(axisNames)
            .collect(Collectors.joining(" ", "HOME ", ""))
        );
    }

    /**
     * Enables a set of axes using {@code ENABLE axisName1 axisName2...}.
     * @param axisNames The array of axis names.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply enableAxes(final String... axisNames) throws DriverException {
        return sendCommand(
            Arrays.stream(axisNames)
            .collect(Collectors.joining(" ", "ENABLE ", ""))
        );
    }


    /**
     * Disables a set of axes using {@code DISABLE axisName1 axisName2...}.
     * @param axisNames The array of axis names.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply disableAxes(final String... axisNames) throws DriverException {
        return sendCommand(
            Arrays.stream(axisNames)
            .collect(Collectors.joining(" ", "DISABLE ", ""))
        );
    }

    /**
     * Aborts motion on a set of axes using {@code ABORT axisName1 axisName2...}.
     * @param axisNames The array of axis names.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply abortAxes(final String... axisNames) throws DriverException {
        return sendCommand(
            Arrays.stream(axisNames)
            .collect(Collectors.joining(" ", "ABORT ", ""))
        );
    }

    /**
     * Attempts to clear all axis faults and task errors using {@code ACKNOWLEDGEALL}.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply clearAllFaults() throws DriverException {
        return sendCommand("ACKNOWLEDGEALL");
    }

    /**
     * Reads the status bits of a plane using {@code PLANESTATUS(planeNum)}.
     * @param planeNum The plane number.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply readPlaneStatus(final int planeNum) throws DriverException {
        return sendCommand(String.format("PLANESTATUS(%d)", planeNum));
    }

    /**
     * Sets the plane that will be used for subsequent coordinated motion.
     * @param planeNum The plane number.
     * @return The Reply object
     * @throws DriverException
     */
    public Reply setPlane(final int planeNum) throws DriverException {
        return sendCommand(String.format("PLANE %d", planeNum));
    }

    /**
     * The possible settings of the command wait mode for non-homing motion commands.
     * @see #setWaitMode(org.lsst.ccs.subsystem.motorplatform.bot.Aerotech.WaitMode)
     */
    public static enum WaitMode {
        /** Don't wait for the command to finish executing. */
        NOWAIT,
        /** Wait until any motion started by the command finishes. */
        MOVEDONE,
        /** Like {@code MOVEDONE} but also requires that the motion finishes in the correct position. */
        INPOS;
    }

    /**
     * Sets the waiting mode for motion commands,
     * except {@link #homeAxes(java.lang.String...)}, using {@code WAIT MODE mode}.
     * @param mode The desired mode.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply setWaitMode(final WaitMode mode) throws DriverException {
        return sendCommand(String.format("WAIT MODE %s", mode));
    }

    /**
     * How non-homing motion commands should ramp up to and down from the target speed of motion.
     * @see #setRampMode(org.lsst.ccs.subsystem.motorplatform.bot.Aerotech.RampMode)
     */
    public static enum RampMode {
        /** Ramp using a given rate of acceleration/deceleration. */
        RATE,
        /** Ramping occurs within a given distance of the end points of the motion . */
        DIST,
        /** Take a given amount of time to ramp up or down. */
        TIME;
    }

    /**
     * Sets the ramp (acceleration) mode for non-homing motion commands using {@code RAMP MODE mode}.
     * @param mode The desired mode.
     * @return The Reply object.
     * @throws DriverException
     * @see #setRampRate(double)
     * @see #setRampDist(double)
     * @see #setRampTime(double)
     */
    public Reply setRampMode(final RampMode mode) throws DriverException {
        return sendCommand(String.format("RAMP MODE %s", mode));
    }

    /**
     * Sets the acceleration for {@code RATE} mode ramping using {@code RAMP RATE accel}.
     * @param accel The desired acceleration.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply setRampRate(final double accel) throws DriverException {
        return sendCommand(String.format("RAMP RATE %g", accel));
    }

    /**
     * Sets the distance for {@code DIST} mode ramping using {@code RAMP DIST dist}.
     * @param dist The distance over which to accelerate to the target speed.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply setRampDist(final double dist) throws DriverException {
        return sendCommand(String.format("RAMP DIST %g", dist));
    }

    /**
     * Sets the time for {@code TIME} mode ramping using {@code RAMP TIME accel}.
     * @param time The time in seconds to take to accelerate to the target speed.
     * @return The Reply object.
     * @throws DriverException
     */
    public Reply setRampTime(final double time) throws DriverException {
        return sendCommand(String.format("RAMP TIME %g", time));
    }

    /**
     * The possible positioning modes
     */
    public static enum PositioningMode {
        /** Target coordinates are relative to the current position of the axis or axes. */
        INC,
        /** Target coordinates are absolute. */
        ABS;
    }

    /**
     * Sets the positioning mode for coordinated moves using either {@code INC} or {@code ABS}.
     * @param mode The desired mode.
     * @return The Reply object.
     * @throws DriverException
     * @see #moveLinear(double, org.lsst.ccs.subsystem.motorplatform.bot.Aerotech.Target...)
     */
    public Reply setPositioningMode(final PositioningMode mode) throws DriverException {
        return sendCommand(String.format("%s", mode));
    }

    /**
     * Specifies an axis and a target coordinate on that axis.
     */
    public static class Target {
        public final String axisName;
        public final double position;
        public Target(final String axisName, final double position) {
            this.axisName = axisName;
            this.position = position;
        }
        @Override public String toString() {return String.format("%s %g", axisName, position);}
    }

    /**
     * Executes a linear coordinated move using {@code LINEAR axis1 pos1 axis2 pos2 ... F speed}.
     * @param targetSpeed The speed to which to ramp up.
     * @param targets The target positions for each axis.
     * @return
     * @throws DriverException
     */
    public Reply moveLinear(final double targetSpeed, final Target... targets) throws DriverException {
        final String cmd =
            Arrays.stream(targets)
            .map(Target::toString)
            .collect(Collectors.joining(" ", "LINEAR ", String.format(" F %g", targetSpeed)));
        return sendCommand(cmd);
    }
}