package org.lsst.ccs.subsystem.bonnshutter.main;

import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.command.annotations.Command.CommandType.ACTION;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING1;
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.bonnshutter.BonnShutter;
import org.lsst.ccs.drivers.bonnshutter.BonnShutter.OpenCloseStatus;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.bonnshutter.states.ShutterState;

/**
 * A subsystem for controlling a Bonn shutter. Loosely modeled on the "real"
 * shutter subsystem, org-lsst-ccs-subsystem-shutter.
 *
 * Essential features of the Bonn Shutter subsystem.
 * <ul>
 * <li>Connect to the shutter
 * <li>Monitor the health of the shutter subsystem, emitting telemetry as
 * appropriate. Examples including monitoring the shutter device voltages
 * <li>Accept commands for opening and closing the shutter
 * <li>Report the status of the shutter blades (ShutterState)
 * <li>Maintain the configuration of the system.
 * </ul>
 *
 * @author tonyj
 */
public class BonnShutterSubsystem extends Subsystem implements HasLifecycle {

    private static final Logger LOG = Logger.getLogger(BonnShutterSubsystem.class.getName());
    private BonnShutter shutter;
    // TODO: This should be configurable
    private final String device = "/dev/ttyS0";
    // TODO: These should be configurable
    private static final double MIN_EXPOSURE_TIME = 0.1;
    private static final double MAX_EXPOSURE_TIME = 60 * 60;
    // TODO: This should be configurable
    private static final int OPEN_CLOSE_TIME_MILLIS = 1000;

    // @LookupField(strategy = LookupField.Strategy.TOP)
    // private Subsystem subsys;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;
    
    private Duration lastExposureTime;

    /**
     *  Constructor
     */
    public BonnShutterSubsystem() {
        super("bonnsub", AgentInfo.AgentType.WORKER);
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }


    @Command(type = ACTION, level = ENGINEERING1, description = "Performs a shutter motion calibration")
    public void calibrate() {
        throw new UnsupportedOperationException("calibrate no timplemented");
    }

    @Command(type = ACTION, level = NORMAL, description = "Closes the shutter", autoAck = false)
    public void closeShutter() {
        helper()
                .precondition(agentStateService.isInState(ShutterState.OPEN), "Shutter is not open")
                .enterFaultOnException(true)
                .action(() -> {
                    shutter.closeShutter(); // This command waits until movement is complete
                });
    }

    @Command(type = ACTION, level = NORMAL, description = "Opens the shutter", autoAck = false)
    public void openShutter() throws DriverException {
        helper()
                .precondition(agentStateService.isInState(ShutterState.CLOSED), "Shutter is not closed")
                .enterFaultOnException(true)
                .action(() -> {
                    //TODO: Should we have some timeout to avoid leaving the shutter
                    // open forever?
                    shutter.openShutter(); // This command waits until movement is complete
                });
    }

    @Command(type = ACTION, level = NORMAL, description = "Performs a timed exposure (open, wait, close)", autoAck = false)
    public void takeExposure(
            @Argument(name = "exposureTime", description = "The duration of the exposure in seconds")
            final double exposureTime) {

        helper()
                .precondition(exposureTime >= MIN_EXPOSURE_TIME, "Exposure time (%5g) below min (%5g)", exposureTime, MIN_EXPOSURE_TIME)
                .precondition(exposureTime <= MAX_EXPOSURE_TIME, "Exposure time (%5g) above max (%5g)", exposureTime, MAX_EXPOSURE_TIME)
                .precondition(agentStateService.isInState(ShutterState.CLOSED), "Shutter is not closed")
                .enterFaultOnException(true)
                .action(() -> {
                    // This command returns immediately and does not wait for exposure to complete.
                    shutter.expose((int) (exposureTime * 1000));
                    // We wait until the shutter is not in state CLOSED, so that we can safely use waitForExposure
                    // immediately after return from this command
                    boolean ok = agentStateService.waitFor((StateBundle t) -> !t.isInState(ShutterState.CLOSED), 1, TimeUnit.SECONDS);
                    if (!ok) {
                        throw new TimeoutException("Timed out while waiting for shutter to start opening");
                    }
                    lastExposureTime = Duration.ofSeconds((long) Math.ceil(exposureTime));
                });

    }

    // TODO: Is this really an action command? 
    @Command(type = ACTION, level = NORMAL, description = "Wait until exposure complete", autoAck = false)
    public void waitForExposure() throws InterruptedException {
        helper()
                .precondition(lastExposureTime != null, "No exposure in progress")
                .duration(lastExposureTime)
                .enterFaultOnException(true)
                .action(() -> {
                    boolean ok = agentStateService.waitFor((StateBundle t) -> t.isInState(ShutterState.CLOSED), lastExposureTime.toMillis() + OPEN_CLOSE_TIME_MILLIS, TimeUnit.MILLISECONDS);
                    lastExposureTime = null;
                    if (!ok) {
                        throw new TimeoutException("Timed out while waiting for shutter to start opening");
                    }
                });
    }

    /*
     * Note, an exception thrown from this method will cause the subsystem to fail to start.
     */
    @Override
    public void start() {
        shutter = new BonnShutter();
        try {
            shutter.open(device);
            String shutterVersion = shutter.getVersion();
            LOG.log(Level.INFO, "Connected to shutter on {0} version {1}", new Object[]{device, shutterVersion});

            // Check the current position of the shutter, and update the ShutterState accordingly.
            OpenCloseStatus shutterStatus = shutter.getOpenCloseStatus();
            switch (shutterStatus) {
                case CLOSED_A:
                case CLOSED_B:
                    agentStateService.updateAgentState(ShutterState.CLOSED);
                    break;
                case OPEN:
                    agentStateService.updateAgentState(ShutterState.OPEN);
                    // Should this also be a fault?
                    break;
                default:
                // We should go into fault state (how)
            }

        } catch (DriverException ex) {
            LOG.log(Level.SEVERE, "Failed to connect to bonn shutter on {0}", device);
            throw new RuntimeException("Failed to connect to bonn shutter on " + device, ex);
        }
        try {
            // If that worked, lets go on to open the GPIO connection
            ShutterGPIOMonitor gpio = new ShutterGPIOMonitor(agentStateService);
            // Check that the initial status is consistent with what we got from the
            // ASCII driver            
            gpio.start();
            LOG.log(Level.INFO, "GPIO Monitoring running");

        } catch (DriverException ex) {
            LOG.log(Level.SEVERE, "Failed to connect to set up GPIO connection to Bonn Shutter", ex);
            throw new RuntimeException("Failed to connect to set up GPIO connection to Bonn Shutter", ex);
        }
    }

}
