package org.lsst.ccs.subsystem.bonnshutter;

import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.command.annotations.Command.CommandType.ACTION;
import static org.lsst.ccs.command.annotations.Command.CommandType.QUERY;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING_ROUTINE;
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.bonnshutter.BonnShutter;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.bonnshutter.states.ShutterState;

/**
 * A subsystem for controlling a Bonn shutter. Loosely modeled on the "real"
 * shutter subsystem, org-lsst-ccs-subsystem-shutter.
 *
 * Essential features of the Bonn Shutter subsystem.
 * <ul>
 * <li>Connect to the shutter
 * <li>Monitor the health of the shutter subsystem, emitting telemetry as
 * appropriate. Examples including monitoring the shutter device voltages
 * <li>Accept commands for opening and closing the shutter
 * <li>Report the status of the shutter blades (ShutterState)
 * <li>Maintain the configuration of the system.
 * </ul>
 *
 * @author tonyj
 */
public class BonnShutterSubsystem extends Subsystem implements HasLifecycle {

    private static final Logger LOG = Logger.getLogger(BonnShutterSubsystem.class.getName());
    private BonnShutter shutter;

    @ConfigurationParameter(category = "General", units = "s",
                            description = "minimum exposure time")
    private volatile double minExposureTime = 0.1;

    @ConfigurationParameter(category = "General", units = "s",
                            description = "maximum exposure time")
    private volatile double maxExposureTime = 60 * 60;

    @ConfigurationParameter(category = "General", units = "ms",
                            description = "added waitForExposure time")
    private volatile int openCloseTimeMillis = 1000;

    // @LookupField(strategy = LookupField.Strategy.TOP)
    // private Subsystem subsys;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private BonnDevice device;
    
    private Duration lastExposureTime;

    /**
     *  Constructor
     */
    public BonnShutterSubsystem() {
        super("bonnsub", AgentInfo.AgentType.WORKER);
    }

   /**
    *   init() phase
    */
    @Override
    public void init() {
        shutter = device.getShutter();
        agentStateService.registerState(ShutterState.class, "Bonn Shutter State", this);
    }

    @Command(type = ACTION, level = ENGINEERING_ROUTINE, description = "Performs a shutter motion calibration")
    public void calibrate() {
        throw new UnsupportedOperationException("calibrate no timplemented");
    }

    @Command(type = ACTION, level = NORMAL, description = "Closes the shutter", autoAck = false)
    public void closeShutter() {
        helper()
	    .precondition(agentStateService.isInState(ShutterState.OPEN) ||
			  agentStateService.isInState(ShutterState.UNKNOWN),
                          "Shutter is not open")
                .enterFaultOnException(true)
                .action(() -> {
                    shutter.closeShutter(); // This command waits until movement is complete
                });
    }

    @Command(type = ACTION, level = NORMAL, description = "Opens the shutter", autoAck = false)
    public void openShutter() throws DriverException {
        helper()
                .precondition(agentStateService.isInState(ShutterState.CLOSED), "Shutter is not closed")
                .enterFaultOnException(true)
                .action(() -> {
                    shutter.openShutter(); // This command waits until movement is complete
                });
    }

    @Command(type = ACTION, level = NORMAL, description = "Performs a timed exposure (open, wait, close)", autoAck = false)
    public void takeExposure(
            @Argument(name = "exposureTime", description = "The duration of the exposure in seconds")
            final double exposureTime) {

        helper()
                .precondition(exposureTime >= minExposureTime, "Exposure time (%5g) below min (%5g)", exposureTime, minExposureTime)
                .precondition(exposureTime <= maxExposureTime, "Exposure time (%5g) above max (%5g)", exposureTime, maxExposureTime)
                .precondition(agentStateService.isInState(ShutterState.CLOSED), "Shutter is not closed")
                .enterFaultOnException(true)
                .action(() -> {
                    // This command returns immediately and does not wait for exposure to complete.
                    shutter.expose((int) (exposureTime * 1000));
                    // We wait until the shutter is not in state CLOSED, so that we can safely use waitForExposure
                    // immediately after return from this command
                    try {
                        agentStateService.waitForStatePredicate((StateBundle t) -> !t.isInState(ShutterState.CLOSED), 1, TimeUnit.SECONDS);
                    } catch (TimeoutException e) {
                        throw new TimeoutException("Timed out while waiting for shutter to start opening");
                    }
                    lastExposureTime = Duration.ofSeconds((long) Math.ceil(exposureTime));
                });

    }

    @Command(type = QUERY, level = NORMAL, description = "Wait until exposure complete", autoAck = false)
    public void waitForExposure() throws InterruptedException {
        helper()
                .precondition(lastExposureTime != null, "No exposure in progress")
                .duration(lastExposureTime)
                .enterFaultOnException(true)
                .action(() -> {
		    try {
                        agentStateService.waitForStatePredicate((StateBundle t) -> t.isInState(ShutterState.CLOSED), lastExposureTime.toMillis() + openCloseTimeMillis, TimeUnit.MILLISECONDS);
                        lastExposureTime = null;
                    } catch (TimeoutException e) {
                        lastExposureTime = null;
                        throw new TimeoutException("Timed out while waiting for shutter to close");
                    }
                });
    }

}
