package org.lsst.ccs.subsystem.bonnshutter;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING1;
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.bonnshutter.BonnShutter;
import org.lsst.ccs.drivers.bonnshutter.BonnShutter.OpenCloseStatus;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.ClearAlertHandler.ClearAlertCode;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.bonnshutter.alerts.BonnShutterAlerts;
import org.lsst.ccs.subsystem.bonnshutter.states.ShutterState;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Interfaces with driver for Bonn Shutter
 *  Code to open connection moved from BonnShutterSubsystem.
 *
 *  @author Al Eisner
 */
public class BonnDevice extends Device {

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService stateService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

   /**
    *  Channel names
    */

    private static final String nameV5 = "Bonn_V5";
    private static final String nameV36 = "Bonn_V36";

   /**
    *  Data fields.
    */

    private final BonnShutter shutter = new BonnShutter();   // Driver
    private ShutterGPIOMonitor gpio;
    private static final Logger LOG = Logger.getLogger(BonnDevice.class.getName());

    /* Numeric Channel counter, returned via checkChannel  */
    private Integer chanKey;

    /* Configuration parameters (none related to hardware settings) */

    @ConfigurationParameter(category = "General", isFinal = true, 
                            description = "port Id", units="unitless")
    private volatile String devcId = "/dev/ttyS0";

    /* Hard-code hardware settings  */

    /**
     *  Provide shutter (driver) instance for use by subsystem code
     */
    BonnShutter getShutter() {
        return shutter;
    }

   /**

    *  Performs basic initialization.  
    *  Verify that all ConfigurationParameters are set.
    */
    @Override
    protected void initDevice() {

        chanKey = 0;

        if (devcId == null) {
            ErrorUtils.reportConfigError(LOG, name, "devcId", "is missing");
        }
        fullName = "BonnShutter " + name + " (" + devcId + ")";
        
        /**
         * Tests whether WRONG_STATE alert can be cleared. Allow it to be
         * cleared if shutter is now in CLOSED state.
         *
         */
        ClearAlertHandler deviceClearAlertHandler = new ClearAlertHandler() {
            @Override
            public ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                String alertId = alert.getAlertId();
                if (!alertId.equals(BonnShutterAlerts.WRONG_STATE.getId())) {
                    return ClearAlertCode.UNKNOWN_ALERT;
                }
                if (stateService.isInState(ShutterState.CLOSED)) {
                    return ClearAlertCode.CLEAR_ALERT;
                } else {
                    return ClearAlertCode.DONT_CLEAR_ALERT;
                }
            }
        };   
        alertService.registerAlert(BonnShutterAlerts.WRONG_STATE.newAlert(), deviceClearAlertHandler);
    }


   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize() {

        try {
            shutter.open(devcId);
            String shutterVersion = shutter.getVersion();
            LOG.log(Level.INFO, "Connected to " + fullName + " version {1}", new Object[]{devcId, shutterVersion});

            // Check the current position of the shutter, and update the ShutterState accordingly.
            OpenCloseStatus shutterStatus = shutter.getOpenCloseStatus();
            switch (shutterStatus) {
                case CLOSED_A:
                case CLOSED_B:
                    stateService.updateAgentState(ShutterState.CLOSED);
                    break;
                case OPEN:
                    stateService.updateAgentState(ShutterState.OPEN);
                    alertService.raiseAlert(BonnShutterAlerts.WRONG_STATE.newAlert(), AlertState.ALARM, "BonnShutter should not start in OPEN state");
                    break;
	        case UNKNOWN:
                default:
		    stateService.updateAgentState(ShutterState.UNKNOWN);
                    alertService.raiseAlert(BonnShutterAlerts.WRONG_STATE.newAlert(), AlertState.ALARM, "BonnShutter should not start in UNKNOWN state");
            }
            initSensors();

            /* If device connection succeeded, open the GPIO connection */
            if (gpio == null) {
                try {
                    gpio = new ShutterGPIOMonitor(stateService, alertService);
                    gpio.start();
                    LOG.log(Level.INFO, "GPIO Monitoring running");
                } catch (Exception ex) {
                    LOG.log(Level.SEVERE, "Failed to start GPIO monitoring of Bonn Shutter", ex);
                    throw new RuntimeException("GPIO not available", ex);
                }
            }

            setOnline(true);
            LOG.log(Level.INFO, "\n Connected to {0}", name);
        } catch (DriverException ex) {
            if (!inited) {
                LOG.log(Level.SEVERE,"Failed to connect to bonn shutter", ex);
            }
	    close();
        }

        inited = true;
    }


   /**
    *  Closes the connection.
    */
    @Override
    protected void close() {
        try {
            shutter.close();
        }
        catch (DriverException e) {
        }
    }

   /**
    *  Checks a channel's parameters for validity.
    *
    *  The only channel types which appear useful are two voltage readings.q
    *  For an invalid channel name, ErroTUtils will throw Exception.
    *
    *  @param  ch   The Channel
    *  @throws Exception
    *
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    */
    @Override
    protected int[] checkChannel(Channel ch) throws Exception 
    {
        String type = ch.getTypeStr();
        String chanName = ch.getName();
        if (type.equals(nameV5) || type.equals(nameV36)) {
            chanKey++;
	} else {
            ErrorUtils.reportChannelError(LOG, chanName, "type", type);
        }
        return new int[]{chanKey, 0};
    }

   /**
    *  Reads a channel.
    *
    *  @param  ch      The Channel
    */
    @Override
    protected double readChannel(Channel ch)
    {
        double value = super.readChannel(ch);   //NaN
        String chanName = ch.getName();
        if (isOnline()) {
            try {
                if (chanName.equals(nameV5)) {
                    value = shutter.getVoltages().getV5();
                } else if (chanName.equals(nameV36)) {
                    value = shutter.getVoltages().getV36();
               }
            }        
            catch (DriverTimeoutException et) {
                LOG.log(Level.SEVERE, name + " timeout reading " + chanName, et);
                setOnline(false);
            }
    	catch (DriverException e) {
                LOG.log(Level.SEVERE, name + " exception reading data " + chanName, e);
            }
        }
        return value;
    }

    /* Commands */

   /**
    * Reset (aftr an error condition)
    *
    * @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION, level = ENGINEERING1, 
             name="reset", description="reset (after an error condition)")
    public void reset() throws DriverException{
        shutter.reset();
    }

   /**
    * Read some parameters and data from shutter controller
    *
    * All DriverExceptions are caught; if one occurs, the data field
    * is replaced by the text (String) associated with the exception.
    *  
    * @return  String reporting all information read and exceptions.
    */
    @Command(type=Command.CommandType.QUERY, level = NORMAL, name="readInfo",
             description="Read some shutter parameters and data")
    public String readInfo()
    {
        String table = "Shutter parameters, status and data\n";

        table += "\n  VelocityProfile  ";
        try {
            BonnShutter.ProfileParameters pp = shutter.getVelocityProfileParameters();
            table += pp.toString();
        } catch (DriverException ex) {
            table += ex.getMessage();
        }
        table += "\n  Voltages         ";
        try {
            BonnShutter.Voltages vv = shutter.getVoltages();
            table += vv.toString();
        } catch (DriverException ex) {
            table += ex.getMessage();
        }
        table += "\n  OpenCloseStatus  ";
        try {
            OpenCloseStatus st = shutter.getOpenCloseStatus();
            table += st.toString();
        } catch (DriverException ex) {
            table += ex.getMessage();
        }
       table += "\n  Error Status     ";
       table += "not yet implemented";
       table += "\n\n";

       return table;
    }
}
