package org.lsst.ccs.subsystem.bonnshutter.main;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.gpio.GPIODriver;
import org.lsst.ccs.subsystem.bonnshutter.main.BonnShutterSubsystem.ShutterState;

/**
 * Monitors the state of the shutter using GPIO and allows the shutter to be
 * opened and closed;
 *
 * @author tonyj
 */
class ShutterGPIOMonitor {

    private static final Logger LOG = Logger.getLogger(ShutterGPIOMonitor.class.getName());
    private final Subsystem subsystem;

    // GPIO only tells is that the blade is closed or not closed. 
    // It cannot tell us if it is in motion or not.
    private enum BladeState { CLOSED, OPEN };
    
    private final int openCloseGPIO = 508;
    private final int shutterAGPIO = 504;
    private final int shutterBGPIO = 505;
    private final int shutterErrorGPIO = 506;
    private final GPIODriver driver;
    private final GPIODriver.GPIOChannel shutterAChannel;
    private final GPIODriver.GPIOChannel shutterBChannel;
    private final GPIODriver.GPIOChannel errorChannel;
    private final GPIODriver.GPIOChannel openCloseChannel;
    private BladeState shutterAState;
    private BladeState shutterBState;
    private boolean errorState;
    private Thread monitorThread;

    ShutterGPIOMonitor(Subsystem subsystem) throws DriverException {
        this.subsystem = subsystem;
        driver = new GPIODriver();
        shutterAChannel = driver.getChannel(shutterAGPIO);
        shutterBChannel = driver.getChannel(shutterBGPIO);
        errorChannel = driver.getChannel(shutterErrorGPIO);
        openCloseChannel = driver.getChannel(openCloseGPIO);
    }

    void start() {
        monitorThread = new Thread("shutterMonitor") {
            @Override
            public void run() {
                try {
                    runMonitor();
                } catch (DriverException ex) {
                    LOG.log(Level.SEVERE, "Monitor thread exited", ex);
                    // TODO: Put the subsystem into fault state?
                }
            }
        };
        monitorThread.setDaemon(true);
        monitorThread.start();
    }

    void stop() {
        if (monitorThread != null && monitorThread.isAlive()) {
            monitorThread.interrupt();
        }
    }

    private void runMonitor() throws DriverException {
        try {
            shutterAChannel.lock();
            shutterBChannel.lock();
            errorChannel.lock();
            while (!Thread.interrupted()) {
                BladeState a = shutterAChannel.read() ? BladeState.OPEN : BladeState.CLOSED;
                BladeState b = shutterBChannel.read() ? BladeState.OPEN : BladeState.CLOSED;
                boolean e = !errorChannel.read();
                if (a != shutterAState) {
                    LOG.log(Level.INFO, "Shutter A State now: {0}", a);
                    shutterAState = a;
                }
                if (b != shutterBState) {
                    LOG.log(Level.INFO, "Shutter B State now: {0}", b);
                    shutterBState = b;
                }
                boolean shutterClosed = a == BladeState.CLOSED || b == BladeState.CLOSED;
                if (subsystem.isInState(ShutterState.CLOSED) && !shutterClosed) {
                    subsystem.updateAgentState(ShutterState.OPEN);
                } else if (subsystem.isInState(ShutterState.OPEN) && shutterClosed) {
                    subsystem.updateAgentState(ShutterState.CLOSED);
                }
                if (e != errorState) {
                    LOG.log(Level.INFO, "Error State now: {0}", e);
                    errorState = e;
                    // TODO: Put subsystem into fault state
                }
            }
        } finally {
            shutterAChannel.unlock();
            shutterBChannel.unlock();
            errorChannel.unlock();
        }
    }

    void open() throws DriverException {
        openCloseChannel.set();
    }

    void close() throws DriverException {
        openCloseChannel.clear();
    }

    void expose(double duration) {
        // Not yet implemented, since we currently use the expose command
        // built in to the shutter.
        throw new UnsupportedOperationException("expose via GPIO not implemented");
    }
}
