package org.lsst.ccs.subsystem.bonnshutter.main;

import java.time.Duration;
import java.util.concurrent.Callable;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;

/**
 * A class to make it easier to execute commands while correctly handling ACKS,
 * NACKS, and exceptions.
 *
 * @author tonyj
 */
public class CommandExecutor {

    private boolean preconditionsMet = true;
    // TODO: If there a way to get the level of the current command? If so we
    // might make this default to true if it is a NORMAL level command.
    private boolean enterFaultOnError = false;
    private String preconditionsReason = "Preconditions not met";
    private final Agent subsys;
    // TODO: Is there a way to get the duraction specified in the annotation?
    private Duration duration;
    private final static Alert defaultAlert = new Alert("CMDFAIL", "Command failed");

    public CommandExecutor(Agent subsys) {
        this.subsys = subsys;
    }

    /**
     * A simple precondition.
     *
     * @param ok If false command will not proceed
     * @return The CommandExecutor, to allow method chaining
     */
    public CommandExecutor precondition(boolean ok) {
        this.preconditionsMet &= ok;
        return this;
    }

    /**
     * A precondition with a reason.
     *
     * @param ok If false command will not proceed
     * @param reason The reason to use if precondition is not met
     * @param args If present then reason is interpreted as a format string, and args 
     * are substituted into the format
     * @return The CommandExecutor, to allow method chaining
     */
    public CommandExecutor precondition(boolean ok, String reason, Object... args) {
        if (preconditionsMet && !ok) {
            preconditionsMet = false;
            if (args.length > 0) {
                preconditionsReason = String.format(reason, (Object) args);
            } else {
                preconditionsReason = reason;
            }
        }
        return this;
    }

    /**
     * Set the duration for the command
     *
     * @param duration The duration for the command, overrides the default
     * duration provided by any annotation.
     * @return The CommandExecutor, to allow method chaining
     */
    CommandExecutor duration(Duration duration) {
        this.duration = duration;
        return this;
    }

    /**
     * Controls whether exceptions should be thrown back to caller, or whether
     * the should cause the subsystem to entry fault state
     *
     * @param enterFaultOnError If <code>true</code> then any exceptions during
     * command execution will be caught, and the subsystem will be put into
     * fault state.
     * @return The CommandExecutor, to allow method chaining
     */
    CommandExecutor enterFaultOnError(boolean enterFaultOnError) {
        this.enterFaultOnError = enterFaultOnError;
        return this;
    }

    /**
     * Executes the specified callable. Designed to be the last method on the
     * method chain.
     *
     * @param <T> The return type of the callable
     * @param callable The callable to call
     * @return The result of the callable
     */
    public <T> T action(Callable<T> callable) {
        handlePrecondtions();
        try {
            return callable.call();
        } catch (Exception x) {
            throw handleException(x);
        }
    }

    public void action(RunnableWithException runnable) {
        handlePrecondtions();
        try {
            runnable.run();
        } catch (Exception x) {
            throw handleException(x);
        }
    }

    private void handlePrecondtions() {
        if (!preconditionsMet) {
            subsys.sendNack(preconditionsReason);
        } else if (duration != null) {
            subsys.sendAck(duration);
        }
    }

    private CommandFailedException handleException(Exception x) {
        if (enterFaultOnError) {
            subsys.getAlertService().raiseAlert(defaultAlert, AlertState.ALARM, "Execution of command failed unexpectedly");
            return new CommandFailedException("Exception while executing command, fault state entered", x);
        } else {
            return new CommandFailedException("Unexpected exception during command execution", x);
        }
    }

    public static interface RunnableWithException {

        void run() throws Exception;
    }

    public static class CommandFailedException extends RuntimeException {

        public CommandFailedException(String message, Throwable cause) {
            super(message, cause);
        }

    }
}
