package org.lsst.ccs.subsystem.ats;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.pfeiffer.TPG361;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *
 * Device class for the Pfeiffer TPG361 vacuum gauge.
 *
 * @author Heather Kelly / Homer Neal
 */
public class Pfeiffer361Device extends Device {

    public static final int
        CHAN_PRESSURE = 0,
        NUM_CHANS = 1;

    @ConfigurationParameter (category="Device")
    private volatile String devcName;

    private final int baudRate = 0;  // Use default

    private static final Logger LOG = Logger.getLogger(Pfeiffer361Device.class.getName());
    private final TPG361 tpg = new TPG361();
    private boolean initError = false;
    

    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        if (devcName == null) {
            ErrorUtils.reportConfigError(LOG, name, "devcName", "is not specified");
        }
        fullName = "Pfeiffer TPC361 (" + devcName + ")";        
    }

    /**
     * Initializes the connection.
     */
    @Override
    protected void initialize() {
        try {
            tpg.openSerial(devcName, baudRate);
            setOnline(true);
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
        } catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
        }
    }

    /**
     * Closes the connection.
     */
    @Override
    public void close() {
        try {
            tpg.close();
        } catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
    }

    @Override
    protected int[] checkChannel(Channel ch) throws Exception {
        int hwChan = ch.getHwChan();
        String name = ch.getName();
        if (hwChan < 0 || hwChan >= NUM_CHANS) {
            ErrorUtils.reportChannelError(LOG, name, "hardware channel", hwChan);
        }
        return new int[]{0, 0};
    }

    @Override
    protected double readChannel(Channel ch) {
        double value = Double.NaN;
        int chan = ch.getHwChan();
        try {
            switch (chan) {
            case CHAN_PRESSURE:
                value = tpg.readPressure()[0]; break;
            }
        } catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error reading channel {0}: {1}", new Object[]{chan, e});
            setOnline(false);
        }
        return value;
    }

}
