package org.lsst.ccs.subsystem.ats;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.cryocon.M24C;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *
 * Device class for the Cryogenic Temperature Controller 24C driver
 *
 * @author Heather Kelly / Homer Neal *
 */
public class AtsCryoCon24cDevice extends Device {

    public enum ControlType {OFF, MAN, PID}
    public static final int
        CHAN_HTR_1 = 0,
        CHAN_HTR_2 = 1,
        CHAN_TEMP_A = 2,
        CHAN_TEMP_B = 3,
        CHAN_TEMP_C = 4,
        CHAN_TEMP_D = 5,
        CHAN_PID_P_1 = 6,
        CHAN_PID_I_1 = 7,
        CHAN_PID_D_1 = 8,
        CHAN_PID_P_2 = 9,
        CHAN_PID_I_2 = 10,
        CHAN_PID_D_2 = 11,
        CHAN_SETPOINT_1 = 12,
        CHAN_SETPOINT_2 = 13,
        NUM_CHANS = 14;

    @ConfigurationParameter(category = "Device")
    volatile String host;

    M24C.ConnType connType = M24C.ConnType.NET;
    int port = 5000;

    private static final Logger LOG = Logger.getLogger(AtsCryoCon24cDevice.class.getName());
    private final M24C m24c = new M24C();
    private boolean initError = false;

    
    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        if (host == null) {
            ErrorUtils.reportConfigError(LOG, name, "host", "is not specified");
        }
        fullName = "Cryocon M24C (" + host + ")";        
    }

    /**
     * Initializes the connection.
     */
    @Override
    protected void initialize() {
        try {
            m24c.open(connType, host, port);
            setOnline(true);
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
        } catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
        }
    }

    /**
     * Closes the connection.
     */
    @Override
    public void close() {
        try {
            m24c.close();
        } catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
    }

    /**
     * Checks a channel's parameters for validity.
     *
     * @param name
     * @param hwChan
     * @param type
     * @param subtype
     * @return 
     * @throws Exception 
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception {
        if (hwChan < 0 || hwChan >= NUM_CHANS) {
            ErrorUtils.reportChannelError(LOG, name, "hardware channel", hwChan);
        }
        return new int[]{0, 0};
    }

    /**
     * Reads a channel.
     *
     * @param chan
     * @param type
     * @return 
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
        try {
            switch (chan) {
            case CHAN_HTR_1:
                value = m24c.getHtrRead(1); break;
            case CHAN_HTR_2:
                value = m24c.getHtrRead(2); break;
            case CHAN_TEMP_A:
                value = getTemp("A"); break;
            case CHAN_TEMP_B:
                value = getTemp("B"); break;
            case CHAN_TEMP_C:
                value = getTemp("C"); break;
            case CHAN_TEMP_D:
                value = getTemp("D"); break;
            case CHAN_PID_P_1:
                value = m24c.getPidP(1); break;
            case CHAN_PID_I_1:
                value = m24c.getPidI(1); break;
            case CHAN_PID_D_1:
                value = m24c.getPidD(1); break;
            case CHAN_PID_P_2:
                value = m24c.getPidP(2); break;
            case CHAN_PID_I_2:
                value = m24c.getPidI(2); break;
            case CHAN_PID_D_2:
                value = m24c.getPidD(2); break;
            case CHAN_SETPOINT_1:
                value = m24c.getSetPoint(1); break;
            case CHAN_SETPOINT_2:
                value = m24c.getSetPoint(2); break;
            }
        } catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error reading channel {0}: {1}", new Object[]{chan, e});
            setOnline(false);
        }
        return (value);
    }

    /**
     * Displays Temp Control 24C identification.
     *
     * @return
     * @throws DriverException
     */
    @Command(type = CommandType.QUERY, description = "Show Temp Control 24C identification")
    public String showIdentification() throws DriverException {
        String[] ident = m24c.getIdentification();
        return   "Manufacturer:  " + ident[0] + "\n"
               + "Model name:    " + ident[1] + "\n"
               + "Serial number: " + ident[2] + "\n"
               + "F/W version:   " + ident[3];
    }

    /**
     * Get maximum set point for loop
     *
     * @param loop
     * @return 
     * @throws DriverException 
     */
    @Command(type = CommandType.QUERY, description = "Retrieve MaxSetPoint for loop")
    public double getMaxSetPoint(@Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getMaxSetPoint(loop);
    }

    /**
     * Get Source for Loop returns Channels A through D
     *
     * @param loop
     * @return 
     * @throws DriverException 
     */
    @Command(type = CommandType.QUERY, description = "Retrieve setPoint for loop")
    public double getSetPoint(@Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getSetPoint(loop);
    }

    /**
     * Get heater reading for specified loop
     *
     * @param  loop  The loop number
     * @return The heater reading
     * @throws DriverException
     */
    @Command(type = CommandType.QUERY, description = "Retrieve heater reading for loop")
    public double getHtrRead(@Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getHtrRead(loop);
    }

    /**
     * Get [P]ID parameter for specified loop
     *
     * @param loop
     * @return 
     * @throws DriverException 
     */
    @Command(type = CommandType.QUERY, description = "Retrieve PID parameter P for loop")
    public double getPidP(@Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getPidP(loop);
    }

    /**
     * Get P[I]D parameter for specified loop
     *
     * @param loop
     * @return 
     * @throws DriverException 
     */
    @Command(type = CommandType.QUERY, description = "Retrieve PID parameter I for loop")
    public double getPidI(@Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getPidI(loop);
    }

    /**
     * Get PI[D] parameter for specified loop
     *
     * @param loop
     * @return 
     * @throws DriverException 
     */
    @Command(type = CommandType.QUERY, description = "Retrieve PID parameter D for loop")
    public double getPidD(@Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getPidD(loop);
    }

    /**
     * Get Source for Loop returns Channels A through D
     *
     * @param loop
     * @return 
     * @throws DriverException 
     */
    @Command(type = CommandType.QUERY, description = "Retrieve source channel for loop")
    public char getLoopSource(@Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getSource(loop);
    }

    /**
     * Get Heater Range for Loop [1-4]
     *
     * @param  loop  The loop number
     * @return The range string
     * @throws DriverException
     */
    @Command(type = CommandType.QUERY, description = "Get heater range for loop")
    public String getHeaterRange(@Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getHeaterRange(loop);
    }

    /**
     * Get maximum heater power for Loop [1-4]
     *
     * @param  loop  The loop number
     * @return The maximum heater power (%)
     * @throws DriverException
     */
    @Command(type = CommandType.QUERY, description = "Get maximum heater power for loop")
    public double getHeaterMax(@Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getHeaterMax(loop);
    }

    /**
     * Get temperature reading for channels A through D
     *
     * @param channel The channel (A-D)
     * @return  The temperature
     * @throws DriverException
     */
    @Command(type = CommandType.QUERY, description = "Retrieve temperature info")
    public double getTemp(@Argument(name = "channel", description = "Temp Channel to read") String channel)
            throws DriverException {
        return m24c.getTemp(channel.charAt(0));
    }

    /**
     * Get Temperature Units Channels A through D *
     * @param channel
     * @return 
     * @throws DriverException 
     */
    @Command(type = CommandType.QUERY, description = "Retrieve temperature units")
    public char getUnit(@Argument(name = "channel", description = "Temp Channel to read") String channel)
            throws DriverException {
        return m24c.getUnit(channel.charAt(0));
    }

    /**
     * Get sensor type for a temperature channel
     *
     * @param channel The temperature channel (A-D)
     * @return The sensor type string
     * @throws DriverException
     */
    @Command(type = CommandType.QUERY, description = "Get sensor type")
    public String getType(@Argument(name = "channel", description = "Temp Channel to set") String channel)
            throws DriverException {
        return m24c.getType(channel.charAt(0));
    }

    /**
     * Gets whether OTD is enabled
     *
     * @return Whether OTD is enabled
     * @throws DriverException
     */
    @Command(type = CommandType.QUERY, description = "Get whether OTD system enabled")
    public boolean isOtdEnabled() throws DriverException {
        return m24c.isOtdEnabled();
    }

    /**
     * Gets OTD source channel
     *
     * @return The source channel (A-D)
     * @throws DriverException
     */
    @Command(type = CommandType.QUERY, description = "Get source channel for the OTD system")
    public char getOtdSource() throws DriverException {
        return m24c.getOtdSource();
    }

    /**
     * Gets OTD temperature limit
     *
     * @return The temperature limit
     * @throws DriverException
     */
    @Command(type = CommandType.QUERY, description = "Get temperature limit for the OTD system")
    public double getOtdTemp() throws DriverException {
        return m24c.getOtdTemp();
    }

    /**
     * Set into control mode
     *
     * @throws DriverException
     */
    @Command(type = CommandType.ACTION, description = "Set the CryoCon24c into control mode")
    public void setToControl() throws DriverException {
        m24c.setToControl();
    }

    /**
     * Stop control mode
     *
     * @throws DriverException
     */
    @Command(type = CommandType.ACTION, description = "Stop the CryoCon24c control mode")
    public void stopControl() throws DriverException {
        m24c.stopControl();
    }

    /**
     * Get control mode
     *
     * @return Whether in control mode
     * @throws DriverException
     */
    @Command(type = CommandType.QUERY, description = "Is the CryoCon24c in control mode")
    public boolean isInControl() throws DriverException {
        return m24c.isInControl();
    }

    /**
     * Set the control type for specified loop
     *
     * @param  loop  The loop number
     * @param  type  The control type (OFF, MAN or PID)
     * @throws DriverException
     */
    @Command(type = CommandType.ACTION, description = "Set the control type for a loop")
    public void setControlType(@Argument(name = "loop", description = "loop number [1-4]") int loop,
                               @Argument(description = "control type (OFF, MAN or PID)") ControlType type)
            throws DriverException {
        m24c.setControlType(loop, type.name());
    }

    /**
     * Get the control type for specified loop
     *
     * @param  loop  The loop number
     * @return  The control type (OFF, MAN or PID)
     * @throws DriverException
     */
    @Command(type = CommandType.QUERY, description = "Set the control type for a loop")
    public String getControlType(@Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getControlType(loop);
    }

    /**
     * Set the manual heater power for specified loop
     *
     * @param  loop   The loop number
     * @param  power  The power as % of maximum
     * @throws DriverException
     */
    @Command(type = CommandType.ACTION, description = "Set the manual heater power for a loop")
    public void setHeaterPower(@Argument(name = "loop", description = "loop number [1-4]") int loop,
                               @Argument(description = "power as % of maximum") double power)
            throws DriverException {
        m24c.setHeaterPower(loop, power);
    }

    /**
     * Get the manual heater power for specified loop
     *
     * @param  loop   The loop number
     * @return  The set power as % of maximum
     * @throws DriverException
     */
    @Command(type = CommandType.QUERY, description = "Get the manual heater power for a loop")
    public double getHeaterPower(@Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getHeaterPower(loop);
    }

}
