package org.lsst.ccs.subsystem.ats;

import java.time.Duration;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.ats.constants.AtsAgentProperties;
import org.lsst.ccs.subsystem.ats.data.AtsState;

/**
 * Implementation of auxiliary test system
 *
 * @author The LSST CCS Team
 */
public class AtsMain extends Subsystem implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;
    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    private static final Logger LOG = Logger.getLogger(AtsMain.class.getName());

    public AtsMain() {
        super("ats", AgentInfo.AgentType.WORKER);
    }

    

    /**
     *  Post initialization
     */
    @Override
    public void postInit() {
        subsys.getAgentService(AgentPropertiesService.class).setAgentProperty(AtsAgentProperties.ATS_TYPE, AtsMain.class.getCanonicalName());
    }


    /**
     *  Post start
     */
    @Override
    public void postStart() {
        LOG.info("ATS subsystem started");
    }


    /**
     *  Gets the state of the AuxTest module.
     *
     *  @return  The full Raft state
     */
    public AtsState getState() {
        return new AtsState(getTickPeriod());
}


    /**
     *  Gets the full state of the AuxTest module.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The full Raft state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the ATS system state", level=0)
    public AtsState getSystemState()
    {
        return getState();
    }    


    /**
     *  Sets the update period.
     *
     *  @param  value  The update period (milliseconds) to set.
     */
    @Command(type=Command.CommandType.ACTION, description="Set the update interval")
    public void setUpdatePeriod(@Argument(description="The tick period (ms)") int value)
    {
        setTickPeriod(value);
        publishState();
    }


    /**
     *  Publishes the state of the AuxTest module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(AtsState.KEY, getState()));
    }


    /**
     *  Sets the monitoring publishing period
     */
    private void setTickPeriod(long period)
    {
        pts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period
     */
    private int getTickPeriod()
    {
        return (int)pts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

}
