package org.lsst.ccs.subsystem.ats;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.pfeiffer.TPG361;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *
 * Device class for the Pfeiffer TPG361 vacuum gauge.
 *
 * @author Heather Kelly / Homer Neal *
 */
public class Pfeiffer361Device extends Device {

    public static final int
        CHAN_PRESSURE = 0,
        NUM_CHANS = 1;

    private String devcName;
    private int baudRate = 0;  // Use default

    private static final Logger LOG = Logger.getLogger(Pfeiffer361Device.class.getName());
    private final TPG361 tpg = new TPG361();
    
    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        if (devcName == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "devcName", "is not specified");
        }
        fullName = "Pfeiffer TPC361 (" + devcName + ")";        
    }

    /**
     * Initializes the connection.
     */
    @Override
    protected void initialize() {
        try {
            tpg.openSerial(devcName, baudRate);
            initSensors();
            setOnline(true);
            LOG.info("Connected to " + fullName);
        } catch (DriverException e) {
            if (!inited) {
                LOG.error("Error connecting to " + fullName + ": " + e);
            }
        }
        inited = true;
    }

    /**
     * Closes the connection.
     */
    @Override
    public void close() {
        try {
            tpg.close();
        } catch (DriverException e) {
            LOG.error("Error disconnecting from " + fullName + ": " + e);
        }
    }

    /**
     * Checks a channel's parameters for validity.
     *
     * @param name
     * @param hwChan
     * @param type
     * @param subtype
     * @return 
     * @throws Exception 
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception {
        if (hwChan < 0 || hwChan >= NUM_CHANS) {
            MonitorLogUtils.reportError(LOG, name, "hardware channel", hwChan);
        }
        return new int[]{0, 0};
    }

    /**
     * Reads a channel.
     *
     * @param chan
     * @param type
     * @return 
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = Double.NaN;
        try {
            switch (chan) {
            case CHAN_PRESSURE:
                value = tpg.readPressure()[0]; break;
            }
        } catch (DriverException e) {
            LOG.error("Error reading channel " + chan + ": " + e);
            setOnline(false);
        }
        return value;
    }

}
