package org.lsst.ccs.subsystem.archon;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.ShortBuffer;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.archon.RawImageData;
import org.lsst.ccs.utilities.ccd.CCDGeometry;
import org.lsst.ccs.utilities.ccd.CCDGeometry.CCDSegment;
import org.lsst.ccs.utilities.ccd.CCDTransform;
import org.lsst.ccs.utilities.ccd.CCDTransform.PixelType;
import org.lsst.ccs.utilities.image.DefaultImageSet;
import org.lsst.ccs.utilities.image.FitsFileWriter;
import org.lsst.ccs.utilities.image.ImageSet;
import org.lsst.ccs.utilities.image.MetaDataSet;

/**
 * Convert an Archon RawImage into an ImageSet as expected by the the
 * FitsFileWriter.
 *
 * @author tonyj
 */
public class RawImageConverter {

    private final static Logger logger = Logger.getLogger(RawImageConverter.class.getName());
    private final CCDGeometry geom;
    private final DefaultImageSet imageSet;
    private final RawImageData rawImage;
    private final static int BUFFER_SIZE = 1000000;

    private double imageAverageSignal = 0.0;

    /**
     * In the case of the archon controller, assuming it has been configured
     * correctly the RawImageData has already been assembled into a complete
     * image, so what this class is really doing is de-assembling the image,
     * into the format that it will be generated in by the LSST electronics.
     *
     * @param rawImage The raw image data from the archon
     * @param geom The CCDGeometry
     */
    RawImageConverter(RawImageData rawImage, CCDGeometry geom) {
        this(rawImage, geom, false);
    }

    /**
     * In the case of the archon controller, assuming it has been configured
     * correctly the RawImageData has already been assembled into a complete
     * image, so what this class is really doing is de-assembling the image,
     * into the format that it will be generated in by the LSST electronics.
     *
     * @param rawImage The raw image data from the archon
     * @param geom The CCDGeometry
     * @param treatOverAndUnderscanAsActive If <code>true</code> then
     * appropriate headers are modified to treat all pixels as active. Note that
     * files written in this way are not compliant with LCA-10140.
     */
    RawImageConverter(RawImageData rawImage, CCDGeometry geom, boolean treatOverAndUnderscanAsActive) {

        this.rawImage = rawImage;
        this.geom = geom;
        assert (rawImage.getData().length == rawImage.getSample().bytes() * geom.getTotalSerialSize() * geom.getTotalParallelSize());
        Map<String, Object> metaData = new HashMap<>();
        metaData.put("ArchonSample", rawImage.getSample());
        metaData.put("ArchonFrameMode", rawImage.getFrameMode());
        metaData.put("ArchonTimestamp", rawImage.getTimestamp());
        metaData.putAll(geom.getPrimaryHeaders());
        imageSet = new DefaultImageSet(metaData);

        ByteBuffer raw = ByteBuffer.wrap(rawImage.getData());
        raw.order(ByteOrder.LITTLE_ENDIAN);
        ShortBuffer shortRaw = raw.asShortBuffer();

        CCDTransform global = geom.getGlobalTransform();

        long start = System.currentTimeMillis();
        for (CCDSegment segment : geom.getSegments()) {
            MetaDataSet metaDataSet = new MetaDataSet();
            final Map<String, Object> imageMetaData = segment.getSegmentHeaders(treatOverAndUnderscanAsActive);
            analyzeImage(global, shortRaw, segment, imageMetaData);
            metaDataSet.addMetaData("channel", imageMetaData);
            imageSet.addImage(geom.getTotalSerialCount(), geom.getTotalParallelCount(), metaDataSet);
        }
        long stop = System.currentTimeMillis();
        logger.log(Level.FINE, "Analyzed image in {0}ms", (stop - start));
    }

    public double getImageAverageSignal() {
        return imageAverageSignal;
    }

    ImageSet getImageSet() {
        return imageSet;
    }

    private void analyzeImage(CCDTransform global, ShortBuffer shortRaw, CCDSegment segment, Map<String, Object> imageMetaData) {
        double pix_sum_active = 0.;
        double pix_sum_sq_active = 0.;
        int npix_active = 0;
        double pix_sum_bias = 0.;
        double pix_sum_sq_bias = 0.;
        int npix_bias = 0;

        for (int col = 0; col < geom.getTotalParallelCount(); col++) {
            for (int row = 0; row < geom.getTotalSerialCount(); row++) {

                int location = getRawLocation(segment, row, col);
                int pixval = shortRaw.get(location) & 0xffff;
                global.setSegmentSerialParallel(segment, row, col);
                PixelType type = global.getPixelType();
                if (type == PixelType.ACTIVE) {
                    pix_sum_active += pixval;
                    pix_sum_sq_active += (double) pixval * pixval;
                    npix_active++;
                } else if (type == PixelType.SERIAL_OVERSCAN) {
                    pix_sum_bias += pixval;
                    pix_sum_sq_bias += (double) pixval * pixval;
                    npix_bias++;
                }
            }
        }

        if (npix_active > 0) {
            double average = (double) pix_sum_active / npix_active;
            double stdev = Math.sqrt(pix_sum_sq_active / npix_active - (average * average));
            imageMetaData.put("AVERAGE", average);
            imageMetaData.put("STDEV", stdev);
        }
        if (npix_bias > 0) {
            double average = (double) pix_sum_bias / npix_bias;
            double stdev = Math.sqrt(pix_sum_sq_bias / npix_bias - (average * average));
            imageMetaData.put("AVGBIAS", average);
            imageMetaData.put("STDVBIAS", stdev);
        }
    }

    /**
     * Find the location in the raw data of a particular pixel
     *
     * @param segment The CCD segment containing the pixel
     * @param x
     * @param y
     * @return The location in the raw data array
     */
    private int getRawLocation(CCDSegment segment, int x, int y) {
        int l = 0;
        l += (segment.getCCDGeometry().getSegmentRowCount() - 1 - segment.getRow()) * geom.getTotalSerialCount();
        l += segment.getColumn() * geom.getTotalParallelCount() * geom.getTotalSerialSize();
        if (!segment.getReadout().isDown()) {
            l += geom.getTotalSerialCount() - 1;
        }
        if (segment.getReadout().isRight()) {
            l += geom.getTotalParallelCount() * geom.getTotalSerialSize() - geom.getTotalSerialSize();
        }
        l += !segment.getReadout().isDown() ? -x : x;
        l += geom.getTotalSerialSize() * (segment.getReadout().isRight() ? -y : y);
        assert (l >= 0 && l < geom.getTotalSerialSize() * geom.getTotalParallelSize());
        return l;
    }

    void pushDataToFile(FitsFileWriter writer) throws IOException {
        long start = System.currentTimeMillis();
        ByteBuffer raw = ByteBuffer.wrap(rawImage.getData());
        raw.order(ByteOrder.LITTLE_ENDIAN);
        ShortBuffer shortRaw = raw.asShortBuffer();

        ByteBuffer dest = ByteBuffer.allocate(BUFFER_SIZE);
        dest.order(ByteOrder.BIG_ENDIAN);

        for (CCDSegment segment : geom.getSegments()) {
            for (int col = 0; col < geom.getTotalParallelCount(); col++) {
                for (int row = 0; row < geom.getTotalSerialCount(); row++) {
                    int location = getRawLocation(segment, row, col);
                    dest.putShort((short) (shortRaw.get(location) - 32768));
                    if (!dest.hasRemaining()) {
                        dest.flip();
                        writer.write(segment.getChannel() - 1, dest);
                        dest.clear();
                    }
                }
            }
            dest.flip();
            writer.write(segment.getChannel() - 1, dest);
            dest.clear();
        }
        long stop = System.currentTimeMillis();
        logger.log(Level.FINE, "Wrote image in {0}ms", (stop - start));
    }
}
