package org.lsst.ccs.subsystem.archon;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Field;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.stream.Collectors;

import nom.tam.fits.FitsException;

import org.astrogrid.samp.client.SampException;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.drivers.archon.ArchonController;
import org.lsst.ccs.drivers.archon.ArchonControllerDriver;
import org.lsst.ccs.drivers.archon.ArchonStatus;
import org.lsst.ccs.drivers.archon.ArchonStatus.BoardStatus;
import org.lsst.ccs.drivers.archon.DummyArchonController;
import org.lsst.ccs.drivers.archon.RawImageData;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.messaging.StatusAggregator;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.subsystem.archon.data.ArchonConfiguration;
import org.lsst.ccs.subsystem.archon.data.ArchonControllerStatus;
import org.lsst.ccs.subsystem.archon.data.ArchonControllerStatus.ArchonBoardStatus;
import org.lsst.ccs.utilities.ccd.CCDGeometry;
import org.lsst.ccs.utilities.ccd.CCDType;
import org.lsst.ccs.utilities.image.HeaderSpecification;
import org.lsst.ccs.utilities.image.HeaderSpecification.HeaderLine;
import org.lsst.ccs.utilities.image.MetaDataSet;
import org.lsst.ccs.utilities.image.samp.SampUtils;
import org.lsst.ccs.utilities.logging.Logger;

public class ArchonTS extends Module implements StatusMessageListener, AgentPresenceListener {

    private final ArchonController ctl;
    private ArchonConfiguration cf;

    private final StatusAggregator sa = new StatusAggregator();
    private final FitsUtilities fitsUtils = new FitsUtilities();

    private String fitsDirectory = ".";
    private String fitsFileName = "ArchonImageFile_${timestamp}.fits";
    private ImageHandler imageHandler;
    private Properties headerProperties;
    private final Map<String, Object> headerOverides = new HashMap<>();
    private final Map<String, Object> otherMetaData = new HashMap<>();
    private boolean sendImagesToDS9 = false;
    private SampUtils su;
    private double imageAverageSignal = 0.0;
    private CCDType defaultCCDType;
    private CCDGeometry ccdGeometry;

    private String teststand_dest = "ts2";

    private long fetch_timeout = 200000;

    @SuppressWarnings("hiding")
    // hiding deprecated logger
    protected static Logger log = Logger.getLogger("org.lsst.ccs.framework");

    // parameter driving the image acquisition
    // for the expose commands to work, the timing script must
    // be idle when this parameter is zero, and take an exposure
    // and decrement if >0.
    private String acqParam = "Nexpo";
    private String lastTestType;
    private String configFileName;
    private String fitsCCDnum = "";
    private String fitsLSSTnum = "";

    public ArchonTS(String name, int tickMillis, String ip) {
        super(name, tickMillis);

        // TODO: Initialization of hardware should not be done in constructor
        try {
            if (ip == null) {
                ctl = new DummyArchonController();
            } else {
                ctl = new ArchonControllerDriver(ip);
            }

            // TODO reset the controller? Move some init code into initModule?
        } catch (DriverException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void initModule() {

        imageHandler = new ImageHandler();

        for (HeaderSpecification spc : imageHandler.getConfig().values()) {
            for (HeaderLine l : spc.getHeaders()) {
                String meta = l.getMetaName();
                if (meta != null && meta.contains("/")) {
                    log.info("adding " + meta + " to agregator");
                    sa.setAggregate(meta, -1, -1); // we just want the last
                    // value, no
                    // aggregate/history.
                }
            }
        }

        headerProperties = BootstrapResourceUtils
                .getBootstrapProperties("header.properties");

        teststand_dest = System.getProperty("lsst.ccs.teststand.tsguidest",
                "ts2");

    }

    @Override
    public void start() {
        Predicate<BusMessage<? extends Serializable, ?>>  allStatusSubsystemDataFilter = BusMessageFilterFactory
                .messageClass(StatusSubsystemData.class);
        getSubsystem().getMessagingAccess().addStatusMessageListener(sa,
                allStatusSubsystemDataFilter);
        Predicate<BusMessage<? extends Serializable, ?>>   filter = BusMessageFilterFactory.messageOrigin(teststand_dest).
                and(BusMessageFilterFactory.messageClass(StatusSubsystemData.class));
        getSubsystem().getMessagingAccess().addStatusMessageListener(this, filter);

    }

    @Override
    public void tick() {
        if (inExposureSeries) {
            return;
        }
        ArchonControllerStatus st = getControllerStatus();
        log.info("sending status, backplane temp = " + st.getBackplaneTemp());
        KeyValueData kd = new KeyValueData("archonControllerStatus", st);
        getSubsystem().publishSubsystemDataOnStatusBus(kd);

    }

    @Command
    public void setConfigFromFile(String fn) {
        try {
            cf = new ArchonConfiguration(fn);
            log.info("config read");
            configFileName = fn;
        } catch (IOException e) {
            log.error("error reading config", e);
        }
    }

    @Command
    public void setAndApplyConfig(ArchonConfiguration c) {
        setConfig(c);
        applyConfig();
    }

    @Command
    public void setAndApplyParams(ArchonConfiguration c) {
        setConfig(c);
        applyParams();
    }

    @Command
    public void saveACF(String fileName) throws IOException {
        try (BufferedWriter wtr = new BufferedWriter(new FileWriter(fileName))) {
            cf.saveACF(wtr);
        }
    }

    @Command
    public ArchonConfiguration getConfig() {
        return cf;
    }

    @Command
    public CCDType getDefaultCCDType() {
        return defaultCCDType;
    }
    
    /**
     * Overrides the default CCDType and forces the given CCDGeometry to be used.
     * Note that setting this to a non-null value will override and automatic determination
     * of the geometry (including regions of interest) from the archon (acf) file.
     * @param ccdGeom The geometry to use, or <code>null</code> to restore default geometry.
     */
    @Command
    public void setCCDGeometry(CCDGeometry ccdGeom) {
        this.ccdGeometry = ccdGeom;
    }

    @Command
    public void setCCDOverScans(int number_overscan_lines) {
	setCCDGeometry(getCCDType().getGeometry().withOverscan(number_overscan_lines));
    }

    @Command
    public void setDefaultCCDType(CCDType ccdType) {
        this.defaultCCDType = ccdType;
    }

    @Command
    public void setDefaultCCDTypeName(String ccdType) {
        this.defaultCCDType = CCDType.valueOf(ccdType);
    }

    @Command
    public CCDType getCCDType() {
        if (defaultCCDType == null) {
            if (cf != null) {
                String parameter = cf.getParameter("CCDType");
                if ("1".equals(parameter)) {
                    return CCDType.ITL;
                } else {
                    return CCDType.E2V;
                }
            } else {
                return CCDType.E2V;
            }
        } else {
            return defaultCCDType;
        }
    }

    @Command
    public void setConfig(ArchonConfiguration c) {
        cf = c;
        log.info("config updated");

    }

    @Command
    public void showPhotoDiodeAnalysis(String pddatfile)
            throws IOException, FitsException {
        fitsUtils.showPhotoDiodeAnalysis(new File(pddatfile));
    }

    @Command
    public void addBinaryTable(String pddatfile, String fitsfile,
            String extnam, String c1name, String c2name, double tstart)
            throws IOException, FitsException {
        fitsUtils.updatePhotoDiodeValues(new File(pddatfile),
                new File(fitsfile), extnam, c1name, c2name, tstart);
    }

    @Command
    public double getFluxStats(String fitsfile) throws FitsException,
            IOException {
        return (fitsUtils.getFluxStats(new File(fitsfile)));
    }

    @Command
    public void setAcqParam(String acqParam) {
        this.acqParam = acqParam;
    }

    public String getAcqParam() {
        return acqParam;
    }

    @Command(description = "Get the archon controller status")
    public ArchonControllerStatus getControllerStatus() {
        log.info("get controller status");
        return buildStatus();
    }

    @Command(description = "Test if images will be sent to DS9 automatically")
    public boolean isSendImagesToDS9() {
        return sendImagesToDS9;
    }

    @Command(description = "Set if images should be sent to DS9 automatically")
    public void setSendImagesToDS9(boolean sendImagesToDS9) {
        this.sendImagesToDS9 = sendImagesToDS9;
    }

    // the two structures, ArchonStatus (driver-level) and
    // ArchonControllerStatus (subsystem-level)
    // are very similar, but this is not necessarily always the case, and they
    // could evolve
    // independently. Furthermore, clients of the subsystem do not have to
    // depend on the driver.
    // TODO : create generic code to copy fields, not properties
    // (a la facon de l'apache commons beanutils but with fields).
    private ArchonControllerStatus buildStatus() {
        try {
            ArchonStatus s = ctl.getStatus();
            ArchonControllerStatus acs = new ArchonControllerStatus();

            copySimpleFields(s, acs);
            ArchonBoardStatus[] abs = new ArchonBoardStatus[12];

            for (int i = 0; i < s.boards.length; i++) {
                BoardStatus bs = s.boards[i];

                if (bs != null) {
                    switch (bs.getClass().getSimpleName()) {
                        case "ADStatus":
                            abs[i] = new ArchonControllerStatus.ArchonADBoardStatus();
                            break;
                        case "DriverStatus":
                            abs[i] = new ArchonControllerStatus.ArchonDriverBoardStatus();
                            break;
                        case "HeaterStatus":
                            abs[i] = new ArchonControllerStatus.ArchonHeaterBoardStatus();
                            break;
                        case "HVBiasStatus":
                            abs[i] = new ArchonControllerStatus.ArchonHVBiasBoardStatus();
                            break;
                        case "LVBiasStatus":
                            abs[i] = new ArchonControllerStatus.ArchonLVBiasBoardStatus();
                            break;
                    }
                }
                if (abs[i] != null) {
                    copySimpleFields(bs, abs[i]);

                }
            }

            Field f = ArchonControllerStatus.class.getDeclaredField("boards");
            f.setAccessible(true);
            f.set(acs, abs);

            return acs;
        } catch (DriverException | NoSuchFieldException | SecurityException | IllegalArgumentException | IllegalAccessException e) {
            log.error("error getting status", e);
            throw new RuntimeException(e);
        }
    }

    private void copySimpleFields(Object source, Class srcClass, Object dest,
            Class destClass) {
        for (Field f : srcClass.getDeclaredFields()) {

            // try catch not the best solution, expensive. Map of fields
            // would be better
            try {
                Field df = destClass.getDeclaredField(f.getName());
                if (df == null) {
                    continue;
                }
                log.debug("copying " + f.getName());

                // if array, copy it.
                df.setAccessible(true);
                df.set(dest, f.get(source));
            } catch (NoSuchFieldException | SecurityException | IllegalArgumentException | IllegalAccessException e) {
                // continue;
            }
        }
    }

    private void copySimpleFields(Object source, Object dest) {
        Class srcClass = source.getClass();
        Class destClass = dest.getClass();

        while (true) {
            copySimpleFields(source, srcClass, dest, destClass);
            srcClass = srcClass.getSuperclass();
            destClass = destClass.getSuperclass();
            if (srcClass == null || destClass == null) {
                break;
            }
        }
    }

    @Command(description = "Set an archon configuration parameter. Requires applyParams of applyConfig after.")
    public void setParameter(String name, String value) {
        if (cf == null) {
            throw new IllegalArgumentException(
                    "cannot set parameter without having defined a configuration from an acf file ");
        }
        cf.setParameter(name, value);
    }

    @Command(description = "Set an archon configuration parameter and applies the one line change.")
    public void setAndApplyParam(String name, String value)
            throws DriverException {
        if (cf == null) {
            throw new IllegalArgumentException(
                    "cannot set parameter without having defined a configuration from an acf file ");
        }
        int iline = cf.setParameter(name, value);

        System.out.println("writing to line " + iline + ":\n"
                + cf.getLine(iline));

        ctl.writeConfigLine(iline, cf.getLine(iline));

        System.out.println("loading parameter "
                + cf.getLine(iline).split("=")[1]);

        ctl.loadParam(cf.getLine(iline).split("=")[1]);

    }

    @Command(description = "Get an archon configuration parameter")
    public String getParameter(String name) {
        return cf.getParameter(name);
    }

    @Command(description = "Set an archon configuration constant. Requires applyTiming of applyConfig after.")
    public void setConstant(String name, String value) {
        if (cf == null) {
            throw new IllegalArgumentException(
                    "cannot set constant without having defined a configuration from an acf file ");
        }
        cf.setConstant(name, value);
    }

    @Command(description = "Get an archon configuration constant")
    public String getConstant(String name) {
        return cf.getConstant(name);
    }

    int modifiedModuleMask = 0;

    @Command(description = "Set an archon configuration labeled value. Requires applyLabed or applyConfig after.")
    public void setLabeled(String name, String value) {
        if (cf == null) {
            throw new IllegalArgumentException(
                    "cannot set constant without having defined a configuration from an acf file ");
        }
        int i = cf.setLabeled(name, value);
        modifiedModuleMask |= (1 << i);
    }

    @Command(description = "Get current labeled value")
    public String getLabeled(String name) {
        return cf.getLabeled(name);
    }

    @Command(description = "Updates archon configuration with new labeled values")
    public void applyLabeled() {
        try {
            log.info("sending configuration to controller");
            ctl.writeConfigLines(cf.getLines());
            for (int i = 0; i < 16; i++) {
                if ((modifiedModuleMask & (1 << i)) != 0) {
                    log.info("applying module config " + i);
                    ctl.applyMod(i);
                }
            }

            modifiedModuleMask = 0;
        } catch (DriverException e) {
            log.error("error sending config", e);
            throw new RuntimeException(e);
        }

    }

    @Command(description = "Override a FITS header")
    public void setHeader(@Argument(name = "name") String name,
            @Argument(name = "value", defaultValue = "") Object value) {
        if (value == null || value.toString().isEmpty()) {
            headerOverides.remove(name);
        } else {
            headerOverides.put(name, value);
            if (name.equals("TestType")) {
                lastTestType = value.toString();
            }
        }
    }

    @Command(description = "Set the directory into which FITS file will be stored")
    public void setFitsDirectory(String dir) {
        fitsDirectory = dir;
    }

    @Command(type = CommandType.QUERY, description = "retrieve the last acq test type")
    public String getLastTestType() {
        return lastTestType;
    }

    /**
     * Allows the name under which the fits file will be stored. The fileName
     * may contain escape sequences like ${variable}, but currently the only
     * supported variable is ${timestamp}.
     *
     * @param fileName The file name, including escape sequences.
     */
    @Command(description = "Set the file name into which FITS file will be stored")
    public void setFitsFilename(String fileName) {
        fitsFileName = fileName;
    }

    @Command(description = "Set the CCD number")
    public void setCCDnum(String ccdnum) {
        fitsCCDnum = ccdnum;
    }

    @Command(description = "Set the LSST number")
    public void setLSSTnum(String lsstnum) {
        fitsLSSTnum = lsstnum;
    }

    @Command(description = "Send archon configuration to controller and apply all. Resets timing core and powers off the ccd.")
    public void applyConfig() {
        if (cf == null) {
            throw new IllegalArgumentException(
                    "cannot apply config without having defined a configuration from an acf file ");
        }
        double bias = (double) sendSyncTSCommand("Bias", "readVoltage");
        if (Math.abs(bias) > 0.01) {
            log.error("error loading configuration");
            throw new RuntimeException(
                    "Bias on when trying to load configuration");
        } else {

            try {
                log.info("sending configuration to controller");
                ctl.writeConfigLines(cf.getLines());
                log.info("configuration sent to controller");
                ctl.applyAll();
                log.info("configuration applied to controller");

                KeyValueData kd = new KeyValueData("archonConfig", cf);
                getSubsystem().publishSubsystemDataOnStatusBus(kd);

            } catch (DriverException e) {
                log.error("error sending config", e);
                throw new RuntimeException(e);
            }
        }
    }

    @ConfigurationParameterChanger
    @Command(description = "Send archon configuration to controller and apply params")
    public void applyParams() {
        if (cf == null) {
            throw new IllegalArgumentException(
                    "cannot apply params without having defined a configuration from an acf file ");
        }
        try {
            log.info("sending configuration to controller");
            ctl.writeConfigLines(cf.getLines());
            log.info("configuration sent to controller");
            ctl.loadParams();
            log.info("params configuration applied to controller");

            KeyValueData kd = new KeyValueData("archonConfig", cf);
            getSubsystem().publishSubsystemDataOnStatusBus(kd);

        } catch (DriverException e) {
            log.error("error sending params", e);
            throw new RuntimeException(e);
        }
    }

    @Command(description = "Send archon configuration to controller and reload timing scripts. Resets timing core.")
    public void applyTiming() {
        if (cf == null) {
            throw new IllegalArgumentException(
                    "cannot apply params without having defined a configuration from an acf file ");
        }
        try {
            log.info("sending configuration to controller");
            ctl.writeConfigLines(cf.getLines());
            log.info("configuration sent to controller");
            ctl.loadTiming();
            log.info("timing configuration applied to controller");

            KeyValueData kd = new KeyValueData("archonConfig", cf);
            getSubsystem().publishSubsystemDataOnStatusBus(kd);

        } catch (DriverException e) {
            log.error("error sending timing", e);
            throw new RuntimeException(e);
        }
    }

    @Command(description = "Turn on the CCD")
    public void powerOnCCD() {
        double bias = (double) sendSyncTSCommand("Bias", "readVoltage");
        if (Math.abs(bias) > 0.01) {
            log.error("error powering on CCD");
            throw new RuntimeException(
                    "Bias on when trying to power ON the CCD");
        } else {
            try {
                ctl.powerOn();
                log.info("CCD power on");
            } catch (DriverException e) {
                log.error("error powering on CCD", e);
                throw new RuntimeException(e);
            }
        }
    }

    @Command(description = "Turn off the CCD")
    public void powerOffCCD() {
        sendSyncTSCommand("Bias", "setVoltage", 0.);
        log.info("Setting bias voltage off");

        try {
            ctl.powerOff();
            log.info("CCD power off");
        } catch (DriverException e) {
            log.error("error powering off CCD", e);
            throw new RuntimeException(e);
        }
    }

    public RawImageData acquireImage() {
        try {
            RawImageData imageData = ctl.fetchNewImage(fetch_timeout);
            if (inExposure) {
                inExposure = false;
                KeyValueData kd = new KeyValueData("inExposure", 0);
                getSubsystem().publishSubsystemDataOnStatusBus(kd);

            }
            log.info("Image acquired");
            return imageData;
        } catch (DriverException e) {
            log.error("error acquiring image", e);
            throw new RuntimeException(e);
        } catch (TimeoutException e) {
            log.error("timeout acquiring image", e);
            throw new RuntimeException(e);
        }
    }

    volatile boolean inExposure = false;
    volatile boolean inExposureSeries = false;

    @Command(description = "trigger image exposure and readout, and acquire the image w/o applying the whole config")
    public String exposeAcquireAndSaveWithoutApply() throws IOException,
            FitsException, DriverException {

        try {
            ctl.markFrame();
        } catch (DriverException e1) {
            e1.printStackTrace();
        }

        KeyValueData kd = new KeyValueData("inExposure", 1);
        getSubsystem().publishSubsystemDataOnStatusBus(kd);

        inExposure = true;
        // setParameter(acqParam, "1"); // this is what this method doesn't do
        // applyParams(); // ...
        this.setAndApplyParam(acqParam, "1");
        return acquireAndSaveImage();
    }

    @Command(description = "trigger image exposure and readout, and acquire the image")
    public String exposeAcquireAndSave() throws IOException, FitsException {

        KeyValueData kd = new KeyValueData("inExposure", 1);
        getSubsystem().publishSubsystemDataOnStatusBus(kd);

        try {
            ctl.markFrame();
        } catch (DriverException e1) {
            e1.printStackTrace();
        }

        inExposure = true;
        setParameter(acqParam, "1");
        applyParams();
        return acquireAndSaveImage();
    }

    @Command(description = "snap an image (filename, exptime (ms)")
    public String snap(
            @Argument(name = "filename", description = "filename with or without the full path") String fln,
            @Argument(name = "exptime", description = "exposure time in an integer number of msec") int exptime)
            throws IOException, FitsException {
        return (snap(fln, exptime, 0, 0));
    }

    @Command(description = "snap an image (filename, exptime (ms), light(1)/dark(0)")
    public String snap(
            @Argument(name = "filename", description = "filename with or without the full path") String fln,
            @Argument(name = "exptime", description = "exposure time in an integer number of msec") int exptime,
            @Argument(name = "light", description = "light(=1) or dark(=0) exposure") int light)
            throws IOException, FitsException {
        return (snap(fln, exptime, light, 0));
    }

    @Command(description = "snap an image (filename, exptime (ms), light(1)/dark(0), fe55(1),no fe55(0)")
    public String snap(
            @Argument(name = "filename", description = "filename with or without the full path") String fln,
            @Argument(name = "exptime", description = "exposure time in an integer number of msec") int exptime,
            @Argument(name = "light", description = "light(=1) or dark(=0) exposure") int light,
            @Argument(name = "fe55", description = "XED actuated(=1) or XED inactive(=0)") int fe55)
            throws IOException, FitsException {
        // add check on ArchonControllerStatus
        // getSubsystem().getStatus(??)
        /*
        sendSyncTSCommand("Bias", "setOutput", 0);
        this.applyConfig();
        this.powerOnCCD();
        sendSyncTSCommand("Bias", "setOutput", 1);
        */
        String[] splitfln = fln.split("/");
        String fln_name_only = splitfln[splitfln.length - 1];
        System.out.println("The output filename will be: " + fln_name_only);
        if (splitfln.length > 1) {
            String dir = fln.substring(0, fln.length() - fln_name_only.length());
            System.out.println("The output directory will be: " + dir);
            this.setFitsDirectory(dir);
        }
        this.setFitsFilename(fln_name_only);
        this.setAcqParam("Expo");
        this.setParameter("Nexpo", "1");
        this.setParameter("ExpTime", String.valueOf(exptime));
        this.setParameter("Light", String.valueOf(light));
        this.setParameter("Fe55", String.valueOf(fe55));
        String fullfln = exposeAcquireAndSave();
        return "created fits file " + fullfln;
    }

    @Command(description = "perform setup for acquisitions")
    public int eoSetup(
            @Argument(name = "acffile", description = "filename of acf file") String acffile,
            @Argument(name = "ccdtype", description = "filename of acf file") String ccdtype)
            throws IOException, FitsException {

        log.info("test stand in ready state, now the controller will be configured.");
        setDefaultCCDTypeName(ccdtype);

        log.info("initializing archon controller with file " + acffile);
        log.info("Loading configuration file into the Archon controller");
        setConfigFromFile(acffile);
        log.info("Applying configuration");
        applyConfig();
        log.info("Powering on the CCD");
        powerOnCCD();
        try {
            Thread.sleep(30000);
        } catch (InterruptedException ex) {
            log.error("Sleep interrupted: " + ex);
        }
        log.info("set controller parameters for an exposure with the shutter closed");
        this.setAcqParam("Expo");
        this.setParameter("Nexpo", "1");
        this.setParameter("Light", "1");
        this.setParameter("Fe55", "0");

        System.out.println("getting controller parameters");
        System.out.println("BackPlaneID =" + this.getBackPlaneID());
        System.out.println("BackPlaneRev =" + this.getBackPlaneRev());
        System.out.println("BackPlaneType =" + this.getBackPlaneType());
        System.out.println("BackPlaneVersion =" + this.getBackPlaneVersion());

        this.setHeader("BackPlaneID", Long.toString(ctl.getBackPlaneID()));
        this.setHeader("BackPlaneRev", ctl.getBackPlaneRev());
        this.setHeader("BackPlaneType", ctl.getBackPlaneType());
        this.setHeader("BackPlaneVersion", ctl.getBackPlaneVersion());

        setFetch_timeout(500000);
        return (1);
    }

    @Command(description = "trigger image exposure and readout, for a series of images")
    public void exposeAcquireAndSaveSeveral(int n) throws IOException,
            FitsException {

        try {
            ctl.markFrame();
        } catch (DriverException e1) {
            e1.printStackTrace();
        }

        System.out.println("taking exposures: " + n);
        KeyValueData kd = new KeyValueData("inExposure", 1);
        getSubsystem().publishSubsystemDataOnStatusBus(kd);

        inExposure = true;
        inExposureSeries = true;
        setParameter(acqParam, Integer.toString(n));
        applyParams();

        CCDGeometry ccdGeom = getCCDGeometry();

        System.out.println("geometry: " + ccdGeom.getTotalParallelCount() + " "
                + ccdGeom.getTotalSerialCount());

        File dir = new File(fitsDirectory);
        if (!dir.exists() && !dir.mkdirs()) {
            throw new IOException("Unable to create or write to "
                    + fitsDirectory);
        }

        LinkedBlockingQueue<RawImageData> imageQueue = new LinkedBlockingQueue<>();

        Runnable writer = new Runnable() {

            @Override
            public void run() {
                try {
                    for (int i = 0; i < n; i++) {
                        RawImageData d = imageQueue.poll(10, TimeUnit.SECONDS);
                        if (d == null) {
                            log.warn("no image in queue for saving, missed frames?");
                            continue;
                        }

                        log.info("  writer thread: saving frame "
                                + d.getFrame());

                        MetaDataSet metaDataSet = new MetaDataSet();
                        metaDataSet.addProperties("default", headerProperties);
                        metaDataSet.addMetaData("override", headerOverides);
                        Map<String, Object> statusAggregatorMetaData = sa
                                .getAllLast();
                        // FIXME: TJ: This looks wrong, we should define another
                        // map for additional
                        // information, not modify the map returned by the
                        // status aggregator
                        statusAggregatorMetaData
                                .put("ExposureTime",
                                        Double.valueOf(getParameter("ExpTime")) / 1000.);

                        Long now = System.currentTimeMillis();

                        metaDataSet.addMetaData("StatusAggregator",
                                statusAggregatorMetaData);
                        // TODO: This does not seem as if it really belongs
                        // here!
                        for (String k : statusAggregatorMetaData.keySet()) {
                            log.debug("got in SA " + k + " -> "
                                    + statusAggregatorMetaData.get(k));
                        }

                        File outputFile = new File(fitsDirectory, escape(
                                fitsFileName, now));

                        RawImageConverter converter = new RawImageConverter(d,
                                ccdGeom);
                        imageHandler.writeImage(outputFile, converter,
                                metaDataSet);
                        log.info("  writer thread: done saving to "
                                + outputFile);
                    }
                } catch (InterruptedException e) {
                    log.error("timeout exception in write queue", e);
                } catch (IOException | FitsException e) {
                    log.error("IO/FITS exception in write queue", e);
                }

            }
        };

        new Thread(writer).start();

        try {
            ctl.pollOff();
        } catch (DriverException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

        for (int i = 0; i < n; i++) {
            log.info("start aquire image");
            final RawImageData rawImage = acquireImage();
            // exception thrown in acquireImage in case of timeout...
            // timeout will happen with current setup if frames are missed.
            log.info("got frame " + rawImage.getFrame());

            System.out.println("acquired frame, size " + rawImage.getHeight()
                    + " " + rawImage.getWidth());

            // push to writing queue
            imageQueue.add(rawImage);
            log.info("queued for writing frame " + rawImage.getFrame());

        }

        try {
            ctl.pollOff();
        } catch (DriverException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

        inExposureSeries = false;

    }

    protected CCDGeometry getCCDGeometry() {
        if (ccdGeometry != null) {
            return ccdGeometry;
        }
        else {
            CCDGeometry ccdGeom = getCCDType().getGeometry();

            // Handle ROI
            String cols = cf.getParameter("READ_COLS");

            if (cols != null) {
                int roiCols = Integer.parseInt(cols);
                int roiRows = Integer.parseInt(cf.getParameter("READ_ROWS"));
                int roiOvCols = Integer.parseInt(cf.getParameter("OVER_COLS"));
                int roiOvRows = Integer.parseInt(cf.getParameter("OVER_ROWS"));

                // WARNING: Does this really work, see warning in withROIGeometry method
                ccdGeom = ccdGeom.withROIGeometry(roiRows, roiCols, roiOvRows,
                        roiOvCols);
            }
            return ccdGeom;
        }
    }

    @Command(description = "sets ROI colStart, cols, rowStart, rows")
    public void setROI(int colStart, int cols, int rowStart, int rows) {
        int totCols = Integer.parseInt(cf.getParameter("CCD_COLS"));
        int totRows = Integer.parseInt(cf.getParameter("CCD_ROWS"));

        if (colStart < 0 || colStart + cols > totCols || rowStart < 0
                || rowStart + rows > totRows) {
            throw new IndexOutOfBoundsException("ROI out of bounds");
        }

        cf.setParameter("PRE_ROWS", Integer.toString(rowStart));
        cf.setParameter("READ_ROWS", Integer.toString(rows));
        cf.setParameter("POST_ROWS",
                Integer.toString(totRows - rows - rowStart));

        cf.setParameter("PRE_COLS", Integer.toString(colStart));
        cf.setParameter("READ_COLS", Integer.toString(cols));
        cf.setParameter("POST_COLS",
                Integer.toString(totCols - cols - colStart));

        cf.replaceLine("PIXELCOUNT", Integer.toString(cols));
        cf.replaceLine("LINECOUNT", Integer.toString(rows));

        cf.dump();

        applyConfig();
    }

    @Command(description = "acquire an image and save it to the default location")
    public String acquireAndSaveImage() throws IOException, FitsException {
        log.info("Received acquireAndSaveImage command for exptime= "
                + getParameter("ExpTime") + " and filename= " + fitsFileName);
        System.out.println("Received acquireAndSaveImage command for exptime= "
                + getParameter("ExpTime") + " and filename= " + fitsFileName);

        Date obsDate = new Date();
        final RawImageData rawImage = acquireImage();

        MetaDataSet metaDataSet = new MetaDataSet();
        metaDataSet.addProperties("default", headerProperties);
        metaDataSet.addMetaData("override", headerOverides);
        Map<String, Object> statusAggregatorMetaData = sa.getAllLast();

        otherMetaData.put("ExposureTime",
                Double.valueOf(getParameter("ExpTime")) / 1000.);
        String[] cfln = configFileName.split("/");
        otherMetaData.put("ConfigFile", cfln[cfln.length - 1]);
        if (!fitsCCDnum.isEmpty()) {
            otherMetaData.put("CCDSerialManufacturer", fitsCCDnum);
        }
        if (!fitsLSSTnum.isEmpty()) {
            otherMetaData.put("LSSTCCDNum", fitsLSSTnum);
        }

        Date fileDate = new Date();
        otherMetaData.put("FileCreationDate", fileDate);
        otherMetaData.put("ObservationDate", obsDate);
        String stampedFilename = "";
        if (!fitsFileName.isEmpty()) {
            stampedFilename = escape(fitsFileName, fileDate.getTime());
            otherMetaData.put("OriginalFileName", stampedFilename);
        }
        metaDataSet.addMetaData("StatusAggregator", statusAggregatorMetaData);
        metaDataSet.addMetaData("OtherAggregator", otherMetaData);
        // TODO: This does not seem as if it really belongs here!
        for (String k : statusAggregatorMetaData.keySet()) {
            log.debug("  got in SA " + k + " -> "
                    + statusAggregatorMetaData.get(k));
        }

        if (!fitsFileName.isEmpty()) {
            File dir = new File(fitsDirectory);
            if (!dir.exists() && !dir.mkdirs()) {
                throw new IOException("Unable to create or write to "
                        + fitsDirectory);
            }
            File outputFile = new File(fitsDirectory, stampedFilename);

            CCDGeometry ccdGeom = getCCDGeometry();

            RawImageConverter converter = new RawImageConverter(rawImage,
                    ccdGeom);
            imageHandler.writeImage(outputFile, converter, metaDataSet);
            log.info("Finished writing image to " + outputFile);

            log.info("Image saved as " + outputFile);
            if (sendImagesToDS9) {
                createSampUtils().display(outputFile);
            }
            imageAverageSignal = converter.getImageAverageSignal();
            return outputFile.getAbsolutePath();
        } else {
            return "";
        }
    }

    @Command(description = "get the average signal of the last image")
    public double getImageAverageSignal() {
        return imageAverageSignal;
    }

    @Command(description = "get the readout rate")
    public double getKpixRate() {
        /*
         * Example:
         * 
         * PARAMETER30="Tsr1=10      # RG high & S1 high & S2 low  & S3 low"
         * PARAMETER31="Tsr2=10      # RG high & S1 high & S2 high & S3 low"
         * PARAMETER32="Tsr3=10      # RG high & S1 low  & S2 high & S3 low "
         * PARAMETER33="Tsr4=10      # RG high & S1 low  & S2 high & S3 high"
         * PARAMETER34
         * ="Tsr5=5       #                S1 low   & S2 low  & S3 high"
         * PARAMETER35="Tsr6=7 # reset int S1 low & S2 low & S3 high & RG low
         * PARAMETER36
         * ="Tsr7=120     #                S1 high  & S2 low  & S3 high"
         * PARAMETER37
         * ="Tsr8=92      #  signal int S1 high  & S2 low  & S3 low  "
         * 
         * LINE121=ReadPixel: LINE122="SR1;SR1(Tsr1)" LINE123="SR2;SR2(Tsr2)"
         * LINE124="SR3;SR3(Tsr3)" LINE125="SR4;SR4(Tsr4)"
         * LINE126="SR5;SR5(Tsr5)" LINE127=PCLK LINE128=NOPCLK
         * LINE129="SR6;SR6(Tsr6)" LINE130="SR7;SR7(Tsr7)"
         * LINE131="SR8;SR8(Tsr8)" LINE132="KEEPER; RETURN ReadPixel"
         */
        double kpix = 0.0;
        final double TCLK = 10.0e-9; // s
        double kperiod = (Double.valueOf(getParameter("Tsr1")) + 1.0) * TCLK
                + (Double.valueOf(getParameter("Tsr2")) + 1.0) * TCLK
                + (Double.valueOf(getParameter("Tsr3")) + 1.0) * TCLK
                + (Double.valueOf(getParameter("Tsr4")) + 1.0) * TCLK
                + (Double.valueOf(getParameter("Tsr5")) + 1.0) * TCLK
                + (Double.valueOf(getParameter("Tsr6")) + 1.0) * TCLK
                + (Double.valueOf(getParameter("Tsr7")) + 1.0) * TCLK
                + (Double.valueOf(getParameter("Tsr8")) + 1.0) * TCLK + 3.0
                * TCLK;
        kpix = 1.0 / kperiod;
        return (kpix);
    }

    @Command
    public long getBackPlaneID() {
        return (ctl.getBackPlaneID());
    }

    @Command
    public int getBackPlaneRev() {
        return (ctl.getBackPlaneRev());
    }

    @Command
    public int getBackPlaneType() {
        return (ctl.getBackPlaneType());
    }

    @Command
    public String getBackPlaneVersion() {
        return (ctl.getBackPlaneVersion());
    }

    @Command(description = "Send a set command to DS9")
    public void ds9Set(String cmd,
            @Argument(name = "url", defaultValue = "") String arg)
            throws URISyntaxException, SampException {
        URI url = null;
        if (arg != null && !arg.isEmpty()) {
            url = new URI(arg);
        }
        createSampUtils().ds9Set(cmd, url, 10000);
    }

    @Command(description = "Send a get command to DS9")
    public List<String> ds9Get(String cmd) throws SampException {
        List<Object> result = createSampUtils().ds9Get(cmd, 10000);
        return result.stream().map(item -> item.toString())
                .collect(Collectors.toList());
    }

    public List<String> ds9Versions() throws SampException {
        return createSampUtils().getDS9Version();
    }

    private SampUtils createSampUtils() {
        if (su == null) {
            su = new SampUtils("ArchonSubsystem", true);
            // Check if at least one DS9 is available, if not issue warning
            try {
                List<String> dS9Version = su.getDS9Version();
                if (dS9Version.isEmpty()) {
                    log.warning("No DS9 is connected to SAMP hub, please start (or restart) ds9");
                }
            } catch (SampException x) {
                log.log(Level.WARNING, "Error while communicating with DS9", x);
            }
        }
        return su;
    }

    // TODO: Replace, temporary to handle just ${timestamp}. Needs to be
    // generalized.
    private String escape(String fitsFileName, Long timestamp) {
        return fitsFileName.replace("${timestamp}", String.valueOf(timestamp))
                .replace(
                        "${TIMESTAMP}",
                        new SimpleDateFormat("yyyyMMddHHmmss").format(new Date(
                                timestamp)));
    }

    @Command(description = "get the timeout for fetching images in millisecs")
    public long getFetch_timeout() {
        return fetch_timeout;
    }

    @Command(description = "set the timeout for fetching images in millisecs")
    public void setFetch_timeout(long fetch_timeout) {
        this.fetch_timeout = fetch_timeout;
    }

    @Command(description = "is an exposure occuring")
    public boolean isInExposure() {
        return inExposure;
    }

    @Command(description = "wait for an exposure to end")
    public int waitForExpoEnd() {
        int nsec = 0;
        while (true) {
            log.info("inExposure = " + inExposure);
            if (!inExposure) {
                break;
            }
            try {
                Thread.sleep(1000); // wait a second and then recheck
            } catch (InterruptedException ex) {
                log.error("Sleep interrupted", ex);
            }
            nsec++;
        }
        return (nsec);
    }

    protected Object sendSyncTSCommand(String target, String name,
            Object... params) {
        String dest = target == null ? teststand_dest : teststand_dest + "/"
                + target;

        CommandRequest cmd = new CommandRequest(dest, name, params);

        ConcurrentMessagingUtils agent = new ConcurrentMessagingUtils(
                getSubsystem().getMessagingAccess());
        try {
            return agent.sendSynchronousCommand(cmd, Duration.ofMillis(10000));
        } catch (Exception e) {
            // throw new RuntimeException("error invoking synchronous command", e);
            log.warning("Unable to perform jgroup communication with destination + "
                    + teststand_dest + " - Exception " + e);
            return null;
        }

    }

    @Override
    public void connecting(AgentInfo agent) {
        if (agent.getName().equals(teststand_dest)) {
            log.info("Connecting to agent " + agent.getName());
        }
    }

    @Override
    public void disconnecting(AgentInfo agent) {
        log.warning("Disconnecting from agent " + agent.getName());
    }

    @Override
    public void onStatusMessage(StatusMessage msg) {

        try {
            StatusSubsystemData ssd = (StatusSubsystemData) msg;
            Object msgObject = ssd.getSubsystemData();
            if (msgObject instanceof KeyValueData) {
                KeyValueData d = (KeyValueData) msgObject;
                log.info("In onDataArrival method. KEY=" + d.getKey());
                if (d.getKey().equals("TS_Hazard")) {
                    log.error("Heard TS_Hazard. Powering OFF CCD to protect it.");
                    log.error("Hazard = " + d.getValue());
                    this.powerOffCCD();
                }

            }

        } catch (RuntimeException e) {
            if (!e.toString().contains("de-serializing")) {
                log.info("Problem unpacking message:" + e);
            }
        }

    }
}
