package org.lsst.ccs.subsystem.archon;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import nom.tam.fits.FitsException;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.utilities.image.FitsFileWriter;
import org.lsst.ccs.utilities.image.FitsFileWriter.BitsPerPixel;
import org.lsst.ccs.utilities.image.HeaderSpecification;

/**
 * Code used for image handling in the Archon controller. Made into separate
 * class to aid in testing.
 *
 * @author tonyj
 */
class ImageHandler {

    private final Map<String, HeaderSpecification> imageWriterConfig = new HashMap<>();

    ImageHandler() {
        // Read the specification files for fits file headers
        readHeaderSpecification("primary");
        readHeaderSpecification("extended");
        readHeaderSpecification("test_cond");
        readHeaderSpecification("ccd_cond");
        readHeaderSpecification("photodiode_cond");
        readHeaderSpecification("photodiode2_cond");
    }

    Map<String, HeaderSpecification> getConfig() {
        return Collections.unmodifiableMap(imageWriterConfig);
    }
    
    private void readHeaderSpecification(String name) throws RuntimeException {
        final String fileName = name + ".spec";
        try (InputStream bootstrapResource = BootstrapResourceUtils
                .getBootstrapResource(fileName)) {
            if (bootstrapResource == null) {
                throw new RuntimeException(
                        "Failed to open bootstrap resource: " + fileName);
            }
            HeaderSpecification desc = new HeaderSpecification(name,
                    bootstrapResource);
            imageWriterConfig.put(name, desc);
        } catch (IOException x) {
            throw new RuntimeException("Error reading " + fileName, x);
        }
    }

    void writeImage(File file, RawImageConverter imageConverter, Map<String, Map<String, Object>> metaDataMap) throws IOException, FitsException {
        try (FitsFileWriter writer = new FitsFileWriter(file, imageConverter.getImageSet(), metaDataMap, imageWriterConfig, BitsPerPixel.BIT16)) {
            imageConverter.pushDataToFile(writer);
        }    
    }
}
