package org.lsst.ccs.subsystem.archon;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.nio.file.StandardCopyOption;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.List;
import nom.tam.fits.BasicHDU;
import nom.tam.fits.BinaryTable;
import nom.tam.fits.BinaryTableHDU;
import nom.tam.fits.Fits;
import nom.tam.fits.FitsException;
import nom.tam.fits.Header;
import nom.tam.util.BufferedFile;
import org.lsst.ccs.utilities.image.FitsCheckSum;

/**
 * Some utilities for dealing with test stand fits files.
 *
 * @author homer
 * @author tonyj
 */
public class FitsUtilities {

    static class PDValue {

        private double time;
        private double value;

        private PDValue(double time, double value) {
            this.time = time;
            this.value = value;
        }

        public double getTime() {
            return time;
        }

        public double getValue() {
            return value;
        }
    }

    List<PDValue> readPhotoDiodeFile(File fileName) throws FileNotFoundException, IOException {
        try (BufferedReader in = new BufferedReader(new FileReader(fileName))) {
            List<PDValue> result = new ArrayList<>();
            for (;;) {
                String line = in.readLine();
                if (line == null) {
                    break;
                }
                String[] tokens = line.split(" ");
                double time = Double.parseDouble(tokens[0]);
                double value = Double.parseDouble(tokens[1]);
                result.add(new PDValue(time, value));
            }
            return result;
        }
    }

    double analyzePhotoDiodeValues(List<PDValue> pddata) {
        double pdsum = 0.0;
        for (PDValue data : pddata) {
            pdsum += data.getValue();
        }
        double pdavg_all = pdsum / pddata.size();
        System.out.println("Average of all PD values is " + pdavg_all);
        pdsum = 0.0;
        int nelem = 0;
        for (PDValue data : pddata) {
            double pdval = data.getValue();
            if (Math.abs(pdval) > Math.abs(pdavg_all)) {
                pdsum += pdval;
                nelem++;
            }
        }
        double pdavg_expo = pdsum / nelem;
        System.out.println("Number elements in luminous curve region = " + nelem);
        System.out.println("Sum of PD values in luminous region = " + pdsum);
        System.out.println("PD average value during exposure = " + pdavg_expo);
        return pdavg_expo / 1.e-9; // convert to nA
    }

    private BinaryTableHDU createPhotoDiodeHDU(List<PDValue> data,String extnam, String c1name, String c2name, double tstart) throws FitsException {
        BinaryTable binTable = new BinaryTable();
        double[] times = new double[data.size()];
        double[] values = new double[data.size()];
        int i = 0;
        for (PDValue datum : data) {
            times[i] = datum.getTime();
            values[i] = datum.getValue();
            i++;
        }
        binTable.addColumn(times);
        binTable.addColumn(values);

        Header header = new Header(binTable);
        
        header.addValue("EXTNAME", extnam,"Name of the extension");
        header.addValue("TSTART", tstart, "Time of Start of Readings");

        final BinaryTableHDU bhdu = new BinaryTableHDU(header, binTable);
        bhdu.setColumnName(0, c1name, "time");
        bhdu.setColumnName(1, c2name, "values");
        bhdu.info(System.out);
        return bhdu;
    }

    /**
     * Update an existing files by adding or replacing the extension HDU
     * containing the binary table of photo diode values. Note this routine
     * currently works by copying the entire file to a temporary file, then
     * renaming the temporary file over the top of the original file.
     *
     * @param pddatfile The file containing the photo diode data
     * @param fitsfile The fits file to update
     * @param extnam The extension name to add/update
     * @param c1name The name for the first column of data
     * @param c2name The name for the second column of data
     * @param tstart The start time of the readings?
     * @throws java.io.IOException
     * @throws nom.tam.fits.FitsException
     */
    public void updatePhotoDiodeValues(File pddatfile, File fitsfile, String extnam, String c1name, String c2name, double tstart) throws IOException, FitsException {

        List<PDValue> data = readPhotoDiodeFile(pddatfile);
        double pdavg_expo = analyzePhotoDiodeValues(data);
        BinaryTableHDU bhdu = createPhotoDiodeHDU(data,extnam,c1name,c2name,tstart);

        File tmpName = File.createTempFile("tmp", "fits", fitsfile.getParentFile());
        try (Fits in = new Fits(fitsfile)) {
            try (BufferedFile out = new BufferedFile(tmpName, "rw")) {
                in.read();
                BasicHDU primary = in.getHDU(0);
                if (primary.getHeader().containsKey("MONDIODE")) {
                    primary.getHeader().deleteKey("MONDIODE");
                }
                primary.getHeader().addValue("MONDIODE", pdavg_expo, "avg PD value (nA) during expo");
                FitsCheckSum.setChecksum(primary);
                primary.write(out);
                for (int ihdu = 1; ihdu < in.getNumberOfHDUs(); ihdu++) {
                    final BasicHDU hdu = in.getHDU(ihdu);
                    String extn = hdu.getHeader().getStringValue("EXTNAME");
                    if (!extn.equals(extnam)) {
                        hdu.write(out);
                    }
                }
                FitsCheckSum.setChecksum(bhdu);
                bhdu.write(out);
            }
        }
        Files.move(tmpName.toPath(), fitsfile.toPath(), StandardCopyOption.REPLACE_EXISTING);
    }

    /**
     * Compute the flux statistics by reading an existing FITS file
     *
     * @param fitsfile The file to read
     * @return The computed flux statistics.
     * @throws nom.tam.fits.FitsException If the fits file can not be read
     * @throws java.io.IOException If an IO exception occurs
     */
    public double getFluxStats(File fitsfile) throws FitsException, IOException {

        try (Fits fits = new Fits(fitsfile)) {
            double minflux = Double.MAX_VALUE;
            double fluxsum = 0.0;
            int fluxentries = 0;
            
            fits.read();
            final Header primary = fits.getHDU(0).getHeader();
            double gain = primary.getDoubleValue("CCDGAIN");
            double exptime = fits.getHDU(0).getHeader().getDoubleValue("EXPTIME");

            for (int ihdu = 1; ihdu < fits.getNumberOfHDUs(); ihdu++) {
                final Header extension = fits.getHDU(ihdu).getHeader();
                String extn = extension.getStringValue("EXTNAME");
                if (extn.contains("Segment")) {
                    double avg = fits.getHDU(ihdu).getHeader().getDoubleValue("AVERAGE"); // data region only
                    double bias = fits.getHDU(ihdu).getHeader().getDoubleValue("AVGBIAS"); // bias region only
                    double signal = (avg - bias) * gain;         // in electrons
                    double flux = Math.max(signal / exptime, 1.0);  // in electrons/second (can't be < 0 !)
                    minflux = Math.min(minflux, flux);
                    fluxsum += flux;
                    fluxentries += 1.0;
                    System.out.println(extn + " signal = " + signal + " exptime = " + exptime + " flux = " + flux + " minflux for CCD = " + minflux);
                }
            }

            return fluxsum / fluxentries;
        }
    }
}
