package org.lsst.ccs.subsystem.archon;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.archon.RawImageData;
import org.lsst.ccs.utilities.image.DefaultImageSet;
import org.lsst.ccs.utilities.image.FitsFileWriter;
import org.lsst.ccs.utilities.image.ImageSet;
import org.lsst.ccs.utilities.image.MetaDataSet;

/**
 * Convert an Archon RawImage into an ImageSet as expected by the the
 * ImageFileWriter. This currently assumes that the image corresponds to an e2v
 * CCD250, some changes may be needed for use with different CCDs, or if the
 * amount of overscan changes.
 *
 * @author tonyj
 */
public class RawImageConverter {

    private final DefaultImageSet imageSet;
    private final RawImageData rawImage;
    private final static int segmentWidth = 2002;
    private final static int segmentHeight = 512;
    private final static int horizontalSegmentCount = 8;
    private final static int verticalSegmentCount = 2;
    private final static int prescan = 10;
    private final static int overscan = 22;
    private final static int parallelOverscan = 46;
    // For strict compliance with LCA-10140 flipReadoutOrder has to be equal to true.
    // However Jim assures me that things will work just as well (and more efficiantly)
    // with flipReadoutOrder = false
    private final static boolean flipReadoutOrder = false;
    private static final int totalSegmentWidth = segmentWidth + parallelOverscan;
    private static final int totalSegmentHeight = prescan + segmentHeight + overscan;
    private final static int BUFFER_SIZE = 1000000;

    RawImageConverter(RawImageData rawImage) {
        assert (rawImage.data.length == rawImage.mode.bytes()
                * horizontalSegmentCount * verticalSegmentCount
                * totalSegmentHeight * totalSegmentWidth);
        this.rawImage = rawImage;
        Map<String, Object> metaData = new HashMap<>();
        metaData.put("mode", rawImage.mode);
        imageSet = new DefaultImageSet(metaData);

        for (int v = 0; v < verticalSegmentCount; v++) {
            for (int h = 0; h < horizontalSegmentCount; h++) {
                MetaDataSet metaDataSet = new MetaDataSet();
                Map<String, Object> imageMetaData = new HashMap<>();
                int vSegment = getVSegment(h, v);
                int hSegment = getHSegment(h, v);
                imageMetaData.put("EXTNAME", "Segment" + (vSegment) + (hSegment));
                imageMetaData.put("CHANNEL", getChannel(h, v));
                imageMetaData.put("DETSIZE", pair(range(1, segmentHeight * horizontalSegmentCount), range(1, segmentWidth * verticalSegmentCount)));
                imageMetaData.put("DATASEC", pair(range(prescan + 1, segmentHeight + prescan), range(1, segmentWidth)));
                imageMetaData.put("DETSEC", pair(range(segmentHeight * hSegment + 1, segmentHeight * (hSegment + 1), flipReadoutOrder && v == 0),
                        range(segmentWidth * v + 1, segmentWidth * (v + 1), flipReadoutOrder && v != 0)));
                imageMetaData.put("CCDSUM", "1 1");
                metaDataSet.addMetaData("channel", imageMetaData);
                imageSet.addImage(totalSegmentHeight, totalSegmentWidth, metaDataSet);
            }
        }
    }

    private int getChannel(int h, int v) {
        return 1 + v * horizontalSegmentCount + h;
    }

    private int getVSegment(int h, int v) {
        return 1 - v;
    }

    private int getHSegment(int h, int v) {
        return v == 0 ? h : horizontalSegmentCount - h - 1;
    }

    private String pair(String a, String b) {
        return "[" + a + "," + b + "]";
    }

    private String range(int a, int b) {
        return a + ":" + b;
    }

    private String range(int a, int b, boolean flip) {
        return flip ? range(b, a) : range(a, b);
    }

    ImageSet getImageSet() {
        return imageSet;
    }

    void pushDataToFile(FitsFileWriter writer) throws IOException {
        assert(!flipReadoutOrder); // Not currently supported
        ByteBuffer src = ByteBuffer.wrap(rawImage.data);
        ByteBuffer[] dest = new ByteBuffer[horizontalSegmentCount];
        for (int h = 0; h < horizontalSegmentCount; h++) {
            dest[h] = ByteBuffer.allocate(BUFFER_SIZE);
        }
        for (int v = 0; v < verticalSegmentCount; v++) {
            for (int r = 0; r < totalSegmentWidth; r++) {
                for (int h = 0; h < horizontalSegmentCount; h++) {
                    src.limit(src.position() + totalSegmentHeight * rawImage.mode.bytes());
                    if (dest[h].remaining() < src.remaining()) {
                        flush(h, v, dest, writer);
                    }
                    dest[h].put(src);
                }
            }
            for (int h = 0; h < horizontalSegmentCount; h++) {
                flush(h, v, dest, writer);
            }
        }
        assert (src.position() == src.limit());
    }

    private void flush(int h, int v, ByteBuffer[] dest, FitsFileWriter writer) throws IOException {
        int imageId = v == 0 ? horizontalSegmentCount - getChannel(h,v) : getChannel(h, v) - 1;
        dest[h].flip();
        writer.write(imageId, dest[h]);
        dest[h].clear();
    }
}
