package org.lsst.ccs.subsystem.archon;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.TimeoutException;
import nom.tam.fits.FitsException;
import org.lsst.ccs.bootstrap.resources.BootstrapResourceUtils;
import org.lsst.ccs.bus.MessagingFactory;
import org.lsst.ccs.bus.Status;
import org.lsst.ccs.bus.StatusAggregator;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.archon.ArchonController;
import org.lsst.ccs.drivers.archon.ArchonControllerDriver;
import org.lsst.ccs.drivers.archon.ArchonStatus;
import org.lsst.ccs.drivers.archon.DummyArchonController;
import org.lsst.ccs.drivers.archon.RawImageData;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.subsystem.archon.data.ArchonConfiguration;
import org.lsst.ccs.utilities.image.HeaderSpecification;
import org.lsst.ccs.utilities.image.HeaderSpecification.HeaderLine;
import org.lsst.ccs.utilities.image.MetaDataSet;

public class Archon extends Module {

    private final ArchonController ctl;
    private ArchonConfiguration cf;

    private final StatusAggregator sa = new StatusAggregator();
    private String fitsDirectory = ".";
    private ImageHandler imageHandler;
    private Properties headerProperties;
    private final Map<String, Object> headerOverides = new HashMap<>();

    public Archon(String name, int tickMillis, String ip) {
        super(name, tickMillis);

        // TODO: Initialization of hardware should not be done in constructor
        try {
            if (ip == null) {
                ctl = new DummyArchonController();
            } else {
                ctl = new ArchonControllerDriver(ip);
            }

        } catch (DriverException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void initModule() {

        // we use a different name here because for some unknown reason
        // we don't receive the status emitted by the archon controller if we
        // use the same name as archon
        MessagingFactory.getInstance().forSubsystem("archon-sl")
                .addStatusListener(sa);

        imageHandler = new ImageHandler();

        for (HeaderSpecification spc : imageHandler.getConfig().values() ) {
            for (HeaderLine l : spc.getHeaders()) {
                String meta = l.getMetaName();
                if (meta != null && meta.contains("/")) {
                    log.info("adding " + meta + " to agregator");
                    sa.setAggregate(meta, -1, -1); // we just want the last value, no aggregate/history.
                }
            }
        }

        
        headerProperties = BootstrapResourceUtils.getBootstrapProperties("header.properties");

    }

    @Override
    public void tick() {
        try {
            ArchonStatus st = ctl.getStatus();
            log.info("sending status, backplane temp = " + st.backplaneTemp);
            publish("archonStatus", st);

            Status s = new Status();
            s.setSummary("archon tick");
            getSubsystem().broadcastStatus(s);

        } catch (DriverException e) {
            log.error("error reading status", e);
        }
    }

    @ConfigChanger
    public void setConfigFromFile(String fn) {
        try {
            cf = new ArchonConfiguration(fn);
            log.info("config read");
            Status s = new Status();
            s.setSummary("config read");
            getSubsystem().broadcastStatus(s);
        } catch (IOException e) {
            log.error("error reading config", e);
        }
    }

    @ConfigChanger
    public void setAndApplyConfig(ArchonConfiguration c) {
        setConfig(c);
        applyConfig();
    }

    @ConfigChanger
    public void setConfig(ArchonConfiguration c) {
        cf = c;
        log.info("config updated");
        Status s = new Status();
        s.setSummary("config updated");
        getSubsystem().broadcastStatus(s);

    }

    @Command(description = "Get the archon controller status")
    public ArchonStatus getControllerStatus() {
        try {
            log.info("get controller status");
            return ctl.getStatus();
        } catch (DriverException e) {
            log.error("error getting status", e);
            throw new RuntimeException(e);
        }
    }

    @ConfigChanger
    @Command(description = "Set an archon configuration parameter")
    public void setParameter(String name, String value) {
        cf.setParameter(name, value);
    }

    @ConfigChanger
    @Command(description = "Set an archon configuration constant")
    public void setConstant(String name, String value) {
        cf.setConstant(name, value);
    }

    @Command(description = "Overide a FITS header")
    public void setHeader(@Argument(name = "name") String name, @Argument(name = "value", defaultValue = "") String value) {
        if (value == null || value.isEmpty()) {
            headerOverides.remove(name);
        } else {
            headerOverides.put(name, value);
        }
    }

    @ConfigChanger
    @Command(description = "Set the directory into which FITS file will be stored")
    public void setFitsDirectory(String dir) {
        fitsDirectory = dir;
    }

    @ConfigChanger
    @Command(description = "Send archon configuration to controller")
    public void applyConfig() {
        try {
            log.info("sending configuration to controller");
            ctl.writeConfigLines(cf.getLines());
            log.info("configuration sent to controller");
            ctl.applyAll();
            log.info("configuration applied to controller");
            Status s = new Status();
            s.setSummary("config applied");
            getSubsystem().broadcastStatus(s);

            publish("archonConfig", cf);

        } catch (DriverException e) {
            log.error("error sending config", e);
            throw new RuntimeException(e);
        }
    }

    @Command(description = "Turn on the CCD")
    public void powerOnCCD() {
        try {
            ctl.powerOn();
            Status s = new Status();
            s.setSummary("CCD power on");
            getSubsystem().broadcastStatus(s);
        } catch (DriverException e) {
            log.error("error powering on CCD", e);
            throw new RuntimeException(e);
        }
    }

    @Command(description = "Turn off the CCD")
    public void powerOffCCD() {
        try {
            ctl.powerOff();
            Status s = new Status();
            s.setSummary("CCD power off");
            getSubsystem().broadcastStatus(s);
        } catch (DriverException e) {
            log.error("error powering off CCD", e);
            throw new RuntimeException(e);
        }
    }

    public RawImageData acquireImage() {
        try {
            RawImageData imageData = ctl.fetchNewImage(5000);
            Status s = new Status();
            s.setSummary("Image acquired");
            getSubsystem().broadcastStatus(s);
            return imageData;
        } catch (DriverException e) {
            log.error("error acquiring image", e);
            throw new RuntimeException(e);
        } catch (TimeoutException e) {
            log.error("timeout acquiring image", e);
            throw new RuntimeException(e);
        }
    }

    @Command(description = "acquire an image and save it to the hardwired location")
    public void acquireAndSaveImage() throws IOException, FitsException {
        log.info("Received acquireAndSaveImage command");
        String fileNameTemplate = "%s/ArchonImageFile_%d.fits";
        Long timestamp = System.currentTimeMillis();
        String fileName = String.format(fileNameTemplate, fitsDirectory,
                timestamp);

        MetaDataSet metaDataSet = new MetaDataSet();
        metaDataSet.addProperties("default", headerProperties);
        metaDataSet.addMetaData("override", headerOverides);
        Map<String, Object> statusAggregatorMetaData = sa.getAllLast();
        metaDataSet.addMetaData("StatusAggregator", statusAggregatorMetaData);
        // This does not seem as if it really belongs here!
        for (String k : statusAggregatorMetaData.keySet()) {
            log.info("got in SA " + k + " -> " + statusAggregatorMetaData.get(k));
        }

        RawImageConverter converter = new RawImageConverter(acquireImage());
        imageHandler.writeImage(fileName, converter, metaDataSet);
        log.info("Finished writing image to " + fileName);
    }

}
