package org.lsst.ccs.subsystem.archon.data;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Map;
import java.util.Map.Entry;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.lsst.ccs.bus.annotations.SkipEncoding;

// Archon software configuratino
// usually read from a .acf file
// minimally parsed so that information can be updated on the fly
// before it is sent to the controller.

@SkipEncoding
public class ArchonConfiguration implements Serializable {
    ArrayList<String> configLines = new ArrayList<>();

    public String[] getLines() {
        return configLines.toArray(new String[0]);
    }

    public String getLine(int iline) {
        return configLines.get(iline);
    }

    public static class Param implements Serializable {

        public Param(int line, int num, String key, String value) {
            super();
            this.line = line;
            this.num = num;
            this.key = key;
            this.value = value;
        }

        public Param(int line, int num, String key, String key2, String value) {
            super();
            this.line = line;
            this.num = num;
            this.key = key;
            this.key2 = key2;
            this.value = value;
        }

        int line;
        int num;
        String key;
        String key2;
        String value;
    }

    Map<String, Param> constants = new TreeMap<String, Param>();
    Map<String, Param> parameters = new TreeMap<String, Param>();
    Map<String, Param> labeled = new TreeMap<String, Param>();
    Map<String, Param> labeledFull = new TreeMap<String, Param>();

    public ArchonConfiguration() {

    }

    public ArchonConfiguration(File f) throws IOException {
        try (BufferedReader rdr = new BufferedReader(new FileReader(f))) {
            readFromACF(rdr);
        }
    }

    public ArchonConfiguration(String fileName) throws IOException {
        this(new File(fileName));
    }

    // todo : check about similar but more limited function in the driver:
    // remove from driver?

    public final void readFromACF(BufferedReader rdr) throws IOException {
        boolean inConfig = false;

        Pattern pConstant = Pattern
                .compile("CONSTANT([0-9]+)=([A-Za-z0-9_]+)=([^ ]+) *(#.*)?");

        Pattern pParam = Pattern
                .compile("PARAMETER([0-9]+)=([A-Za-z0-9_]+)=([^ ]+) *(#.*)?");

        // MOD4\LVLC_LABEL12=OG
        // MOD4\LVLC_V12=3.0

        Pattern pLabel = Pattern
                .compile("(MOD[0-9]+.[A-Za-z0-9]+)_LABEL([0-9]+)=([A-Za-z0-9]*)");

        while (true) {
            String line = rdr.readLine();
            if (line == null)
                break;
            if (line.startsWith("[CONFIG]")) {
                inConfig = true;
                continue;
            }
            if (line.startsWith("[SYSTEM]")) {
                inConfig = false;
                continue;
            }
            if (!inConfig)
                continue;
            line = line.replaceAll("\"", "");
            line = line.replaceAll("\\\\", "/");

            // todo remove also \t ?

            configLines.add(line);

            // CONSTANT0="P_SLEW=50"

            // Pattern.compile("CONSTANT[0-9]+=[A-Za-z0-9_]+=[^ ]+");

            Matcher m = pConstant.matcher(line);

            if (m.matches()) {
                Param c = new Param(configLines.size() - 1, Integer.parseInt(m
                        .group(1)), m.group(2), m.group(3));
                constants.put(c.key, c);
            } else {
                // PARAMETER0="LCLK=0          # 1-no clocks or 0 clocks"
                // PARAMETER30="Tsr2=10      # RG high & S1 high & S2 high & S3 low"
                m = pParam.matcher(line);

                if (m.matches()) {
                    Param c = new Param(configLines.size() - 1,
                            Integer.parseInt(m.group(1)), m.group(2),
                            m.group(3));
                    parameters.put(c.key, c);
                } else {
                    m = pLabel.matcher(line);
                    if (m.matches() && !m.group(3).equals("")) {
                        // MOD4\LVLC_LABEL12=OG
                        Param c = new Param(-1, Integer.parseInt(m.group(2)),
                                m.group(3), m.group(1), "");
                        // System.out.println("FOR LABEL " + c.key + " " + c.num
                        // + " " + c.key2);
                        labeled.put(c.key, c);
                    }
                }
            }

        }

        Pattern pLabeledEntry = Pattern
                .compile("(MOD[0-9]+.[A-Za-z0-9]+)_([A-Z]+)([0-9]+)=([A-Za-z0-9\\.]*)");

        // complete the label data
        for (int i = 0; i < configLines.size(); i++) {
            String line = configLines.get(i);
            if (line.contains("_LABEL"))
                continue;
            if (line.contains("_ORDER"))
                continue;
            // MOD4\LVLC_V12=3.0
            // MOD4\LVLC_IL12=3.0

            for (Entry<String, Param> e : labeled.entrySet()) {
                Param p = e.getValue();
                if (line.startsWith(p.key2)) {
                    Matcher m = pLabeledEntry.matcher(line);
                    if (m.matches()) {
                        if (Integer.parseInt(m.group(3)) == e.getValue().num) {
                            Param pp = new Param(i, p.num, p.key + "_"
                                    + m.group(2), m.group(1) + "_" + m.group(2)
                                    + m.group(3), m.group(4));
                            labeledFull.put(pp.key, pp);

                            System.out.println("LABEL " + pp.key + " => "
                                    + pp.line + " " + pp.key2 + " = "
                                    + pp.value);
                        }
                    }

                }
            }

        }

    }

    public Set<String> getConstants() {
        return constants.keySet();
    }

    public Set<String> getParameters() {
        return parameters.keySet();
    }

    public String getConstant(String name) {
        Param p = constants.get(name);
        if (p == null)
            return null;
        return p.value;
    }

    public String getParameter(String name) {
        Param p = parameters.get(name);
        if (p == null)
            return null;
        return p.value;
    }

    public int setParameter(String name, String value) {
        Param p = parameters.get(name);
        if (p == null)
            throw new NoSuchElementException();
        p.value = value;
        String line = "PARAMETER" + p.num + "=" + p.key + "=" + p.value;
        configLines.set(p.line, line);
        return p.line;
    }

    public void setConstant(String name, String value) {
        Param p = constants.get(name);
        if (p == null)
            throw new NoSuchElementException();
        p.value = value;
        String line = "CONSTANT" + p.num + "=" + p.key + "=" + p.value;
        configLines.set(p.line, line);
    }

    public int setLabeled(String name, String value) {
        Param p = labeledFull.get(name);
        if (p == null)
            throw new NoSuchElementException();
        p.value = value;
        String line = p.key2 + "=" + p.value;
        System.out.println(configLines.get(p.line) + " -> " + line);
        configLines.set(p.line, line);
        // return the module number
        int i = p.key2.indexOf('/');
        return Integer.parseInt(p.key2.substring(3, i));
    }

    public String getLabeled(String name) {
        Param p = labeledFull.get(name);
        if (p == null)
            return null;
        return p.value;
    }

    public static void main(String[] args) throws IOException {
        System.out.println("reading conf");
        ArchonConfiguration c = new ArchonConfiguration("archon.acf");
        for (String key : c.constants.keySet()) {
            Param p = c.constants.get(key);
            System.out.println("CONSTANT " + p.line + " " + p.num + " " + p.key
                    + " : " + p.value);
        }
        for (String key : c.parameters.keySet()) {
            Param p = c.parameters.get(key);
            System.out.println("PARAMETER " + p.line + " " + p.num + " "
                    + p.key + " : " + p.value);
        }

        c.setConstant("RG_SLEW", "450");
        c.setParameter("LCLK", "1");

        c.dump();

    }

    public void saveACF(BufferedWriter wtr) throws IOException {
        for (int i = 0; i < configLines.size(); i++) {
            wtr.write(configLines.get(i));
            wtr.newLine();
        }
        wtr.flush();
    }

    public void dump() {
        for (int i = 0; i < configLines.size(); i++) {
            System.out.println(i + " : " + configLines.get(i));
        }

    }

    public void replaceLine(String key, String value) {
        String pfx = key + "=";
        for (int i = 0; i < configLines.size(); i++) {
            if (configLines.get(i).startsWith(pfx)) {
                configLines.set(i, key + "=" + value);
                return;
            }
        }
        throw (new NoSuchElementException(key));
    }

}
