package org.lsst.ccs.subsystem.archon.data;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

// Archon software configuratino
// usually read from a .acf file
// minimally parsed so that information can be updated on the fly
// before it is sent to the controller.

public class ArchonConfiguration implements Serializable {
    ArrayList<String> configLines = new ArrayList<>();

    public String[] getLines() {
        return configLines.toArray(new String[0]);
    }

    public static class Param implements Serializable {

        public Param(int line, int num, String key, String value) {
            super();
            this.line = line;
            this.num = num;
            this.key = key;
            this.value = value;
        }

        int line;
        int num;
        String key;
        String value;
    }

    Map<String, Param> constants = new HashMap<>();
    Map<String, Param> parameters = new HashMap<>();

    public ArchonConfiguration() {

    }
    
    public ArchonConfiguration(File f) throws IOException {
        try (BufferedReader rdr = new BufferedReader(new FileReader(f))) {
            readFromACF(rdr);
        }
    }


    public ArchonConfiguration(String fileName) throws IOException {
        this(new File(fileName));
    }

    // todo : check about similar but more limited function in the driver:
    // remove from driver?

    public final void readFromACF(BufferedReader rdr) throws IOException {
        boolean inConfig = false;
        while (true) {
            String line = rdr.readLine();
            if (line == null)
                break;
            if (line.startsWith("[CONFIG]")) {
                inConfig = true;
                continue;
            }
            if (!inConfig)
                continue;
            line = line.replaceAll("\"", "");
            line = line.replaceAll("\\\\", "/");

            configLines.add(line);

            // CONSTANT0="P_SLEW=50"

            // Pattern.compile("CONSTANT[0-9]+=[A-Za-z0-9_]+=[^ ]+");

            Pattern p = Pattern
                    .compile("CONSTANT([0-9]+)=([A-Za-z0-9_]+)=([^ ]+) *(#.*)?");
            Matcher m = p.matcher(line);

            if (m.matches()) {
                Param c = new Param(configLines.size() - 1, Integer.parseInt(m
                        .group(1)), m.group(2), m.group(3));
                constants.put(c.key, c);
            } else {
                p = Pattern
                        .compile("PARAMETER([0-9]+)=([A-Za-z0-9_]+)=([^ ]+) *(#.*)?");
                // PARAMETER0="LCLK=0          # 1-no clocks or 0 clocks"
                // PARAMETER30="Tsr2=10      # RG high & S1 high & S2 high & S3 low"
                m = p.matcher(line);

                if (m.matches()) {
                    Param c = new Param(configLines.size() - 1,
                            Integer.parseInt(m.group(1)), m.group(2),
                            m.group(3));
                    parameters.put(c.key, c);
                }

            }

        }

    }

    public String getConstant(String name) {
        Param p = constants.get(name);
        if (p == null)
            return null;
        return p.value;
    }

    public String getParameter(String name) {
        Param p = parameters.get(name);
        if (p == null)
            return null;
        return p.value;
    }

    public void setParameter(String name, String value) {
        Param p = parameters.get(name);
        if (p == null)
            throw new NoSuchElementException();
        p.value = value;
        String line = "PARAMETER" + p.num + "=" + p.key + "=" + p.value;
        configLines.set(p.line, line);
    }

    public void setConstant(String name, String value) {
        Param p = constants.get(name);
        if (p == null)
            throw new NoSuchElementException();
        p.value = value;
        String line = "CONSTANT" + p.num + "=" + p.key + "=" + p.value;
        configLines.set(p.line, line);
    }

    public static void main(String[] args) throws IOException {
        System.out.println("reading conf");
        ArchonConfiguration c = new ArchonConfiguration("archon.acf");
        for (String key : c.constants.keySet()) {
            Param p = c.constants.get(key);
            System.out.println("CONSTANT " + p.line + " " + p.num + " " + p.key
                    + " : " + p.value);
        }
        for (String key : c.parameters.keySet()) {
            Param p = c.parameters.get(key);
            System.out.println("PARAMETER " + p.line + " " + p.num + " "
                    + p.key + " : " + p.value);
        }

        c.setConstant("RG_SLEW", "450");
        c.setParameter("LCLK", "1");

        c.dump();

    }

    private void dump() {
        for (int i = 0; i < configLines.size(); i++) {
            System.out.println(i + " : " + configLines.get(i));
        }

    }

}
