package org.lsst.ccs.subsystem.airwatch.main;

import java.time.Instant;
import static java.util.Collections.unmodifiableMap;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Represents an instrument cluster connected to a PC running Lighthouse's LMS Express RT program. Immutable.
 * @author tether
 */
public class RestfulLocation implements Location {
    private static final Logger log = Logger.getLogger("org.lsst.ccs.subsystem.airwatch");

    private final boolean enabled;
    private final String location;
    private final Instant lastDataTime;
    private final Map<String, DataPoint> points;

    /**
     * Constructs a new instance from parsed JSON data.
     * @param locData the map of data objects gotten for one location.
     * @param enabled the names of the locations that are enabled.
     */
    public RestfulLocation(final Map<String, Object> locData, final Set<String> enabled) {
        this.location = (String)locData.get("location");
        this.enabled = enabled.contains(this.location);
        this.lastDataTime = Instant.now();
        this.points = new HashMap<>();
        for (final String key: locData.keySet()) {
            if (!key.equals("location")) {
                makeDataPoint(key, (Map<String, Object>)locData.get(key) );
            }
        }
    }

    /**
     * Constructs a new instance using a value specified for each field.
     * @param enabled The counter-is-enabled flag.
     * @param location The name of the location being monitored.
     * @param lastDataTime The most recent timestamp seen.
     * @param points The data points for the location.
     */
    private RestfulLocation(
        final boolean enabled,
        final String location,
        final Instant lastDataTime,
        final Map<String, DataPoint> points
    )
    {
        this.enabled = enabled;
        this.location = location;
        this.lastDataTime = lastDataTime;
        this.points = points;
    }

    /** {@inheritDoc } */
    @Override
    public final LocationStatus getStatus() {
        return new LocationStatus(enabled, location, lastDataTime);
    }

    /** {@inheritDoc } */
    @Override
    public final Map<String, DataPoint> getDataPoints() {
        return unmodifiableMap(points);
    }

    /** {@inheritDoc } */
    @Override
    public final Location disable() {
        return new RestfulLocation(
            false,
            location,
            lastDataTime,
            points
        );
    }

    /** {@inheritDoc } */
    @Override
    public final Location enable() {
        return new RestfulLocation(
            true,
            location,
            lastDataTime,
            points
        );
    }
    
    /** {@inheritDoc } */
    @Override
    public final void checkData(final AlertService alertSvc) {
        for (final String chanName: points.keySet()) {
            final DataPoint p = points.get(chanName);
            p.checkData(alertSvc, location, chanName, this.enabled);
        }
    }

    /** {@inheritDoc }    */
    public final void publishGoodData(final Subsystem subsys) {
        for (final String chanName: points.keySet()) {
            final DataPoint p = points.get(chanName);
            if (p.getQuality().equals("Good") && !p.hasMalfunction()) {
                final KeyValueDataList kvdl = p.makeKvdList(location, chanName);
                subsys.publishSubsystemDataOnStatusBus(p.makeKvdList(location, chanName));
            }
        }
    }

    private void makeDataPoint(final String chanName, final Map<String, Object> chanData){
        if (Character.isDigit(chanName.charAt(0))) {
            points.put(chanName, new CounterPoint(
                (Double)chanData.get("absolute"),
                (Double)chanData.get("density"),  // density
                (Boolean)chanData.get("limitViolation"),
                (Boolean)chanData.get("malfunction"),
                (String)chanData.get("quality"),
                (String)chanData.get("time")
            ));
        }
        else {
            points.put(chanName, new AnalogPoint(
                (Double)chanData.get("value"),
                (Boolean)chanData.get("limitViolation"),
                (Boolean)chanData.get("malfunction"),
                (String)chanData.get("quality"),
                (String)chanData.get("time"),
                (Double)chanData.get("lowBound"),
                (Double)chanData.get("highBound")
        ));
        }
    }
}
