package org.lsst.ccs.subsystem.airwatch.main;

import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.services.alert.AlertService.RaiseAlertStrategy;

/**
 * All the alerts that this subsystem can raise.
 * @author tether
 */
public enum Alerts {

    /**
     * Raised if we can't read sensor data from the Lighthouse PC.
     */
    SENSOR_IO(new Alert("SENSOR_IO", "I/O error when attempting to read sensor data.")),

    /**
     * Raised if the Lighthouse application has flagged a sensor reading as out of bounds.
     */
    LIMIT_VIOLATION(new Alert("LIMIT_VIOLATION", "Sensor measurement is out of bounds.")),

    /**
     * Raised if something went wrong when the sensor tried to make a new measurement.
     */
    DATA_QUALITY(new Alert("DATA_QUALITY", "Quality flag != Good or an instrument malfunction was reported."));
    
    private final Alert ccsAlert;
    
    Alerts(final Alert ccsAlert) {this.ccsAlert = ccsAlert;}
    
    /**
     * Raises the alert associated with this enumeration member.
     * @param alertSvc the CCS alert service.
     * @param severity the state which the raised alert is to have.
     * @param cause the cause string which the raised alert is to have.
     * @param strategy the alert-raising strategy.
     */
    public void raise(
        final AlertService alertSvc,
        final AlertState severity,
        final String cause,
        final RaiseAlertStrategy strategy)
    {
        alertSvc.raiseAlert(ccsAlert, severity, cause, strategy);
    }
    
    /**
     * Register all alert objects, associating the clear-alert handler
     * {@link  org.lsst.ccs.services.alert.AlertService#ALWAYS} with each.
     * @param alertSvc 
     */
    public static void registerAll(final AlertService alertSvc) {
        for (final Alerts a: Alerts.values()) {
            alertSvc.registerAlert(a.ccsAlert, AlertService.ALWAYS);
        }
    }
    
    /**
     * Constructs a cause message of the form<p>
     * LOCATION CHANNEL QUALITY|Data quality is QUALITY for CHANNEL at LOCATION.
     * @param locName the location of the sensor, for example MAIN_NW.
     * @param chanName the sensor channel, for example TEMP.
     * @param quality the data quality estimate, for example Questionable.
     */
    public static String dataQualityMsg(
        final String locName,
        final String chanName,
        final String quality
    )
    {
        return String.format(
            "%1$s %2$s %3$s|Data quality is %3$s for %2$s at %1$s.",
            locName,
            chanName,
            quality
            );
    }
    
    /**
     * Constructs a cause message of the form<p>
     * LOCATION CHANNEL VALUE LOWER UPPER|Limit violation for CHANNEL at LOCATION. Value is VALUE, allowed range is [LOWER, UPPER].
     * @param locName the location of the sensor, for example MAIN_NW.
     * @param chanName the sensor channel, for example TEMP.
     * @param value the sensor reading.
     * @param lowerLimit the lower limit.
     * @param upperLimit the upper limit.
     * @return the formatted message.
     */
    public static String analogViolationMsg(
        final String locName,
        final String chanName,
        final double value,
        final double lowerLimit,
        final double upperLimit
    )
    {
        return String.format(
            "%1$s %2$s %3$.1f %4$.1f %5$.1f|Limit violation for %2$s at %1$s. Value is %3$.1f, allowed range is [%4$.1f, %5$.1f].",
            locName,
            chanName,
            value,
            lowerLimit,
            upperLimit
            );
    }
    
    /**
     * Constructs a cause message of the form<p>
     * LOCATION CHANNEL VALUE |Density of VALUE for particle size CHANNEL at LOCATION is too high.
     * @param locName the location of the sensor, for example MAIN_NW.
     * @param chanName the sensor channel, for example TEMP.
     * @param value the sensor reading.
     * @return the formatted message.
     */
    public static String counterViolationMsg(
        final String locName,
        final String chanName,
        final double value
    )
    {
        return String.format(
            "%1$s %2$s %3$.4g|Density of %3$.4g/ft^3 for particle size %2$s microns at %1$s is too high.",
            locName,
            chanName,
            value
            );
    }

}
