package org.lsst.ccs.subsystem.airwatch.main;

import java.io.IOException;
import java.net.URL;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import static java.util.Objects.requireNonNull;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import static java.util.concurrent.TimeUnit.SECONDS;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import org.lsst.ccs.Subsystem;
import static org.lsst.ccs.bootstrap.BootstrapResourceUtils.getResourceURL;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.data.RunMode;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.command.annotations.Command.CommandType.ACTION;
import static org.lsst.ccs.command.annotations.Command.CommandType.QUERY;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.SignalHandler;
import org.lsst.ccs.framework.SignalLevel;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import static org.lsst.ccs.services.alert.AlertService.RaiseAlertStrategy.ON_SEVERITY_CHANGE;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;

/** The main module for the Airwatch subsystem. This module is responsible for:
 * <ul>
 * <li>Establishing initial contact with the configured instruments.</li>
 * <li>Providing and managing the periodic readout task.</li>
 * <li>Alert management.</li>
 * <li>Location management and status reporting.</li>
 * </ul>
 *  * @author tether
 */
public final class AirwatchMain
extends Subsystem
implements ClearAlertHandler, HasLifecycle, SignalHandler
{
/* This module maintains a shared map of immutable Location objects which describe the state
 * of each set of instrument channels and specify the allowed operations. Mutating operations therefore follow
 * the read-modify-write pattern: an entirely new map is generated by each operation, based on the current
 * list and possibly new sensor data, then the shared map is replaced in its entirety. The map might
 * be changed by a subsystem command or by the task that reads sensor data.
 *
 * Take for example the case of disabling alerts for a previously enabled location with
 * no prior readout data. The disable() command copies the location state and
 * produces a new state with the enabled flag set to false. While that's happening
 * the readout task might copy the same old state and produce a new state with new readout data.
 * Depending on who writes back first either the enable flag will remain true or the
 * new readout data will become inaccessible.
 *
 * For this reason there's an extra lock used to synchronize the readout task with mutating
 * action commands. The readout task will hold the lock while it copies the map, reads sensor data,
 * creates and installs a new map. The mutating action commands must take the lock before doing anything else,
 * delaying the readout task until they finish their execution, which should be brief.
 * They in turn release the lock after completing their changes.
 *
 * The readout task checks for interruption after performing read-and-publish for the map.
 * This is needed to ensure that the location's last-data-time is consistent with
 * what has been posted for trending.
 *
 */

    // How long in ms callIfNotBusy() will wait for the readout lock.
    private static final int BUSY_TIMEOUT = 120_000;

    // Where to get JSON data if CCS run mode is simulation.
    private static final String SIM_DATA_FILE = "simData.json";

    ////////// Commands //////////

    /**
     * Gives the status of each of the locations. Includes instrument number, location,
     * enable flag and the timestamp of the last record read.
     * @return A string with a heading line followed by one line per instrument.
     */
    @Command(type=QUERY, description="Displays a short report on location status.", level=Command.NORMAL)
    public String locations() {return new LocationReport(locations.get()).toString();}

    /**
     * Suppresses data check alerts for a given location. Use if the sensor is broken.
     * @param locName the name of a location. Case is ignored.
     * @return "OK", or "BUSY" if we timed out waiting for the lock.
     * @throws IllegalArgumentException if {@code locName} is invalid.
     */
    @Command(type=ACTION, description="Suppresses counter check alerts for a location. Use if the counter "
        + "is broken.", level=Command.ENGINEERING_ROUTINE,
        timeout=BUSY_TIMEOUT+100)
    public String disable(
        @Argument(description="The location name. Case is ignored.")
        final String locName
    ) throws Exception
    {
        return callIfNotBusy(() -> {
            final Map<String, Location> mylocs = new HashMap<>(locations.get());
            final Location loc = mylocs.get(locName.toUpperCase());
            if (loc == null) {throw new IllegalArgumentException("Invalid location name.");}
            mylocs.put(locName.toUpperCase(), loc.disable());
            locations.set(mylocs);
            return "OK";
        });
    }

    /**
     * Enables data check alerts for a location.
     * @param locName the name of a location. Case is ignored.
     * @return "OK", or "BUSY" if we timed out waiting for the lock.
     * @throws IllegalArgumentException if {@code locName} is invalid.
     */
    @Command(type=ACTION, description="Enables data check alerts for a location.", level=Command.ENGINEERING_ROUTINE,
        timeout=BUSY_TIMEOUT+100)
    public String enable(
        @Argument(description="The location name. Case is ignored.")
        final String locName
    ) throws Exception
    {
        return callIfNotBusy(() -> {
            final Map<String, Location> mylocs = new HashMap<>(locations.get());
            final Location loc = mylocs.get(locName.toUpperCase());
            if (loc == null) {throw new IllegalArgumentException("Invalid location name.");}
            mylocs.put(locName.toUpperCase(), loc.enable());
            locations.set(mylocs);
            return "OK";
        });
    }

    /**
     * Signal handler for stop and abort.
     * @param sig The signal.
     * @return TreeWalkerDiag.GO.
     */
    @Override
    public TreeWalkerDiag signal(Signal sig) {
        if (sig.getLevel() == SignalLevel.STOP) {
            readoutTask.cancel(false); // Don't interrupt any current readout.
        }
        else if (sig.getLevel() == SignalLevel.HALT) {
            readoutTask.cancel(true); // Interrupt any current readout.
        }
        return TreeWalkerDiag.GO;
    }

    ////////// Configuration data //////////

    @ConfigurationParameter(description = "The interval between readings of server data.", units = "s")
    private volatile Duration readoutInterval;

    @ConfigurationParameter(description = "Gives both the valid location names and whether each has "
        + "a working particle counter (1) or a broken one (0).", maxLength=10)
    private volatile Map<String, Integer> workingLocations;

    @ConfigurationParameter(description = "The set of valid channel names.", maxLength=10)
    private volatile List<String> channelNames;

    @ConfigurationParameter(description = "The URL used to contact the RESTful location-data server.")
    private volatile URL restfulUrl;
    
    @ConfigurationParameter(description = "How long to wait for a connection to the server.", units="s")
    private volatile Duration connectionTimeout;
    
    @ConfigurationParameter(description = "How long to wait for data from the server after "
        + "making the connection.", units = "s")
    private volatile Duration readTimeout;

    @ConfigurationParameter(description = "Use these locations, when possible, to calculate "
        + "a dew point temperature for the IR2 main clean room.", maxLength=10)
    private volatile List<String> dewLocations;


    ////////// References to other subsystem components //////////

    @LookupField(strategy = Strategy.TREE)
    private volatile AlertService alertService;


    ////////// Subsystem state data //////////

    /**
     * The descriptions of all locations being managed, indexed by location name. Updated by ACTION
     * commands and the readout task, read by QUERY commands.
     */
    private final AtomicReference<Map<String, Location>> locations;

    /**
     * Held for all read-modify-write operations on the location list.
     */
    private final Lock stateChangeLock;

    /**
     * Regularly scheduled to read the instruments.
     */
    private PeriodicTask readoutTask;

    /**
     * The subsystem's logger.
     */
    private final Logger log;


    ////////// Lifecycle methods //////////
    // Arranged in chronological order of calling.

    /**
     * Constructor.
     */
    public AirwatchMain() {
        // This is a worker subsystem.
        super("airwatch", AgentInfo.AgentType.WORKER);

        alertService = null;
        log = Logger.getLogger(AirwatchMain.class.getName());
        readoutTask = null;
        stateChangeLock = new ReentrantLock();
        locations = new AtomicReference<>();
    }

    /**
     * Checks whether all the subsystem components have been constructed and
     * registers the status bus messages published by the subsystem.
     * Called just after the subsystem component tree has been built.
     * <p>
     * <em>Note that this code will have to be adjusted if the data gotten from the Lighthouse PC changes.</em>
     * @throws Error if anything is missing.
     */
    @Override
    public void init() {
        final List<String> missing = new ArrayList<>();
        if (alertService == null) {
            missing.add("CCS alert service");
        }
        if (!missing.isEmpty()) {
            throw new RuntimeException("Can't find " + String.join(", ", missing));
        }
        final DataProviderDictionaryService dictionary =
                this.getAgentService(DataProviderDictionaryService.class);
        for (final String locName: workingLocations.keySet()) {
            for (final String chanName: channelNames) {
                KeyValueDataList msg;
                // Counter channel names are particle sizes in microns and begin with a digit.
                if (Character.isDigit(chanName.charAt(0))) {
                    msg = new CounterPoint().makeKvdList(locName, chanName);
                }
                else {
                    msg = new AnalogPoint().makeKvdList(locName, chanName);
                }
                dictionary.registerData(msg);
            }
        }
        final KeyValueDataList dew = new KeyValueDataList();
        dew.addData(N_DEW_TEMPS_KEY, 0);
        dew.addData(N_DEW_HUMIDS_KEY, 0);
        dew.addData(DEW_POINT_KEY, 0.0);
        dictionary.registerData(dew);
        
        Alerts.registerAll(alertService);
    }

    @Override
    public void postStart() {
        locations.set(workingLocations.entrySet().stream()
            .map(loc -> new DummyLocation(loc.getKey(), loc.getValue() != 0))
            .collect(toMap(dummy -> dummy.getStatus().location, dummy -> dummy))
        );

        this.readoutTask =
            this
                .getScheduler()
                .scheduleWithFixedDelay(
                    this::readoutTaskBody,
                    0,
                    readoutInterval.toMillis(),
                    TimeUnit.MILLISECONDS);
    }

    private void readoutTaskBody() {
        //  Synchronize with mutating action commands.
        try {
            stateChangeLock.lockInterruptibly();
        }
        catch (InterruptedException exc) {
            Thread.currentThread().interrupt();
            return;
        }

        // Read, update status, post trendables.
        try {
            // Collect the names of enabled locations.
            final Set<String> enabledLocs = locations.get().entrySet().stream()
                .filter( entry -> entry.getValue().getStatus().enabled )
                .map( entry -> entry.getKey() )
                .collect(toSet());
            // Get the data.
            final URL dataSource = 
                RunMode.isSimulation()
                ? requireNonNull(getResourceURL(SIM_DATA_FILE), "Can't find resource " + SIM_DATA_FILE)
                : restfulUrl;
            final LocationSource source =
                new RestfulLocationSource(dataSource, connectionTimeout, readTimeout, enabledLocs);
            // Make a new map of locations.
            final Map<String, Location> newLocs =
                source.getLocations().stream()
                .collect(toMap( loc -> loc.getStatus().location, loc -> loc ));
            // Replace the old location map with the new and publish it to other threads.
            locations.set(newLocs);
            
            // Check for data values out of bounds, data of quality other than "Good", etc.
            // If a location is disabled then we don't post any alerts about it.
            // Good readings, out of bounds or not, are always published.
            for (final Location loc: locations.get().values()) {
                final LocationStatus status = loc.getStatus();
                loc.checkData(alertService);
                loc.publishGoodData(this);
            }
            
            // Calculate and publish a dew point for IR2 main room.
            publishDewPoint();
            
            // Allow some time for mutating commands to get access.
            SECONDS.sleep(2);
        }
        catch (final InterruptedException exc) {
            Thread.currentThread().interrupt();
        }
        catch (final IOException exc) {
            Alerts.SENSOR_IO.raise(alertService, AlertState.WARNING, exc.getMessage(), ON_SEVERITY_CHANGE);
        }
        finally {
            stateChangeLock.unlock();
        }
    }

    ////////// Utilities //////////

    /**
     * Calls a {@code Callable<String>} on behalf of a command that needs to synchronize
     * with the periodic readout. Upon exit the current thread will
     * not hold the readout lock.
     * @param cmd The code to run.
     * @return The result from {@code cmd}.
     * @throws Exception if {@code cmd.call()} throws.
     */
    private String callIfNotBusy(Callable<String> cmd) throws Exception {
        if (!stateChangeLock.tryLock(BUSY_TIMEOUT, TimeUnit.MILLISECONDS)) {
            // Send rejection of the command.
            return "BUSY";
        }
        try {
            // Execute the command.
            return cmd.call();
        } finally {
            stateChangeLock.unlock();
        }
    }

    public static final String N_DEW_TEMPS_KEY = "calculated/nDewTemps";
    public static final String N_DEW_HUMIDS_KEY = "calculated/nDewHumids";
    public static final String DEW_POINT_KEY = "calculated/dewPoint";

    private void publishDewPoint() {
        // Collect all the good temperature and humidity readings from the locations 
        // specified in the configuration.
        final List<Double> temps = getGoodAnalogData("temp");
        final List<Double> humids = getGoodAnalogData("humid");
        // Calculate the mean temperature T and mean humidity RH.
        if (temps.isEmpty() || humids.isEmpty()) {
            log.log(Level.WARNING, "Cannot calculate dew point: nDewTemps = {0}, nDewHumids = {1}.",
                    new Object[]{temps.size(), humids.size()});
            return;
        }
        final double tempMean = temps.stream().mapToDouble(Double::doubleValue).average().getAsDouble();
        final double humidMean = humids.stream().mapToDouble(Double::doubleValue).average().getAsDouble();
        // - Calculate Tdp from T and RH.
        final double tdp = Utility.dewPoint(tempMean, humidMean);
        // - Create and publish a new KVD list with the current time as the master timestamp
        //   and KV pairs for no. of temps, no.of humids and dew point.
        final KeyValueDataList calc = new KeyValueDataList();
        calc.addData(N_DEW_TEMPS_KEY, temps.size());
        calc.addData(N_DEW_HUMIDS_KEY, humids.size());
        calc.addData(DEW_POINT_KEY, tdp);
        // - Publish K on the status bus.
        this.publishSubsystemDataOnStatusBus(calc);
    }
    
    private List<Double> getGoodAnalogData(final String chanName) {
        final Map<String, Location> locs = locations.get();
        return locs.keySet().stream()
            .filter(locName -> dewLocations.contains(locName))
            .map(locName -> locs.get(locName))
            .filter(loc -> loc.getDataPoints().containsKey(chanName))
            .map(loc -> loc.getDataPoints().get(chanName))
            .map(dp -> (AnalogPoint)dp)
            .filter(ap -> ap.getQuality().equals("Good") && !ap.hasMalfunction())
            .map(ap -> ap.getValue())
            .collect(toList());
    }
}
