package org.lsst.ccs.subsystem.airwatch.main;

import static java.lang.String.format;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Contains fields common to all data points. Immutable unless subclasses play games.
 * @author tether
 */
public abstract class DataPoint {
    
    private final boolean limitViolation;
    
    private final boolean malfunction;
    
    private final String quality;
   
    private final Instant time;
    
    
    public DataPoint(boolean limitViolation, boolean malfunction, final String quality, final String isoTime) {
        this.limitViolation = limitViolation;
        this.malfunction = malfunction;
        this.quality = quality;
        this.time = Instant.from(DateTimeFormatter.ISO_DATE_TIME.parse(isoTime));
    }

    public final boolean hasLimitViolation() {
        return limitViolation;
    }

    public final boolean hasMalfunction() {
        return malfunction;
    }

    public final String getQuality() {
        return quality;
    }

    public final Instant getTime() {
        return time;
    }
 
    public abstract void checkData(final AlertService alerts, final String locName, final String chanName);
    
    public KeyValueDataList makeKvdList(final String locName, final String chanName) {
        final KeyValueDataList kvdl = new KeyValueDataList(
            format("%s/%s", locName, chanName),
            CCSTimeStamp.currentTimeFromMillis(time.toEpochMilli())
        );
        kvdl.addData(kvdl.getKey() + "/limitViolation", hasLimitViolation() ? 1 : 0);
        kvdl.addData(kvdl.getKey() + "/malfunction", hasMalfunction() ? 1 : 0);
        return kvdl;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("limitViolation=").append(limitViolation);
        sb.append(", malfunction=").append(malfunction);
        sb.append(", quality=").append(quality);
        sb.append(", time=").append(time);
        return sb.toString();
    }

}
