package org.lsst.ccs.subsystem.airwatch.main;

import static java.lang.String.format;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.services.alert.AlertService;
import static org.lsst.ccs.services.alert.AlertService.RaiseAlertStrategy.ON_SEVERITY_CHANGE;

/**
 * Contains the data for one reading of one particle counter channel.
 * @author tether
 */
public class CounterPoint extends DataPoint {
    
    /** The absolute number of particles detected in the air sample. */
    private final Double  absolute;

    /** The particle density (particles/volume). */
    private final Double  density;

    /** Sets the field values as given. */
    public CounterPoint(
        final Double absolute,
        final Double density,
        final boolean limitViolation,
        final boolean malfunction,
        final String quality,
        final String time)
    {
        super(limitViolation, malfunction, quality, time);
        this.absolute = absolute;
        this.density = density;
    }
    
    /** Creates a dummy instance used for message registration.
     *  @see AirwatchMain#init() 
     */
    CounterPoint() {
        this(0.0, 0.0, false, false, "Good", "2021-05-22T13:55:21-07:00");
    }

    public final Double getAbsolute() {
        return absolute;
    }

    public final Double getDensity() {
        return density;
    }

    @Override
    public final KeyValueDataList makeKvdList(final String locName, final String chanName) {
        final KeyValueDataList kvdl = super.makeKvdList(locName, chanName);
        kvdl.addData(kvdl.getKey() + "/density", getDensity());
        kvdl.addData(kvdl.getKey() + "/absolute", getAbsolute());
        return kvdl;
    }

    @Override
    public final void checkData(final AlertService alsvc, final String locName, final String chanName) {
        if (!getQuality().equals("Good") || hasMalfunction()) {
            alsvc.raiseAlert(
                Alerts.dataQualityAlert(locName, chanName),
                AlertState.WARNING,
                format("Data quality = %s, malfunction = %s.", getQuality(), hasMalfunction()),
                ON_SEVERITY_CHANGE);
        }
        
        if (hasLimitViolation()) {
            alsvc.raiseAlert(
                Alerts.limitViolationAlert(locName, chanName),
                AlertState.WARNING,
                format("Particle density of %.4g/ft^3 is above the operating limit.", getDensity()),
                ON_SEVERITY_CHANGE);
        }
    }

    @Override
    public final String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("CounterPoint{absolute=").append(absolute);
        sb.append(", density=").append(density);
        sb.append(", ");
        sb.append(super.toString());
        sb.append('}');
        return sb.toString();
    }
    
}
