package org.lsst.ccs.subsystem.airwatch.main;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.time.Duration;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
/**
 * Retrieves the data from the Lighthouse environment sensors via HTTP. The server is a Python script
 * running on the Lighthouse PC which knows how to read the data from the LMS Express OPC DA server. Immutable.
 * @author tether
 */
public class RestfulClient {
    
    private final URL serverUrl;
    
    private final Duration connectionTimeout;
    
    private final Duration readTimeout;

    /**
     * Stores the HTTP URL.
     * @param serverUrl the URL for the sensor PC's data server.
     * @param connectionTimeout how long to wait for a connection to complete.
     * @param readTimeout how long to wait for data after connecting.
     */
    public RestfulClient(final URL serverUrl, final Duration connectionTimeout, final Duration readTimeout){
        this.serverUrl = serverUrl;
        this.connectionTimeout = connectionTimeout;
        this.readTimeout = readTimeout;
    }
    
    /** 
     * Dumps the text received from the server without trying to parse it.
     * @return The text sent from the server.
     * @throws IOException if the the server could not be found or if data could not be read from it.
     */
    public final String dumpData() throws IOException {
        final URLConnection connection = serverUrl.openConnection();
        connection.setConnectTimeout((int)connectionTimeout.toMillis());
        connection.setReadTimeout((int)readTimeout.toMillis());
        connection.connect();
        final BufferedReader sensorStream =
            new BufferedReader(new InputStreamReader(connection.getInputStream()));
        final List<String> sensorLines = new LinkedList<>();
        String line;
        try {
            while ((line = sensorStream.readLine()) != null) {
                sensorLines.add(line);
            }
        }
        finally {
            sensorStream.close();
        }
        sensorLines.add("");
        return String.join("\n", sensorLines);
    }

    /**
     * Reads sensor information in JSON format and parses it.
     * @return A list of {@code Map<String, Object>}. In each map the value with key "location" is a string
     * giving the name of the location such as "MAIN_NW" or "AUX". The other keys are all channel names.
     * A numerical channel name key such as "0.3" or "3.0" denotes a particle counter channel for
     * particles of that size or greater in microns. Anything else is the name of an analog channel such as
     * "temp".
     * <p>
     * A value for a particle counter channel is itself a map of items appropriate for such a channel:
     * <ul>
     * <li> density (double): The value of the particle density in particles/ft^3.</li>
     * <li> quality (String): The OPC DA quality for the value. Should be "Good" if the value is well measured.</li>
     * <li> time (String): The timestamp for the value in ISO format including the local offset from UTC.</li>
     * <li> status (String): If the quality is not "Good", this is a brief description of the trouble</i>
     * <li> limitViolation (boolean): True if and only if the value is out of bounds. Distinct from quality of measurement.</li>
     * <li> malfunction (boolean): true if and only if a sensor malfunction occurred.</li>
     * <li> absolute (double): The absolute particle count for the air sample, not divided by sample volume.</li>
     * </ul>
     * <p>
     * A value for an analog channel is itself a map. It also has the keys "quality", "time",
     * "status", "limitViolation" and "malfunction". In addition it has:
     * <ul>
     * <li>value (double): The sensor reading.
     * <li>lowBound (double): The lowest acceptable reading.</li>
     * <li>highBound (double): The highest acceptable reading. </li>
     * </ul>
     * @throws IOException if no data could be read or could not be parsed.
     */
    public final List<Map<String, Object>> fetchData() throws IOException {
        final String data = dumpData();
        final ObjectMapper mapper = new ObjectMapper();
        final Map<String, Map<String, Object>> result = mapper.readValue(data,
            new TypeReference<Map<String, Map<String, Object>>>() {});
        return new LinkedList<>(result.values());
    }
    
    public static void main(final String[] args) throws MalformedURLException, IOException {
        final RestfulClient client =
            new RestfulClient(
                new URL("http://10.0.1.18:8080/sensors"),
                Duration.ofSeconds(10),
                Duration.ofSeconds(10)
            );
        final List<Map<String, Object>> data = client.fetchData();
        for (final Map<String, Object> cluster: data) {
            System.out.println("----------");
            for (final String key: cluster.keySet()) {
                System.out.println(key);
                if (key.equals("location")) {
                    System.out.println("    " + cluster.get(key));
                }
                else {
                    final Map<String, Object> channel = (Map<String, Object>)cluster.get(key);
                    for (final String key2: channel.keySet()) {
                        System.out.printf("    %s(%s): %s%n",
                            key2, channel.get(key2).getClass().getSimpleName(), channel.get(key2));
                    }
                }
            }
        }
        System.out.println(client.dumpData());
    }
}
