package org.lsst.ccs.subsystem.airwatch.main;

import static java.lang.String.format;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.services.alert.AlertService;

/**
 * Contains the data for an analog sensor reading such as temperature. Immutable.
 * @author tether
 */

public class AnalogPoint extends DataPoint {
    
    /** The value read from the sensor. */
    private final Double  value;
    
    /** The lowest desired value. */
    private final Double lowBound;
    
    /** The highest desired value. */
    private final Double highBound;

    /** Sets the field values as given. */
    public AnalogPoint(
        final Double value,
        final boolean limitViolation,
        final boolean malfunction,
        final String quality,
        final String time,
        final Double lowBound,
        final Double highBound
    )
    {
        super(limitViolation, malfunction, quality, time);
        this.value = value;
        this.lowBound = lowBound;
        this.highBound = highBound;
    }
    
    /** Creates a dummy instance used for message registration.
     *  @see AirwatchMain#init() 
     */
    AnalogPoint() {
        this(0.0, false, false, "Good", "2021-05-22T13:55:21-07:00", 0.0, 0.0);
    }

    public final Double getValue() {
        return value;
    }

    public final Double getLowBound() {
        return lowBound;
    }

    public final Double getHighBound() {
        return highBound;
    }

    @Override
    public final KeyValueDataList makeKvdList(final String locName, final String chanName) {
        final KeyValueDataList kvdl = super.makeKvdList(locName, chanName);
        kvdl.addData(kvdl.getKey() + "/value", getValue());
        return kvdl;
    }

    @Override
    public final void checkData(final AlertService alsvc, final String locName, final String chanName) {
        if (!getQuality().equals("Good")  || hasMalfunction()) {
            alsvc.raiseAlert(
                Alerts.dataQualityAlert(locName, chanName),
                AlertState.WARNING,
                format("Data quality = %s, malfunction = %s.", getQuality(), hasMalfunction()));
        }
        
        if (hasLimitViolation()) {
            alsvc.raiseAlert(
                Alerts.limitViolationAlert(locName, chanName),
                AlertState.WARNING,
                format("Sensor reading of %.1f is outside the range to %.1f to %.1f.",
                    getValue(), getLowBound(), getHighBound()));
        }
    }

    @Override
    public final String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("AnalogPoint{value=").append(value);
        sb.append(", ");
        sb.append(super.toString());
        sb.append('}');
        return sb.toString();
    }
}
