package org.lsst.ccs.subsystem.airwatch.main;

import static java.lang.String.format;
import org.lsst.ccs.bus.data.Alert;

/**
 * Contains a set of static factory functions for the alerts used by the subsystem. Can't be instantiated.
 * @author tether
 */
public class Alerts {
   
    // No instances allowed.
    private Alerts() {}

    /**
     * Creates an alert about a failure to read sensor data.
     * @return A new Alert instance with an ID of {@code SensorIO}.
     */
    public static Alert sensorIOAlert() {
        return new Alert("SensorIO", "I/O error when attempting to read sensor data.");
    }

    /**
     * Creates an alert about an a sensor reading that's outside of the desired range.
     * @param location The location of the instrument, e.g., ENTRY, MAIN_SE.
     * @param channelName Which quantity has gone out of range, e.g., 0.3, Humidity.
     * @return A new Alert instance with an ID of
     * {@code format("LimitViolation-%s-%s", location, channelName)}.
     */
    public static Alert limitViolationAlert(final String location, final String channelName) {
        return new Alert(
                format("LimitViolation-%s-%s", location, channelName),
                "Value out of bounds.");
    }
    
    
    /**
     * Creates an alert about a sensor malfunction or questionable data quality.
     * @param location The short location name of the instrument, e.g., AUX, MAIN_SE.
     * @param channelName The name of the channel that had the failure.
     * @return A new Alert instance with an ID of {@code format("DataQuality-%s-%s", location, channelName)}.
     */
    public static Alert dataQualityAlert(final String location, final String channelName) {
        return new Alert(
            format("DataQUality-%s-%s", location, channelName),
            "Quality flag != Good or an instrument malfunction was reported."
        );
    }
    
}
