package org.lsst.ccs.subsystem.airwatch.main;

import org.lsst.ccs.commons.annotations.ConfigurationParameter;


import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;

import java.util.ArrayList;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static java.util.function.Function.identity;

import java.util.Collections;

import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import org.lsst.ccs.AlertService;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.PersistencyService;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.HasLifecycle;

/**
 * Implements a local configuration service based on the CCS subsystem configuration and persistence APIs.
 * An instance will be the receptacle of all the subsystem configuration/persistent data which is declared
 * here using the {@link ConfigurationParameter} or {@link Persist} annotations.
 *
 * The last-read time and last assigned location for each instrument is normally obtained from the
 * persistence service unless overridden by the configuration.
 * @author tether
 */
public class CCSConfiguration implements LocalConfigService, HasLifecycle {

    @LookupField(strategy=Strategy.TREE)
    private volatile ConfigurationService ccsConfig;

    @LookupField(strategy=Strategy.TREE)
    private volatile PersistencyService ccsPersist;

    @LookupField(strategy=Strategy.TREE)
    private volatile AlertService ccsAlert;

    ////////// Implementation of LocalConfigService //////////
    private volatile Duration readoutInterval;
    private volatile List<InstrumentConfig> instConfigs;
    private volatile List<LocationConfig> locConfigs;


    /**
     * Checks configuration and persistent data and makes the configuration objects.
     * Must only be called when full configuration data is available.
     */
    @Override
    public void makeConfigurationObjects() {
        try {
            readoutInterval = Duration.parse(readoutIntervalString);
        }
        catch (java.time.DateTimeException exc) {
            throw new IllegalArgumentException("Bad configuration. Invalid readout interval.");
        }
        makeLocationConfigs();
        makeInstrumentConfigs();
    }

    @SuppressWarnings("unchecked")
    private void makeLocationConfigs() {
        // For each particle size the configuration has a map giving the particle
        // count limit for each location. Associate each such map with the
        // abstract particle channel for that size particle.
        final Map<InstrumentChannel, Map<String, Long>> chanToLoc =
            Stream.of(InstrumentChannel.values())
            .filter(InstrumentChannel::isParticleChannel)
            .collect(Collectors.toMap(
                identity(),
                chan -> {
                    try {
                        // The following cast required suppressing the "unchecked"
                        // warning. Seems to be unavoidable when using reflection.
                        return
                            (Map<String, Long>)this.getClass()
                            .getDeclaredField(channelThresholdsConfigVarName(chan))
                            .get(this);
                    } catch(NoSuchFieldException exc) {
                        throw new IllegalArgumentException("Missing configuration variable?", exc);
                    } catch(IllegalAccessException exc) {
                        throw new Error("Impossible!", exc);}
                    }));

        // Now perform validity checks on each threshold value map.
        // All locations named must be in knownLocations.
        // All threshold values must be in the range [0, Integer.MAX_VALUE].
        final List<String> knownLocs =
            Stream.of(knownLocations)
            .map(String::toLowerCase)
            .collect(Collectors.toList());
        chanToLoc.forEach((chan, configVar) -> {
            checkStringSet(
                configVar.keySet().stream(),
                knownLocs.stream(),
                "Bad locations in " + channelThresholdsConfigVarName(chan));
            checkThresholds(configVar, channelThresholdsConfigVarName(chan));
        });

        // Create the location configurations. Transpose: one map-by-location per channel
        // becomes one map-by-channel per location.
        final List<Map<InstrumentChannel, Long>> locToChan =
            IntStream.range(0, knownLocs.size())
            .mapToObj(i -> new EnumMap<InstrumentChannel, Long>(InstrumentChannel.class))
            .collect(ArrayList::new, ArrayList::add, ArrayList::addAll);

        chanToLoc.forEach((chan, configVar) -> {
            configVar.forEach((locName, threshold) -> {
                final int i = knownLocs.indexOf(locName);
                locToChan.get(i).put(chan, threshold);
            });
        });

        locConfigs = Collections.unmodifiableList(
            IntStream.range(0, knownLocs.size())
            .mapToObj(i -> new LocationConfig(knownLocations[i], i, locToChan.get(i)))
            .collect(ArrayList::new, ArrayList::add, ArrayList::addAll)
            );
    }

    private void makeInstrumentConfigs() {
        // Locations and last-read times  from configuration override those from persistence.
        if (configInstrumentLocations.length > 0) {
            instrumentLocations = configInstrumentLocations;
        }
        if (configInstrumentLastDataTimes.length > 0) {
            instrumentLastDataTimes = configInstrumentLastDataTimes;
        }

        final List<String> knownLocs =
            Stream.of(knownLocations)
            .map(String::toLowerCase)
            .collect(Collectors.toList());
        final String[] types = instrumentTypes;
        final String[] conns = instrumentConnections;
        final String[] locs =
            Stream.of(instrumentLocations)
            .map(String::toLowerCase)
            .collect(Collectors.toList())
            .toArray(new String[instrumentLocations.length]);
        final String[] times = instrumentLastDataTimes;
        // Validity checking.
        if ((types.length != conns.length)
            || (types.length != locs.length)
            || (types.length != times.length)) {
            throw new IllegalArgumentException(
                "Bad configuration. Not all the instrumentXxx[] arrays are the same length."
            );
        }
        if (types.length == 0) {
            throw new IllegalArgumentException("Bad configuration. No instruments defined!");
        }
        checkStringSet(Stream.of(locs), knownLocs.stream(), "Unknown instrument locations");
        checkStringSet(
            Stream.of(types),
            Stream.of(InstrumentType.values()).map(InstrumentType::getConfigName),
            "Unknown instrument types");
        checkStringSet(
            Stream.of(times),
            Stream.of(times).filter(t -> {
                try {
                    Instant.parse(t);
                    return true;
                } catch (java.time.format.DateTimeParseException exc) {
                    return false;
                }
            }),
            "Last data-times rejected by Instant.parse()"
        );
        // Construction of configuration objects.
        instConfigs = Collections.unmodifiableList(
            IntStream
            .range(0, types.length)
            .mapToObj(i -> new InstrumentConfig(
                    i,
                    InstrumentType.parse(types[i]).get(),
                    conns[i],
                    locs[i],
                    Instant.parse(times[i]),
                    this,
                    ccsAlert
            ))
            .collect(Collectors.toList()));
    }

    private static void checkStringSet(
        Stream<String> mentioned,
        Stream<String> legal,
        String errorMsg)
    {
        Set<String> unknown = mentioned.collect(Collectors.toSet());
        unknown.removeAll(legal.collect(Collectors.toSet()));
        if (unknown.size() > 0) {
            throw new IllegalArgumentException(
                "Bad configuration. " + errorMsg + ": " +
                unknown.stream().collect(Collectors.joining(", ")));
        }
    }

    private static void checkThresholds(Map<String, Long> threshVar, String varName) {
        threshVar.forEach((locName, threshold) -> {
            if ((threshold < 0) || (threshold > Integer.MAX_VALUE)) {
                throw new IllegalArgumentException(
                    String.format(
                        "Bad configuration. Bad threshold value %d for location %s in %s.",
                        threshold,
                        locName,
                        varName
                        )
                );
            }
        });
    }

    private String channelThresholdsConfigVarName(InstrumentChannel chan) {
        return "locationThresholds_" + chan.getKey().replaceFirst("\\.", "_");
    }

    /** {@inheritDoc } */
    @Override
    public List<InstrumentConfig> getInstrumentConfigs() {
        return instConfigs;
    }

    /** {@inheritDoc } */
    @Override
    public void updateInstrument(InstrumentStatus stat) {
        // Safely publish new location array using defensive copy. Needed
        // since elements of a voltile array are not volatile.
        String[] newLocs = Arrays.copyOf(instrumentLocations, instrumentLocations.length);
        newLocs[stat.index] = stat.location;
        instrumentLocations = newLocs; // Volatile write makes changes visible.

        // Similar logic for last-data times.
        String[] newTimes = Arrays.copyOf(instrumentLastDataTimes, instrumentLastDataTimes.length);
        newTimes[stat.index] = stat.lastDataTime.toString();
        instrumentLastDataTimes = newTimes;

        // Save these changes using the CCS persistence mechanism.
        ccsPersist.persistNow();
    }

    /** {@inheritDoc } */
    @Override
    public List<LocationConfig> getLocationConfigs() {
        return locConfigs;
    }

    /** {@inheritDoc } */
    @Override
    public Duration getReadoutInterval() {return readoutInterval;}
    
    /** {@iinheritDoc} */
    @Override
    public String getEmailSender() {return emailSender;}
    
    /** {@inheritDoc } */
    @Override
    public String getEmailBounceAddress() {return emailBounceTo;}
    
    /** {@inheritDoc } */
    @Override
    public String getSMTPServer() {return SMTPServer;}
    
    /** {@inheritDoc } */
    @Override
    public List<String> getEmailRecipients() {
        return Collections.unmodifiableList(Arrays.asList(emailRecipients));
    }

    @Override
    /** {@inheritDoc} */
    public List<String> getOPCServerInfo(final String serverKey) {
        return Collections.unmodifiableList(this.opcServers.get(serverKey));
    }

    ////////// Lifecycle methods //////////
    /** Makes an instance. */
    public CCSConfiguration() {
        super();
    }

    @Override
    public void init() {
        final List<String> missing = new ArrayList<>();
        if (ccsConfig == null) {
            missing.add("CCS configuration service");
        }
        if (ccsPersist == null) {
            missing.add("CCS persistence service");
        }
        ccsPersist.setAutomatic(true /* At startup */, true /* At shutdown */);
        if (!missing.isEmpty()) {
            throw new RuntimeException("Can't find " + String.join(", ", missing));
        }
    }

    ////////// Configuration data //////////

    /**
     * The time between readouts of all the instruments.
     */
    @ConfigurationParameter(name="readoutInterval", description="The time between readouts.")
    private volatile String readoutIntervalString = "PT1H";

    /**
     * The set of all legal names for instrument locations.
     */
    @ConfigurationParameter(description="All valid location names.")
    private volatile String[] knownLocations = new String[] {};

    /**
     * Maps location names to alarm thresholds for the 0.3 micron channel.
     */
    @ConfigurationParameter(description="Alarm thresholds for the 0.3 micron channel.")
    private volatile Map<String, Long> locationThresholds_0_3 = Collections.emptyMap();

    /**
     * Maps location names to alarm thresholds for the 0.5 micron channel.
     */
    @ConfigurationParameter(description="Alarm thresholds for the 0.5 micron channel.")
    private volatile Map<String, Long> locationThresholds_0_5 = Collections.emptyMap();

    /**
     * Maps location names to alarm thresholds for the 1.0 micron channel.
     */
    @ConfigurationParameter(description="Alarm thresholds for the 1.0 micron channel.")
    private volatile Map<String, Long> locationThresholds_1_0 = Collections.emptyMap();

    /**
     * Maps location names to alarm thresholds for the 2.5 micron channel.
     */
    @ConfigurationParameter(description="Alarm thresholds for the 2.5 micron channel.")
    private volatile Map<String, Long> locationThresholds_2_5 = Collections.emptyMap();

    /**
     * Maps location names to alarm thresholds for the 3.0 micron channel.
     */
    @ConfigurationParameter(description="Alarm thresholds for the 3.0 micron channel.")
    private volatile Map<String, Long> locationThresholds_3_0 = Collections.emptyMap();

    /**
     * Maps location names to alarm thresholds for the 5.0 micron channel.
     */
    @ConfigurationParameter(description="Alarm thresholds for the 5.0 micron channel.")
    private volatile Map<String, Long> locationThresholds_5_0 = Collections.emptyMap();

    /**
     * Maps location names to alarm thresholds for the 10.0 micron channel.
     */
    @ConfigurationParameter(description="Alarm thresholds for the 10.0 micron channel.")
    private volatile Map<String, Long> locationThresholds_10_0 = Collections.emptyMap();

    /**
     * For each instrument, its make and/or model, for example "lighthouse".
     */
    @ConfigurationParameter(description="Instrument make and/or model.")
    private volatile String[] instrumentTypes = new String[] {};

    /**
     * For each instrument, a string containing connection information to be parsed.
     * For example "serial /dev/ttyUSB0" or "net foo.slac.stanford.edu 2250"
     */
    @ConfigurationParameter(description="How to open a connection to the instrument.")
    private volatile String[] instrumentConnections = new String[] {};

    /**
     * For each instrument, its location, drawn from the set of known locations. A set of
     * values loaded from the configuration trumps a set from persistence.
     */
    @Persist
    private volatile String[] instrumentLocations = new String[] {};

    @ConfigurationParameter(name="instrumentLocations", description="Locations of instruments.")
    private volatile String[] configInstrumentLocations = new String[] {};

    /**
     * For each instrument, the time of the most recent data read from it. Subsequent
     * reads will obtain data later than this. The string must be acceptable to
     * Instant.parse(). A set of values loaded from the configuration trumps a set from persistence.
     */
    @Persist
    private volatile String[] instrumentLastDataTimes = new String[] {};

    @ConfigurationParameter(name="instrumentLastDataTimes",
                            description="The last time each instrument was read.")
    private volatile String[] configInstrumentLastDataTimes = new String[] {};
    
    /**
     * The name or IPv4 address of the SMTP server to use to send emails.
     */
    @ConfigurationParameter(description="The name or IPv4 address of the SMTP server.")
    private volatile String SMTPServer = "";
    
    /**
     * What to put in the From: fields of emails headers.
     */
    @ConfigurationParameter(description="Email address, possibly fake, used as email sender.")
    private volatile String emailSender = "";
    
    /**
     * What to put in the Bounce-to: fields for alarm emails. Must be a the address of
     * a real mailbox.
     */
    @ConfigurationParameter(description="Real email address which receives bounced messages.")
    private volatile String emailBounceTo = "";
    
    /**
     * The list of email recipients. A set of real email addresses.
     */
    @ConfigurationParameter(description="List of recipients of alarm emails.")
    private volatile String[] emailRecipients = new String[] {};

    /**
     * A map where each value is a list of strings used to connect to an OPC server
     * publishing Lighthouse sensor data. The key either the Windows hostname
     * or some nickname for it.
     */
    @ConfigurationParameter(description="Connection information for LMS Express OPC servers.")
    private volatile Map<String, List<String>> opcServers = Collections.emptyMap();
}
